<?php
declare(strict_types=1);
session_start();
date_default_timezone_set('America/Guayaquil');

/* ===== Seguridad de sesión ===== */
if (!isset($_SESSION['ingreso']) || $_SESSION['ingreso'] !== 'YES') {
  http_response_code(401);
  echo 'Sesión no válida.'; exit;
}
$id_usuario = (int)($_SESSION['idusuario'] ?? 0);
if ($id_usuario <= 0) { http_response_code(401); echo 'Usuario no identificado.'; exit; }

/* ===== Conexión BD (resolución robusta de ruta) ===== */
$pathConex = realpath(__DIR__ . '/../conex.php') ?: realpath(__DIR__ . '/../../conex.php');
if (!$pathConex) { http_response_code(500); echo 'No se encontró conex.php'; exit; }
require $pathConex;

if (!isset($conn) || !$conn instanceof mysqli) { http_response_code(500); echo 'Sin conexión a la base de datos.'; exit; }
mysqli_set_charset($conn, 'utf8mb4');

/* ===== Parámetros obligatorios (compat con tu flujo) ===== */
$iddai   = isset($_GET['id'])      ? (int)$_GET['id'] : 0;
$VIN     = strtoupper(trim((string)($_GET['itemdai'] ?? '')));   // VIN/CHASIS
$MARCA   = strtoupper(trim((string)($_GET['m'] ?? '')));         // marca
$MODELO  = strtoupper(trim((string)($_GET['mo'] ?? '')));        // modelo
$COLOR   = strtoupper(trim((string)($_GET['d'] ?? '')));         // color

if ($VIN === '') {
  http_response_code(400);
  echo 'Parámetros inválidos: id y itemdai (VIN) son obligatorios.'; exit;
}

/* ===== Parámetros extra para DISEÑO ===== */
$EMPRESA  = strtoupper(trim((string)($_GET['empresa']   ?? 'LATKIC S.A.')));
$VERSION  = strtoupper(trim((string)($_GET['version']   ?? '')));
$MOTOR    = strtoupper(trim((string)($_GET['motor']     ?? '')));
$TRACCION = strtoupper(trim((string)($_GET['traccion']  ?? '')));  // p.ej. 4X2
$TRANS    = strtoupper(trim((string)($_GET['trans']     ?? '')));  // p.ej. TM

/* Tamaño de la etiqueta en milímetros (por defecto 80x40mm) */
$LABEL_W_MM = (int)($_GET['w'] ?? 80);
$LABEL_H_MM = (int)($_GET['h'] ?? 40);
$autoprint  = isset($_GET['print']) && (int)$_GET['print'] === 1;

/* ===== Datos auditoría/actualización ===== */
$id_ubicacion = 1;                 // Ubicación por defecto
$actividad    = 'REGISTRO DE UNIDADES';
$my_ip        = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
$fecha_actual = date('Y-m-d');
$hora_actual  = date('H:i:s');

/* ===== Transacción: actualiza y audita ===== */
$conn->begin_transaction();
try {
  // UPDATE id_estado = 2
  $st1 = $conn->prepare("UPDATE tb_datadai SET id_estado = 2 WHERE iddatadai = ?");
  $st1->bind_param('i', $iddai);
  $st1->execute();
  if ($st1->affected_rows < 0) { throw new RuntimeException('No se pudo actualizar estado.'); }

  // UPDATE id_ubicacion = 1 (o el que corresponda)
  $st2 = $conn->prepare("UPDATE tb_datadai SET id_ubicacion = ? WHERE iddatadai = ?");
  $st2->bind_param('ii', $id_ubicacion, $iddai);
  $st2->execute();
  if ($st2->affected_rows < 0) { throw new RuntimeException('No se pudo actualizar ubicación.'); }

  // INSERT auditoría
  $st3 = $conn->prepare("INSERT INTO tb_auditoria (fecha, hora, ip, detalle, id_datadai, id_usuario)
                         VALUES (?, ?, ?, ?, ?, ?)");
  $st3->bind_param('ssssii', $fecha_actual, $hora_actual, $my_ip, $actividad, $iddai, $id_usuario);
  $st3->execute();

  $st1->close(); $st2->close(); $st3->close();
  $conn->commit();
} catch (Throwable $e) {
  $conn->rollback();
  http_response_code(500);
  echo 'Error al registrar cambios: ' . htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8');
  exit;
}

/* ===== Librería QR (resolución robusta de ruta) ===== */
$pathQR = realpath(__DIR__ . '/qrlib.php') ?: realpath(__DIR__ . '/../qr/qrlib.php');
if (!$pathQR) { http_response_code(500); echo 'No se encontró qrlib.php'; exit; }
require_once $pathQR;

/* ===== Carpeta temp para PNG ===== */
$PNG_TEMP_DIR = __DIR__ . DIRECTORY_SEPARATOR . 'temp' . DIRECTORY_SEPARATOR;
$PNG_WEB_DIR  = 'temp/';
if (!is_dir($PNG_TEMP_DIR)) {
  if (!mkdir($PNG_TEMP_DIR, 0775, true) && !is_dir($PNG_TEMP_DIR)) {
    http_response_code(500); echo 'No se pudo crear carpeta temp para QR.'; exit;
  }
}
if (!is_writable($PNG_TEMP_DIR)) { http_response_code(500); echo 'Carpeta temp no es escribible.'; exit; }

/* ===== Config QR (override por GET: level=L|M|Q|H, size=1..10) ===== */
$errorCorrectionLevel = (isset($_GET['level']) && in_array($_GET['level'], ['L','M','Q','H'], true)) ? $_GET['level'] : 'M';
$matrixPointSize      = isset($_GET['size']) ? max(1, min((int)$_GET['size'], 10)) : 4;

/* ===== Contenido del QR =====
   Por defecto: solo VIN.
   Si quieres JSON, descomenta y comenta la línea de $qrData = $VIN;
   $qrData = json_encode([
     'vin'=>$VIN,'marca'=>$MARCA,'modelo'=>$MODELO,'version'=>$VERSION,
     'motor'=>$MOTOR,'traccion'=>$TRACCION,'trans'=>$TRANS,'color'=>$COLOR,
     'id'=>$iddai,'fecha'=>$fecha_actual
   ], JSON_UNESCAPED_UNICODE);
*/
$qrData = $VIN;

/* ===== Archivo de QR ===== */
$slug = preg_replace('/[^A-Z0-9_-]+/','-', $VIN);
$filename = $PNG_TEMP_DIR . 'QR_' . $slug . '_' . $iddai . '.png';
QRcode::png($qrData, $filename, $errorCorrectionLevel, $matrixPointSize, 2);
$qrUrl = $PNG_WEB_DIR . basename($filename);

/* ===== Líneas de la tarjeta (lado derecho) ===== */
$lineas = array_values(array_filter([
  $MARCA ?: null,
  $MODELO ?: null,
  $VERSION ?: null,
  trim(($MOTOR ? $MOTOR.' ' : '').($TRACCION ?: '')) ?: null,
  trim(($TRANS ? $TRANS.' ' : '').($COLOR ?: '')) ?: null
]));
?>
<!doctype html>
<html lang="es">
<head>
  <meta charset="utf-8" />
  <title>SGO · Etiqueta VIN</title>
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <link rel="icon" href="../../resources/img/icon.ico">
  <style>
  html,body{
    height:100%; margin:0; padding:0;
    font-family: system-ui,-apple-system,"Segoe UI",Roboto,Arial,sans-serif;
  }
  body{
    display:flex; align-items:center; justify-content:center;
    background:#f1f3f5;
  }

  .label{
    width:100%;              /* ocupa todo el ancho */
    height:auto;
    background:#fff; color:#000;
    border: .6mm solid #d4d6d9; border-radius: 3mm;
    padding: 8px 12px;
    display:grid;
    grid-template-rows: auto auto 1fr;
    gap: 4px;
  }

  .empresa{
    font-size: 0.8rem;
    font-weight: 700;
    text-align:center;       /* 👈 centrado */
    text-transform: uppercase;
    margin-bottom: 1px;
  }

  .vin{
    font-weight: 700;
    font-size: 1.0rem;
    letter-spacing: 2px;
    margin: 2px 0;
    word-break: break-all;
    text-align:center;       /* VIN también centrado */
  }

  .fila{
    display:grid;
    grid-template-columns: 80px 1fr;
    align-items: center;
    gap: 0px;
  }

  .qr{
    width: 70px; height: 70px;
    border:1px solid #ccc; border-radius: 4px;
    display:flex; align-items:center; justify-content:center;
    overflow:hidden;
    margin:auto;             /* centra el QR en su columna */
  }

  .qr img{ width:100%; height:100%; object-fit:contain; }

  .detalle{
    display:flex; flex-direction:column; gap:1px;
    font-size:0.7rem; line-height:1.0;
  }
  .detalle .marca{ font-weight:500; font-size:1.05rem; }

  @page{ size: auto; margin:0; }
  @media print{
    body{ background:#fff; }
    .label{ box-shadow:none; border:0; }
  }
</style>
<style>
  @media print {
    .ticket { page-break-after: always; } /* fuerza salto por ticket */
  }
</style>

</head>
<body>
  <div class="label">
    <div class="empresa"><?= htmlspecialchars($EMPRESA, ENT_QUOTES, 'UTF-8') ?></div>
    <div class="vin"><?= htmlspecialchars($VIN, ENT_QUOTES, 'UTF-8') ?></div>

    <div class="fila">
      <div class="qr"><img src="<?= htmlspecialchars($qrUrl, ENT_QUOTES, 'UTF-8') ?>" alt="QR"></div>
      <div class="detalle">
        <?php foreach ($lineas as $i => $t): ?>
          <div class="<?= $i === 0 ? 'marca' : '' ?>"><?= htmlspecialchars($t, ENT_QUOTES, 'UTF-8') ?></div>
        <?php endforeach; ?>
      </div>
    </div>
  </div>

  <?php if ($autoprint): ?>
  <script> addEventListener('load', ()=>{ window.print(); }); </script>
  <?php endif; ?>
</body>
</html>
