<?php
// views/pages/patio.php — RÁPIDO y LISTO PARA PEGAR
// Mejoras de performance: delegación de eventos, DOM batch, cache colores, sin zoom.
declare(strict_types=1);
session_start();
if (!isset($_SESSION['ingreso']) || $_SESSION['ingreso'] !== 'YES') {
  header("Location: ../../views/index.php");
  exit();
}
$usuario = $_SESSION['usuario'] ?? '';
$permiso = (int)($_SESSION['permiso'] ?? 0);
require_once __DIR__ . '/conex.php';
?>
<?php include 'partials/header.php'; ?>

<style>
  :root{
    --cell: 60px;
    --car-h: calc(var(--cell)*1.8);
    --gap: 6px;
    --hd-h: 22px;
  }

  body{margin:0;font-family:system-ui,Arial}
  .wrap{max-width:1300px;margin:2px auto;padding:2px}
  .plano{position:relative;display:inline-block;line-height:0}
  .plano img{max-width:100%;height:auto;display:block}

  /* Encabezados opcionales */
  #patio-wrap{ overflow:auto; }
  #colheads, #grid{ display:grid; gap: var(--gap); }
  #colheads{ grid-auto-flow: column; }
  #rowheads{ display:grid; gap: var(--gap); }
  #colheads .hd, #rowheads .hd{
    width: var(--cell); height: var(--hd-h);
    display:flex; align-items:center; justify-content:center;
    font-size:12px; color:#334155;
    background:#f1f5f9; border:1px solid #e5e7eb; border-radius:6px;
    user-select:none;
  }
  #grid{ grid-auto-rows: var(--car-h); }

  /* Lote grande (por si lo usas) */
  .cell{
    width: var(--cell); height: var(--car-h);
    border:1px dashed #e5e7eb; border-radius:8px; position:relative;
    display:flex; align-items:center; justify-content:center;
    cursor:pointer; user-select:none; background:#f8fafc;
  }
  .cell.free{ background:#f0fdf4; }
  .cell.busy{ background:#fee2e2; }
  .cell.off{ background:#f8fafc; color:#94a3b8; border-style:dashed; }

  /* Tooltip global */
  .tooltip{
    position:fixed; padding:6px 10px; border-radius:8px; background:#111; color:#fff;
    font-size:13px; pointer-events:none; opacity:0; transform:translate(-50%,-120%);
    transition:opacity .08s ease; box-shadow:0 6px 18px rgba(0,0,0,.18);
    border:1px solid rgba(255,255,255,.1);
    white-space:normal; max-width:280px; line-height:1.25; z-index: 99999;
  }
  .tooltip .dot{
    display:inline-block;width:10px;height:10px;border-radius:999px;margin-right:6px;
    vertical-align:middle;border:1px solid rgba(0,0,0,.25);background:#fff;
  }
  .tooltip .title{ font-weight:700; }

  /* Overlay de áreas */
  #hotspots{ position:absolute; inset:0; overflow:visible; pointer-events:auto; }

  .area{
    position:absolute;
    cursor: pointer;
    border: 1px solid rgba(0,0,0,.15);
    border-radius: 8px;
    outline: 1px dashed transparent;
    contain: content;            /* aislar layout/paint/composite */
  }
  .area:hover{ outline:1px dashed rgba(0,0,0,.35); }

  /* Subgrilla */
  .area .subgrid{
    position:absolute; inset:8px;
    display:grid; gap:6px;
    contain: content;
  }
  .area .subcell{
    position:relative; border:1px dashed rgba(0,0,0,.18); border-radius:6px;
    background: rgba(255,255,255,.25); overflow:hidden;
    contain: content;
  }
  .area .subcell.free { background:#dcfce7; border-color:#22c55e; }
  .area .subcell.busy { background:#fee2e2; border-color:#ef4444; }

  /* Vehículo miniatura (se mantiene look) */
  .mini-car{
    pointer-events:none; position:absolute; inset:12%;
    border-radius: 10px / 18px;
    box-shadow: inset 0 1px 0 rgba(255,255,255,.35), 0 2px 6px rgba(0,0,0,.12);
  }
  .mini-car.free {  background: linear-gradient(#22c55e,#16a34a); }
  .mini-car.busy {  background: linear-gradient(#ef4444,#dc2626); }
  .mini-car::before, .mini-car::after{
    content:''; position:absolute; left:14%; right:14%; height:12%;
    background: rgba(255,255,255,.55); border-radius:6px; box-shadow: inset 0 1px 2px rgba(0,0,0,.12);
  }
  .mini-car::before{ top:7%; } 
  .mini-car::after { bottom:7%; }
  .mini-car .roof{
    position:absolute; left:18%; right:18%; top:50%; height:18%; transform:translateY(-50%);
    background: rgba(255,255,255,.28); border-radius:8px; box-shadow: inset 0 1px 2px rgba(0,0,0,.08);
  }
  .mini-wheel{
    position:absolute; width:36%; height:10%; background:#111827; border-radius:12px;
    box-shadow:0 1px 0 rgba(255,255,255,.15), inset 0 0 6px rgba(0,0,0,.5);
    filter: drop-shadow(0 1px 1px rgba(0,0,0,.25));
  }
  .mini-wheel-tl{ top:10%;  left:-12%; transform: rotate(90deg); }
  .mini-wheel-tr{ top:10%;  right:-12%; transform: rotate(90deg); }
  .mini-wheel-bl{ bottom:10%; left:-12%; transform: rotate(90deg); }
  .mini-wheel-br{ bottom:10%; right:-12%; transform: rotate(90deg); }

  /* Autos de lado si el área es horizontal */
  .subgrid.horizontal .mini-car{
    left:50%; top:50%;
    width:88%; height:64%;
    transform: translate(-50%,-50%) rotate(90deg);
  }

  /* Ocultar etiquetas/leyendas de slots (como pediste) */
  .slot-tag{ display:none !important; }
</style>

<body>
<div class="container-fluid">
  <div class="row">
    <?php include 'partials/menu.php'; ?>

    <main class="col-md-9 ms-sm-auto col-lg-10 px-0">
      <?php include 'partials/topbar.php'; ?>

      <section class="px-3 py-2">
        <div class="d-flex justify-content-between align-items-center mb-2">
          <h2 class="h6 m-0">Distribución de Patio</h2>
        </div>

        <div class="wrap">
          <div class="plano" id="plano">
            <img id="imgPlano" src="plano_quito4.png" alt="Plano">
            <div id="hotspots" aria-label="overlay de áreas"></div>
          </div>
        </div>
        <div class="tooltip" id="tip"><span class="dot" id="dot"></span><span id="tipText"></span></div>
      </section>

      <div id="hovercard" class="position-fixed d-none"></div>
    </main>
  </div>
</div>

<?php include 'partials/footer.php'; ?>

<script>
/* ================== CONFIG ================== */
const CONFIG = { 
  areas: {
    A: { nombre: "Área A", color: "lightcoral",     coords: [60,740,165,870]},
    T: { nombre: "Área T", color: "lightsalmon",    coords: [165,740,230,920]},
    F: { nombre: "Área F", color: "wheat",          coords: [230,740,500,920]},
    J: { nombre: "Área J", color: "khaki",          coords: [520,740,920,880]},
    H: { nombre: "Área H", color: "lightseagreen",  coords: [950,740,1290,880]},
    P: { nombre: "Área P", color: "paleturquoise",  coords: [180,340,1140,700]},
    C: { nombre: "Área C", color: "palegreen",      coords: [1140,340,1550,590]},
    B: { nombre: "Área B", color: "lightskyblue",   coords: [880,5,1150,180]},
    L: { nombre: "Área L", color: "plum",           coords: [880,180,1150,300]},
    D: { nombre: "Área D", color: "mediumpurple",   coords: [1150,5,1550,300]},
    E: { nombre: "Área E", color: "thistle",        coords: [590,145,665,300]},
    G: { nombre: "Área G", color: "gold",           coords: [665,180,710,300] },
    K: { nombre: "Área K", color: "peachpuff",      coords: [420,110,495,300]}
  }
};

/* Mantengo todas verticales como tu último mensaje.
   Si quieres K de lado: cambia a horizontal:true */
const AREA_SPECS = {
  A: { cols: 4,  rows: 3,  horizontal: false },
  T: { cols: 2,  rows: 4,  horizontal: false },
  F: { cols: 10, rows: 4,  horizontal: false },
  J: { cols: 17, rows: 3,  horizontal: false },
  H: { cols: 15, rows: 3,  horizontal: false },
  P: { cols: 38, rows: 9,  horizontal: false },
  C: { cols: 17, rows: 6,  horizontal: false },
  D: { cols: 15, rows: 7,  horizontal: false },
  B: { cols: 10, rows: 4,  horizontal: false },
  L: { cols: 10, rows: 3,  horizontal: false },
  G: { cols: 1,  rows: 3,  horizontal: false },
  E: { cols: 2,  rows: 4,  horizontal: false },
  K: { cols: 2,  rows: 5,  horizontal: false }
};

/* ================== HELPERS (optimizados) ================== */
const COLOR_CACHE = {};
function cssColorToRGBA(color, alpha=0.22){
  const key = color + '|' + alpha;
  if (COLOR_CACHE[key]) return COLOR_CACHE[key];
  const tmp = document.createElement('div');
  tmp.style.color = color;
  document.body.appendChild(tmp);
  const cs = getComputedStyle(tmp).color;
  document.body.removeChild(tmp);
  const m = cs.match(/rgba?\((\d+),\s*(\d+),\s*(\d+)/i);
  const out = m ? `rgba(${m[1]},${m[2]},${m[3]},${alpha})` : `rgba(37,99,235,${alpha})`;
  COLOR_CACHE[key] = out;
  return out;
}
function normalizeCoords([x0,y0,x1,y1]){ if (x1 < x0) [x0, x1] = [x1, x0]; if (y1 < y0) [y0, y1] = [y1, y0]; return [x0,y0,x1,y1]; }
function pxToPerc(x0,y0,x1,y1, naturalW, naturalH){
  return {
    left:  (x0/naturalW)*100,
    top:   (y0/naturalH)*100,
    w:     ((x1-x0)/naturalW)*100,
    h:     ((y1-y0)/naturalH)*100
  };
}
function rowIndexToLetters(i){ let n=i+1,s=''; while(n>0){const r=(n-1)%26; s=String.fromCharCode(65+r)+s; n=Math.floor((n-1)/26);} return s; }

/* ================== STATE/DOM ================== */
const img = document.getElementById('imgPlano');
const overlay = document.getElementById('hotspots');
const tip = document.getElementById('tip');
const tipText = document.getElementById('tipText');
const dot = document.getElementById('dot');

let ANDEN_MAP = {}; // { 'A': { 'fila-col': {...} } }

/* ================== RENDER SUBGRID (más rápido) ================== */
function buildSubgridHTML(letter, spec, def){
  const rgbaBg  = cssColorToRGBA(def.color, 0.22);
  const rgbaBor = cssColorToRGBA(def.color, 0.45);

  let html = `<div class="subgrid${spec.horizontal ? ' horizontal':''}" style="
      display:grid;grid-template-columns:repeat(${spec.cols},1fr);
      grid-template-rows:repeat(${spec.rows},1fr);gap:6px;position:absolute;inset:8px;">
  `;
  const areaMap = ANDEN_MAP[letter] || {};
  const getInfo = (r,c)=> areaMap[`${r}-${c}`];

  for (let r=1; r<=spec.rows; r++){
    for (let c=1; c<=spec.cols; c++){
      const info = getInfo(r,c);
      const ocupado = !!(info && (info.ocupado==1 || info.activo==1 || String(info.estado||'').toUpperCase()==='OCUPADO'));
      const cls = ocupado ? 'busy' : 'free';
      html += `
        <div class="subcell ${cls}" data-area="${letter}" data-fila="${r}" data-columna="${c}" data-id="${info?.id_anden ?? ''}"
             style="border-color:${rgbaBor}">
          <div class="mini-car ${cls}">
            <div class="roof"></div>
            <i class="mini-wheel mini-wheel-tl"></i>
            <i class="mini-wheel mini-wheel-tr"></i>
            <i class="mini-wheel mini-wheel-bl"></i>
            <i class="mini-wheel mini-wheel-br"></i>
          </div>
        </div>`;
    }
  }
  html += `</div>`;
  return html;
}

/* ================== HOTSPOTS ================== */
function createHotspots(){
  if (!overlay) return;
  overlay.innerHTML = '';
  const naturalW = img.naturalWidth, naturalH = img.naturalHeight;

  const frag = document.createDocumentFragment();

  for (const [letter, def] of Object.entries(CONFIG.areas)){
    const spec = AREA_SPECS[letter]; if (!spec) continue;

    const [X0,Y0,X1,Y1] = normalizeCoords(def.coords);
    const {left,top,w,h} = pxToPerc(X0,Y0,X1,Y1,naturalW,naturalH);

    const area = document.createElement('div');
    area.className='area';
    area.style.left = left+'%';
    area.style.top  = top+'%';
    area.style.width  = w+'%';
    area.style.height = h+'%';
    area.style.background  = cssColorToRGBA(def.color, 0.22);
    area.style.borderColor = cssColorToRGBA(def.color, 0.45);
    area.dataset.letter = letter;
    area.dataset.nombre = def.nombre;
    area.dataset.color  = def.color;

    // Subgrid via innerHTML (rápido)
    area.innerHTML = buildSubgridHTML(letter, spec, def);

    frag.appendChild(area);
  }
  overlay.appendChild(frag);
}

/* ================== FETCH + RENDER ================== */
async function fetchAndenes(query='?bodega=Quito'){
  try{
    const res = await fetch(`api/patio_map.php${query}`, { cache:'no-store' });
    const json = await res.json();
    if (!json.success) throw new Error(json.message || 'Error API');

    const tmp = {};
    for (const [area, arr] of Object.entries(json.areas || {})){
      tmp[area] = {};
      for (const it of arr){
        const f = Number(it.fila) || 0, c = Number(it.columna) || 0;
        tmp[area][`${f}-${c}`] = it;
      }
    }
    ANDEN_MAP = tmp;
  }catch(e){
    console.error('Error cargando mapa de andenes:', e);
    ANDEN_MAP = {};
  }

  if (img.complete && img.naturalWidth) {
    createHotspots();
  } else {
    img.addEventListener('load', createHotspots, { once:true });
  }
}

/* ================== EVENTOS DELEGADOS (tooltip solo dentro del área) ================== */
overlay.addEventListener('mousemove', (e)=>{
  const areaBox = e.target.closest('.area');

  // Si el cursor no está sobre NINGUNA área, ocultar el tooltip
  if (!areaBox) {
    tip.style.opacity = '0';
    return;
  }

  const cell = e.target.closest('.subcell');

  // Tooltip para el área (cuando no estás encima de una subcelda)
  if (!cell) {
    tipText.innerHTML = `<span class="title">${areaBox.dataset.letter} – ${areaBox.dataset.nombre}</span>`;
    dot.style.background = areaBox.dataset.color;
  } else {
    // Tooltip para la subcelda
    const letter = cell.dataset.area;
    const r = Number(cell.dataset.fila), c = Number(cell.dataset.columna);
    const info = (ANDEN_MAP[letter] || {})[`${r}-${c}`];
    const ocupado = cell.classList.contains('busy');
    const etiquetaDB = (info && typeof info.etiqueta === 'string') ? info.etiqueta.trim()
                     : `${letter}-${rowIndexToLetters(r-1)}${c}`;
    const fmt = (v)=>{ const s = (v ?? '').toString().trim(); return s==='' ? '—' : s; };

    const html = (ocupado && info)
      ? `<span class="title">${etiquetaDB}</span><br>
         <small>Chasis: ${fmt(info.chasis)}<br>
         Marca: ${fmt(info.marca)}<br>
         Modelo: ${fmt(info.modelo)}<br>
         Color: ${fmt(info.color)}</small>`
      : `<span class="title">${etiquetaDB}</span><br><small>Libre</small>`;

    tipText.innerHTML = html;
    dot.style.background = areaBox.dataset.color;
  }

  // Posicionar y mostrar
  tip.style.left = e.clientX + 'px';
  tip.style.top  = e.clientY + 'px';
  tip.style.opacity = '1';
}, { passive:true });

/* Ocultar al salir de un área hacia cualquier lugar que no sea otra área */
overlay.addEventListener('mouseout', (e)=>{
  const fromArea = e.target.closest('.area');
  const toArea = e.relatedTarget && e.relatedTarget.closest ? e.relatedTarget.closest('.area') : null;
  if (fromArea && !toArea) {
    tip.style.opacity = '0';
  }
}, { passive:true });

/* Ocultar al salir completamente del overlay */
overlay.addEventListener('mouseleave', ()=>{
  tip.style.opacity = '0';
}, { passive:true });

/* Mantener el tooltip dentro del viewport mientras esté visible */
document.addEventListener('mousemove', (e)=>{
  if (tip.style.opacity !== '1') return;
  const pad=8, vw=window.innerWidth, vh=window.innerHeight;
  let x=e.clientX, y=e.clientY;
  x=Math.max(pad, Math.min(vw-pad, x));
  y=Math.max(pad, Math.min(vh-pad, y));
  tip.style.left=x+'px';
  tip.style.top=y+'px';
}, { passive:true });


/* Init */
fetchAndenes('?bodega=Quito');
</script>
</body>
</html>
