<?php 
// dai_simple.php — SGO · Módulo SIMPLE (carga .xlsx, DataTable, impresión múltiple de QR)
declare(strict_types=1);
session_start();

if (!isset($_SESSION['ingreso']) || $_SESSION['ingreso'] !== 'YES') {
  header('Location: ../../views/index.php'); exit();
}

$usuario = $_SESSION['usuario'] ?? 'Invitado';
$permiso = (int)($_SESSION['permiso'] ?? 0);

/* Conexión: resuelve conex.php de forma robusta */
$pathConex = realpath(__DIR__ . '/conex.php') ?: realpath(__DIR__ . '/../../conex.php');
if (!$pathConex) { die('No se encontró conex.php'); }
require_once $pathConex;

// ⚠️ Este archivo genera HTML. No lances JSON aquí en excepciones; loguea en backend si es necesario.
// ini_set('display_errors','0'); error_reporting(E_ALL);
?>
<!doctype html>
<html lang="es">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>SGO · Módulo SIMPLE</title>

  <!-- Bootstrap 5 + Icons -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet"/>

  <!-- DataTables + Buttons -->
  <link href="https://cdn.datatables.net/1.13.8/css/dataTables.bootstrap5.min.css" rel="stylesheet"/>
  <link href="https://cdn.datatables.net/buttons/2.4.2/css/buttons.bootstrap5.min.css" rel="stylesheet"/>

  <style>
    :root{
      --fs-table:12px; /* compacto */
    }
    body{ -webkit-font-smoothing:antialiased; }
    #dai{ font-size:var(--fs-table); }
    #dai thead th{ font-size:var(--fs-table); }
    #dai tbody td{ font-size:calc(var(--fs-table) - 1px); vertical-align:middle; }
    .tabla-preview{ font-size:10px; width:100%; table-layout:auto; }
    .tabla-preview th,.tabla-preview td{ white-space:nowrap; padding:4px 8px; vertical-align:middle; }

    .panel { background:#fff; border:1px solid #e9ecef; border-radius:.5rem; }
    .dataTables_wrapper .top{ gap:.5rem; }
    .dataTables_wrapper .bottom{ padding:.25rem .5rem; }
    .dt-buttons .btn{ padding:.25rem .5rem; }
    .form-label{ margin-bottom:.25rem; }
  </style>
</head>
<body>
  <div class="container-fluid">
    <div class="row">
      <?php include __DIR__ . '/partials/menu.php'; ?>
      <main class="col-md-9 ms-sm-auto col-lg-10 px-0">
        <?php include __DIR__ . '/partials/topbar.php'; ?>

        <div class="row g-2 p-2">
          <div class="col-12">
            <div class="panel p-3">
              <div class="row g-2 mb-2">
                <div class="col-md-6">
                  <button class="btn btn-primary w-100" data-bs-toggle="modal" data-bs-target="#archivos">
                    <i class="fa-solid fa-file-arrow-up me-1"></i> Cargar archivo SIMPLE.xlsx
                  </button>
                </div>
                <div class="col-md-6">
                  <button id="btnPrintQR" class="btn btn-success w-100" disabled>
                    <i class="fa fa-print me-1"></i> Imprimir QR seleccionados
                  </button>
                </div>
              </div>

              <div class="table-responsive">
                <table id="dai" class="table table-hover table-bordered w-100">
                  <thead class="table-light">
                    <tr>
                      <th class="text-center" style="width:54px">
                        <div class="form-check d-flex justify-content-center align-items-center">
                          <input class="form-check-input" type="checkbox" id="checkAll" title="Seleccionar todos" />
                          <label for="checkAll" class="ms-1 mb-0" title="Seleccionar todos los QR">
                            <i class="fa fa-qrcode"></i>
                          </label>
                        </div>
                      </th>
                      <th>Origen</th>
                      <th>Pedido</th>
                      <th>Naviera</th>
                      <th>Regimen</th>
                      <th>Marca</th>
                      <th>Modelo</th>
                      <th>Nº de Chasis</th>
                      <th>Color</th>
                      <!--th>Impronta</th-->
                      <th>Estado</th>
                      <th style="width:70px">Acción</th>
                    </tr>
                  </thead>
                  <tbody></tbody>
                </table>
              </div>

            </div>
          </div>
        </div>
      </main>
    </div>
  </div>

  <!-- Modal: Cargar archivo -->
  <div class="modal fade" id="archivos" tabindex="-1" aria-labelledby="modalArchivosLabel" aria-hidden="true">
    <div class="modal-dialog modal-xl">
      <form method="post" id="formImport" action="importar_s/import.php" enctype="multipart/form-data" class="modal-content">
        <div class="modal-header">
          <h5 class="modal-title" id="modalArchivosLabel">Ingresos Régimen Simple</h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
        </div>
        <div class="modal-body">
          <div class="row g-2">
            <div class="col-md-7">
              <label for="importador" class="form-label">CONSIGNATARIO EN RA.:*</label>
              <select id="importador" name="importador" class="form-select" required>
                <option value="">Cargando…</option>
              </select>
            </div>
            <div class="col-md-3">
              <label for="bodega" class="form-label">BODEGA LK.:*</label>
              <select id="bodega" name="bodega" class="form-select" required>
                <option value="">Cargando…</option>
              </select>
            </div>            
          </div>

          <div class="mt-3">
            <label for="name" class="form-label fw-bold">Archivo Excel (.xlsx)</label>
            <input type="file" name="name" id="name" class="form-control" accept=".xlsx" required>
          </div>

          <div class="table-responsive mt-3" style="max-height:400px;overflow:auto">
            <table id="previewTable" class="table table-striped table-bordered table-hover mb-0 tabla-preview">
              <thead class="table-light"></thead>
              <tbody></tbody>
            </table>
          </div>

          <div class="mt-3">
            <button type="submit" id="btnImportar" class="btn btn-success w-100" disabled>
              <i class="fa-solid fa-cloud-arrow-up me-1"></i> Importar Datos
            </button>
          </div>
        </div>
        <div class="modal-footer">
          <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cerrar</button>
        </div>
      </form>
    </div>
  </div>

  <!-- Modal: Confirmar eliminación -->
  <div class="modal fade" id="modalEliminar" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
      <div class="modal-content">
        <div class="modal-header bg-danger text-white">
          <h5 class="modal-title">¿Eliminar Registro?</h5>
          <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Cerrar"></button>
        </div>
        <div class="modal-body">¿Deseas eliminar el chasis <strong id="chasisEliminar"></strong>?</div>
        <div class="modal-footer">
          <button class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
          <button id="confirmarEliminar" class="btn btn-danger">
            <span id="spinnerEliminar" class="spinner-border spinner-border-sm me-2 d-none"></span>
            Eliminar
          </button>
        </div>
      </div>
    </div>
  </div>

  <!-- ====== Librerías ====== -->
  <script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

  <!-- DataTables + Buttons + deps PDF/Excel -->
  <script src="https://cdn.datatables.net/1.13.8/js/jquery.dataTables.min.js"></script>
  <script src="https://cdn.datatables.net/1.13.8/js/dataTables.bootstrap5.min.js"></script>
  <script src="https://cdn.datatables.net/buttons/2.4.2/js/dataTables.buttons.min.js"></script>
  <script src="https://cdn.datatables.net/buttons/2.4.2/js/buttons.bootstrap5.min.js"></script>
  <script src="https://cdnjs.cloudflare.com/ajax/libs/jszip/3.10.1/jszip.min.js"></script>
  <script src="https://cdnjs.cloudflare.com/ajax/libs/pdfmake/0.2.7/pdfmake.min.js"></script>
  <script src="https://cdnjs.cloudflare.com/ajax/libs/pdfmake/0.2.7/vfs_fonts.js"></script>
  <script src="https://cdn.datatables.net/buttons/2.4.2/js/buttons.html5.min.js"></script>

  <!-- SheetJS para previsualización de .xlsx -->
  <script src="https://cdn.jsdelivr.net/npm/xlsx@0.18.5/dist/xlsx.full.min.js"></script>

  <script>
  // ====== Filtro personalizado: múltiples chasis (columna 7) ======
  // Normaliza: quita espacios, guiones y símbolos para comparar sólo alfanuméricos
  const _norm = s => (s || '')
    .toString()
    .trim()
    .toLowerCase()
    .replace(/[^a-z0-9]/g, '');

  // Arreglo global con términos de chasis normalizados
  window._chasisTerms = [];

  // Importante: chasis está en la columna #8 => índice 7
  $.fn.dataTable.ext.search.push(function (settings, data) {
    if (!settings.nTable || settings.nTable.id !== 'dai') return true;
    if (!window._chasisTerms || window._chasisTerms.length === 0) return true;

    const chasis = _norm(data[7] || '');
    return window._chasisTerms.some(t => chasis.includes(t));
  });

  $(function(){
    const dt = $('#dai').DataTable({
      ajax: {
        url: 'api/dai_simple.php',
        type: 'GET',
        dataSrc: (json) => Array.isArray(json?.data) ? json.data : (Array.isArray(json) ? json : []),
        error: (xhr) => {
          console.error('Error DataTable:', xhr.responseText);
          Swal.fire('Error', 'No se pudo cargar la tabla.', 'error');
        }
      },
      columns: [
        { data:'check', orderable:false, searchable:false, defaultContent:'', className:'text-center' },
        { data:'origen',   defaultContent:'' },
        { data:'pedido',   defaultContent:'' },
        { data:'naviera',  defaultContent:'' },
        { data:'regimen',  defaultContent:'' },
        { data:'marca',    defaultContent:'' },
        { data:'modelo',   defaultContent:'' },
        { data:'chasis',   defaultContent:'' },
        { data:'color',    defaultContent:'' },
        /*{ data:'impronta', defaultContent:'' },*/
        { data:'estado',   defaultContent:'' },
        { data:'accion',   orderable:false, searchable:false, defaultContent:'', className:'text-center' }
      ],
      dom: '<"top d-flex flex-wrap gap-2 align-items-center"Bf><"bottom d-flex justify-content-between align-items-center"l>rtip',
      lengthMenu: [[10,25,50,100,-1],[10,25,50,100,'Todos']],
      buttons:[
        { extend:'copy',  text:'Copiar', exportOptions:{ columns:':visible:not(:first-child):not(:last-child)' } },
        { extend:'excel', text:'Excel',  exportOptions:{ columns:':visible:not(:first-child):not(:last-child)' } },
        { extend:'pdfHtml5', text:'PDF', orientation:'landscape', pageSize:'A4',
          exportOptions:{ columns:':visible:not(:first-child):not(:last-child)' },
          customize:(doc)=>{
            doc.styles.tableHeader={bold:true,fontSize:9,color:'black',fillColor:'#e9ecef',alignment:'center'};
            doc.defaultStyle.fontSize=8;
            const layout={hLineWidth:()=>0.4,vLineWidth:()=>0.4,hLineColor:()=>'#bbb',vLineColor:()=>'#bbb',paddingLeft:()=>4,paddingRight:()=>4,paddingTop:()=>2,paddingBottom:()=>2};
            if(doc.content[1]?.table){
              doc.content[1].layout=layout;
              doc.content[1].table.widths = Array(doc.content[1].table.body[0].length).fill('*');
            }
          }
        }
      ],
      pageLength: 10,
      language: { url:'../../resources/js/es-ES.json' },
      initComplete: function(){
        // Fallback si el JSON de idioma no carga
        if(!this.api().settings()[0].oLanguage.sUrl){
          this.api().settings()[0].oLanguage = {
            processing:"Procesando...", search:"Buscar:", lengthMenu:"Mostrar _MENU_ registros",
            info:"Mostrando _START_ a _END_ de _TOTAL_ registros",
            infoEmpty:"Mostrando 0 a 0 de 0 registros", infoFiltered:"(filtrado de _MAX_ en total)",
            loadingRecords:"Cargando...", zeroRecords:"No se encontraron resultados",
            emptyTable:"Sin datos disponibles",
            paginate:{ first:"Primero", previous:"Anterior", next:"Siguiente", last:"Último" },
            aria:{ sortAscending:": activar para ordenar ascendente", sortDescending:": activar para ordenar descendente" },
            buttons:{ copy:"Copiar", excel:"Excel", pdf:"PDF" }
          };
        }
        hookMultiChasis();
      }
    });

    function hookMultiChasis(){
      const $buscador = $('#dai_filter input[type="search"]');
      if (!$buscador.length) return;

      $buscador.attr('placeholder','Buscar · pega varios chasis separados por espacio / coma / ; / Enter');

      let debounce;
      $buscador.off('.multichasis').on('input.multichasis paste.multichasis keyup.multichasis', function () {
        clearTimeout(debounce);
        debounce = setTimeout(() => {
          const raw = (this.value || '').trim();

          if (!raw) {
            window._chasisTerms = [];
            dt.search('');
            dt.draw();
            return;
          }

          // Normaliza cada token y deduplica
          const tokens = raw
            .split(/[\s,;]+/)
            .map(t => _norm(t))
            .filter(Boolean);

          if (tokens.length <= 1) {
            // 1 término: permitimos búsqueda global para que también encuentre por marca/modelo, etc.
            window._chasisTerms = tokens;
            dt.search(raw).draw();
          } else {
            // Varios términos: usamos solo el filtro personalizado (columna chasis)
            window._chasisTerms = Array.from(new Set(tokens));
            dt.search('');
            dt.draw();
          }
        }, 120);
      });
    }

    // ====== Selección / impresión ======
    $(document).on('change','#checkAll',function(){
      $('.check-row').prop('checked', this.checked);
      toggleBtnQR();
    });
    $(document).on('change','.check-row', toggleBtnQR);
    function toggleBtnQR(){ $('#btnPrintQR').prop('disabled', $('.check-row:checked').length===0); }

    $('#btnPrintQR').on('click',function(){
      const seleccionados=[];
      $('.check-row:checked').each(function(){
        seleccionados.push({
          id:     $(this).val(),
          chasis: $(this).data('chasis'),
          color:  $(this).data('color'),
          marca:  $(this).data('marca'),
          modelo: $(this).data('modelo')
        });
      });
      if(!seleccionados.length) return;

      // Si el array es grande, considera POST JSON (ya te dejé un imprimir_multiple.php que lo soporta)
      const params = encodeURIComponent(JSON.stringify(seleccionados));
      const win = window.open(`qr/imprimir_multiple.php?data=${params}`,'_blank');

      const it = setInterval(()=>{
        if(win && win.closed){
          clearInterval(it);
          dt.ajax.reload(null,false);
          $('#checkAll').prop('checked',false);
          $('.check-row').prop('checked',false);
          $('#btnPrintQR').prop('disabled',true);
        }
      }, 600);
    });
  });

  // ====== Modal: carga combos al abrir ======
  document.addEventListener('DOMContentLoaded', ()=>{
    const modal = document.getElementById('archivos');
    modal.addEventListener('show.bs.modal', ()=>{
      cargarImportadores();
      cargarBodegas();
    });
  });

  function cargarImportadores(){
    $.getJSON('api/importadores.php')
      .done(data=>{
        const $sel = $('#importador');
        $sel.empty().append('<option value="">Seleccione</option>');
        (data||[]).forEach(it => $sel.append(`<option value="${it.id}">${it.nombre}</option>`));
      })
      .fail(()=> Swal.fire('Error','No se pudo cargar importadores','error'));
  }
  function cargarBodegas(){
    $.getJSON('api/bodegas.php')
      .done(data=>{
        const $sel = $('#bodega');
        $sel.empty().append('<option value="">Seleccione</option>');
        (data||[]).forEach(it => $sel.append(`<option value="${it.id}">${it.nombre}</option>`));
      })
      .fail(()=> Swal.fire('Error','No se pudo cargar bodegas','error'));
  }

  // ====== Previsualización .xlsx + envío ======
  (function(){
    const input  = document.getElementById('name');
    const btnImport = document.getElementById('btnImportar');
    const form   = document.getElementById('formImport');
    const prev   = document.getElementById('previewTable');

    input.addEventListener('change', e=>{
      const file = e.target.files && e.target.files[0];
      if(!file){ btnImport.disabled = true; return; }
      const ext = (file.name.split('.').pop() || '').toLowerCase();
      if(ext !== 'xlsx'){
        btnImport.disabled = true; input.value = '';
        Swal.fire({icon:'error',title:'Archivo inválido',text:'Selecciona un .xlsx válido'});
        return;
      }
      btnImport.disabled = false;

      // Previsualizar con SheetJS
      const fr = new FileReader();
      fr.onload = ev=>{
        try{
          const data = new Uint8Array(ev.target.result);
          const wb   = XLSX.read(data, {type:'array'});
          const ws   = wb.Sheets[wb.SheetNames[0]];
          const rows = XLSX.utils.sheet_to_json(ws, {header:1});
          prev.querySelector('thead').innerHTML = '';
          prev.querySelector('tbody').innerHTML = '';
          if(rows.length){
            const tr = document.createElement('tr');
            (rows[0] || []).forEach(col=>{
              const th = document.createElement('th'); th.textContent = col ?? ''; tr.appendChild(th);
            });
            prev.querySelector('thead').appendChild(tr);
            for(let i=1;i<rows.length;i++){
              const trb = document.createElement('tr');
              (rows[i] || []).forEach(col=>{
                const td = document.createElement('td'); td.textContent = col ?? ''; trb.appendChild(td);
              });
              prev.querySelector('tbody').appendChild(trb);
            }
          }
        }catch(err){
          console.error(err);
          Swal.fire('Error','No se pudo leer el archivo. Revisa el formato.','error');
        }
      };
      fr.readAsArrayBuffer(file);
    });

    form.addEventListener('submit', e=>{
      e.preventDefault();
      const file = input.files && input.files[0];
      const impEl = document.getElementById('importador');
      const bodEl = document.getElementById('bodega');
      const imp  = impEl ? impEl.value : '';
      const bod  = bodEl ? bodEl.value : '';

      if(!imp) return Swal.fire('Importador requerido','Selecciona un consignatario','warning');
      if(!bod) return Swal.fire('Bodega requerida','Selecciona una bodega LK.','warning');
      if(!file || (file.name.split('.').pop()||'').toLowerCase()!=='xlsx')
        return Swal.fire('Archivo requerido','Debes seleccionar un .xlsx válido','warning');

      Swal.fire({
        title:'Importando...',
        html:'Procesando archivo, espera por favor',
        allowOutsideClick:false,
        didOpen:()=>Swal.showLoading()
      });

      const fd = new FormData(form);
      fetch('importar_s/import.php', { method:'POST', body:fd })
        .then(async r=>{
          const text = await r.text();
          try{
            const json = JSON.parse(text);
            if(json?.success === false){
              throw new Error(json?.message || 'Error en importación');
            }
            Swal.fire({
              icon:'success',
              title:'Resumen',
              html:`<b>Insertados:</b> ${json.insertados ?? 0}<br>
                    <b>Duplicados:</b> ${json.duplicados ?? 0}<br>
                    <b>Errores:</b> ${json.errores ?? 0}`,
              width:'30%'
            }).then(()=>{
              $('#dai').DataTable().ajax.reload(null,false);
              const m = bootstrap.Modal.getInstance(document.getElementById('archivos')); if(m) m.hide();
              form.reset();
              prev.querySelector('thead').innerHTML='';
              prev.querySelector('tbody').innerHTML='';
              btnImport.disabled = true;
            });
          }catch(parseErr){
            console.error('Respuesta import.php:', text);
            Swal.fire('Error', 'Respuesta inválida del servidor de importación.', 'error');
          }
        })
        .catch(err=>{
          console.error(err);
          Swal.fire('Error','No se pudo completar la importación.','error');
        });
    });
  })();

  // ====== Eliminación de registro ======
  (function(){
    let idEliminar = null;
    $(document).on('click','.btn-eliminar',function(){
      idEliminar = $(this).data('id');
      $('#chasisEliminar').text($(this).data('chasis') || '');
      new bootstrap.Modal('#modalEliminar').show();
    });

    $('#confirmarEliminar').on('click', function(){
      if(!idEliminar) return;
      $('#spinnerEliminar').removeClass('d-none');
      $('#confirmarEliminar').prop('disabled', true);

      $.post('controllers/eliminar_unidad.php', { id:idEliminar }, null, 'json')
        .done(resp=>{
          $('#spinnerEliminar').addClass('d-none');
          $('#confirmarEliminar').prop('disabled', false);

          if(resp && resp.success){
            bootstrap.Modal.getInstance(document.getElementById('modalEliminar'))?.hide();
            const $row = $(`button[data-id='${idEliminar}']`).closest('tr');
            $row.css('background-color','#f8d7da');
            setTimeout(()=>{$('#dai').DataTable().ajax.reload(null,false);}, 350);
            Swal.fire('Eliminado','Registro eliminado correctamente.','success');
          }else{
            Swal.fire('Error', resp?.error || 'No se pudo eliminar.','error');
          }
        })
        .fail(xhr=>{
          $('#spinnerEliminar').addClass('d-none');
          $('#confirmarEliminar').prop('disabled', false);
          console.error(xhr.responseText);
          Swal.fire('Error','Error en la conexión.','error');
        });
    });
  })();
  </script>
</body>
</html>
