<?php
// index.php — SGO · Inventario · Categorías (full height, recarga estable, sin ID visible)
declare(strict_types=1);
session_start();
if (!isset($_SESSION['ingreso']) || $_SESSION['ingreso'] !== 'YES') {
  header("Location: ../../views/index.php");
  exit();
}
$usuario = $_SESSION['usuario'] ?? '';
$permiso = (int)($_SESSION['permiso'] ?? 0);
?>
<?php include __DIR__ . '/partials/header.php'; ?>
<!doctype html>
<html lang="es" data-theme="light">
<head>
  <meta charset="utf-8"/>
  <meta name="viewport" content="width=device-width, initial-scale=1"/>
  <title>SGO · Inventario · Categorías</title>

  <!-- CSS base (si ya los cargas en header.php, puedes quitar estas líneas para evitar duplicados) -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.datatables.net/1.13.8/css/dataTables.bootstrap5.min.css" rel="stylesheet">
  <!-- Font Awesome sin SRI para evitar bloqueo por integrity -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" referrerpolicy="no-referrer" />
  <link href="https://cdn.jsdelivr.net/npm/@sweetalert2/theme-bootstrap-4@5/bootstrap-4.min.css" rel="stylesheet">

  <style>
    :root { font-size: 15.5px; }
    .dt-center { text-align:center; }

    /* Tema oscuro global (lo activa topbar.php escribiendo data-theme en <html>) */
    html[data-theme="dark"] body { background:#0f1115; color:#e7e7e7; }
    html[data-theme="dark"] .table { color:#e7e7e7; }
    html[data-theme="dark"] .table thead th { color:#f0f0f0; }

    /* ===== Layout full-height ===== */
    html, body, .container-fluid, .row { height: 100%; }
    .flex-fill { min-height: 0; } /* vital para scroll interno en flex */
    #workArea, #tableWrap { min-height: 0; padding: 0; } /* sin padding para no “robar” alto */
    .dataTables_wrapper { padding: 0; }

    /* Tabla sin márgenes que resten alto */
    #tabla { border-collapse: separate; }
  </style>
</head>
<body>
<div class="container-fluid">
  <div class="row">
    <!-- Sidebar -->
    <?php include __DIR__ . '/partials/menu.php'; ?>

    <!-- Contenido principal -->
    <main class="col-md-9 ms-sm-auto col-lg-10 px-0 d-flex flex-column">
      <?php include __DIR__ . '/partials/topbar.php'; ?>

      <!-- Área de trabajo a toda pantalla bajo la topbar -->
      <section id="workArea" class="flex-fill d-flex flex-column">
        <!-- Toolbar -->
        <div id="toolbar" class="d-flex align-items-center gap-2 px-3 py-2 border-bottom">
          <h6 class="mb-0"><i class="fa-solid fa-tags me-2"></i>Categorías de Accesorios</h6>
          <div class="ms-auto d-flex gap-2">
            <input id="txtBuscar" class="form-control form-control-sm" placeholder="Buscar...">
            <button id="btnNueva" class="btn btn-primary btn-sm">
              <i class="fa-solid fa-plus"></i> Nueva
            </button>
          </div>
        </div>

        <!-- Contenedor que ocupa TODO el alto disponible (scroll interno para la tabla) -->
        <div id="tableWrap" class="flex-fill d-flex flex-column">
          <div class="flex-fill overflow-hidden">
            <table id="tabla" class="table table-striped table-hover w-100 m-0">
              <thead>
                <tr>
                  <th>Nombre</th>
                  <th class="dt-center" style="width:120px">Estado</th>
                  <th class="dt-center" style="width:180px">Acciones</th>
                </tr>
              </thead>
              <tbody></tbody>
            </table>
          </div>
        </div>
      </section>
    </main>
  </div>
</div>

<!-- Modal CRUD -->
<div class="modal fade" id="modalForm" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog">
    <form class="modal-content" id="formCat">
      <div class="modal-header py-2">
        <h6 class="modal-title"><span id="ttlForm">Nueva Categoría</span></h6>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
      </div>
      <div class="modal-body">
        <input type="hidden" id="id_categoria" name="id_categoria" value="0">
        <div class="mb-2">
          <label class="form-label">Nombre *</label>
          <input type="text" class="form-control" id="nombre" name="nombre" maxlength="120" required>
        </div>
        <div class="form-check">
          <input class="form-check-input" type="checkbox" id="condicion" name="condicion" checked>
          <label class="form-check-label" for="condicion">Activo</label>
        </div>
      </div>
      <div class="modal-footer py-2">
        <button type="button" class="btn btn-secondary btn-sm" data-bs-dismiss="modal">Cancelar</button>
        <button type="submit" class="btn btn-primary btn-sm">Guardar</button>
      </div>
    </form>
  </div>
</div>

<?php include __DIR__ . '/partials/footer.php'; ?>

<!-- JS -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.datatables.net/1.13.8/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.8/js/dataTables.bootstrap5.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

<script>
const apiBase = (window.SGO_API_BASE ?? 'api/');
let dt = null, modal, form;

document.addEventListener('DOMContentLoaded', () => {
  modal = new bootstrap.Modal(document.getElementById('modalForm'));
  form  = document.getElementById('formCat');

  initTableWithDynamicHeight();
  window.addEventListener('resize', debounce(resizeTableHeight, 150));

  document.getElementById('txtBuscar').addEventListener('input', e => dt.search(e.target.value).draw());

  document.getElementById('btnNueva').addEventListener('click', () => {
    form.reset();
    document.getElementById('id_categoria').value = 0;
    document.getElementById('condicion').checked = true;
    document.getElementById('ttlForm').textContent = 'Nueva Categoría';
    modal.show();
  });

  form.addEventListener('submit', onGuardar);
});

/* ===== DataTable full-height SIN ajax (carga manual) ===== */
function initTableWithDynamicHeight(){
  if (dt) return; // prevenir doble init
  const h = computeScrollY();
  dt = new DataTable('#tabla', {
    data: [], // cargaremos manualmente
    columns: [
      { data: 'nombre' },
      {
        data: 'condicion',
        className:'dt-center',
        render: (v)=> v==1 ? '<span class="badge bg-success">Activo</span>'
                           : '<span class="badge bg-secondary">Inactivo</span>'
      },
      { data: null, className:'dt-center', orderable:false, render: (row)=>accionesHTML(row) },
    ],
    order: [[0,'asc']],                  // orden por Nombre
    scrollY: h + 'px',
    scrollCollapse: true,
    paging: true,
    language: { url: 'https://cdn.datatables.net/plug-ins/1.13.8/i18n/es-ES.json' }
  });

  // Carga inicial
  recargarTabla();
}

function recargarTabla(){
  fetch(`${apiBase}categorias_list.php?_=${Date.now()}`)
    .then(r=>r.json())
    .then(j=>{
      if(!j.success) throw new Error(j.message||'Error');
      const pagina = dt.page();               // preservar página
      dt.clear();
      dt.rows.add(j.data || []);
      dt.draw(false);                         // no resetear paginación
      dt.page(pagina).draw(false);
      resizeTableHeight();
    })
    .catch(err=>{
      console.error(err);
      Swal.fire('Error','No se pudo cargar la lista','error');
    });
}

function resizeTableHeight(){
  if (!dt) return;
  const h = computeScrollY();
  const wrapper = document.querySelector('#tabla').closest('.dataTables_wrapper');
  const scroller = wrapper ? wrapper.querySelector('.dataTables_scrollBody') : null;
  if (scroller) {
    scroller.style.height = h + 'px';
    scroller.style.maxHeight = h + 'px';
  }
  dt.columns.adjust();
}

/* Calcula scrollY restando cabecera, info y paginación para que no “crezca” el hueco */
function computeScrollY(){
  const tableWrap = document.getElementById('tableWrap');
  const wrapper   = document.querySelector('#tabla')?.closest('.dataTables_wrapper');

  const headH = wrapper?.querySelector('.dataTables_scrollHead')?.offsetHeight || 0; // cabecera fija
  const infoH = wrapper?.querySelector('.dataTables_info')?.offsetHeight || 0;       // "Mostrando 1 a ..."
  const pagH  = wrapper?.querySelector('.dataTables_paginate')?.offsetHeight || 0;   // paginación
  const gap   = 16; // colchón

  const wrapH = tableWrap?.clientHeight || 400;
  const available = wrapH - headH - infoH - pagH - gap;

  return Math.max(160, available);
}

/* ===== Acciones ===== */
function accionesHTML(row){
  const id = row.id_categoria; // viene en el JSON aunque no lo mostremos
  const btnEdit   = `<button class="btn btn-sm btn-outline-primary me-1" onclick="editar(${id})" title="Editar"><i class="fa-solid fa-pen"></i></button>`;
  const btnToggle = `<button class="btn btn-sm btn-outline-warning me-1" onclick="toggleEstado(${id})" title="Activar/Desactivar"><i class="fa-solid fa-power-off"></i></button>`;
  const btnDel    = `<button class="btn btn-sm btn-outline-danger" onclick="eliminar(${id})" title="Eliminar/Inactivar"><i class="fa-solid fa-trash"></i></button>`;
  return btnEdit + btnToggle + btnDel;
}

function editar(id){
  fetch(`${apiBase}categoria_get.php?id=${id}`)
    .then(r=>r.json())
    .then(j=>{
      if(!j.success) return Swal.fire('Atención', j.message||'No encontrado', 'warning');
      const d = j.data;
      document.getElementById('id_categoria').value = d.id_categoria;
      document.getElementById('nombre').value = d.nombre;
      document.getElementById('condicion').checked = (parseInt(d.condicion)===1);
      document.getElementById('ttlForm').textContent = 'Editar Categoría';
      modal.show();
    })
    .catch(()=> Swal.fire('Error','No se pudo obtener el registro','error'));
}

function onGuardar(e){
  e.preventDefault();
  const payload = {
    id_categoria: parseInt(document.getElementById('id_categoria').value || '0'),
    nombre: document.getElementById('nombre').value.trim(),
    condicion: document.getElementById('condicion').checked ? 1 : 0
  };
  if (!payload.nombre) {
    document.getElementById('nombre').focus();
    return Swal.fire('Validación','El nombre es obligatorio','info');
  }

  fetch(`${apiBase}categoria_save.php`, {
    method:'POST',
    headers:{'Content-Type':'application/json'},
    body: JSON.stringify(payload)
  })
  .then(r=>r.json())
  .then(j=>{
    if(!j.success) return Swal.fire('Atención', j.message||'No se pudo guardar','warning');
    Swal.fire('Éxito', j.message||'Guardado', 'success');
    modal.hide();
    recargarTabla(); // recarga estable
  })
  .catch(()=> Swal.fire('Error','No se pudo guardar','error'));
}

function toggleEstado(id){
  fetch(`${apiBase}categoria_toggle.php`, {
    method:'POST',
    headers:{'Content-Type':'application/json'},
    body: JSON.stringify({id_categoria:id})
  })
  .then(r=>r.json())
  .then(j=>{
    if(!j.success) return Swal.fire('Atención', j.message||'No se pudo actualizar', 'warning');
    recargarTabla(); // recarga estable
  })
  .catch(()=> Swal.fire('Error','No se pudo actualizar','error'));
}

function eliminar(id){
  Swal.fire({
    title: '¿Eliminar?',
    text: 'Puedes inactivarla en lugar de eliminarla.',
    icon: 'warning',
    showCancelButton: true,
    confirmButtonText: 'Eliminar',
    cancelButtonText: 'Cancelar'
  }).then(res=>{
    if(!res.isConfirmed) return;
    fetch(`${apiBase}categoria_delete.php`, {
      method:'POST',
      headers:{'Content-Type':'application/json'},
      body: JSON.stringify({id_categoria:id, hard:0})
    })
    .then(r=>r.json())
    .then(j=>{
      if(!j.success) return Swal.fire('Atención', j.message||'No se pudo eliminar', 'warning');
      Swal.fire('Listo', j.message||'Eliminado', 'success');
      recargarTabla(); // recarga estable
    })
    .catch(()=> Swal.fire('Error','No se pudo eliminar','error'));
  });
}

/* ===== Util ===== */
function debounce(fn, wait=200){
  let t; return (...args)=>{ clearTimeout(t); t=setTimeout(()=>fn.apply(this,args), wait); };
}
</script>
</body>
</html>
