<?php
// index.php — SGO · Medición de baterías (listado + historial)
// - Sin régimen / motor / usuario
// - Columna Observación (tb_bateria.observacion del último registro)
// - Reglas DÍAS: <15 = ámbar, =15 = verde, >15 = rojo
// - SIN medición: en la COLUMNA "Días" se pinta AZUL y muestra días desde tb_datadai.fecha_ingreso
// - Botón “Mediciones” deshabilitado cuando no hay medición (tiene_medicion=0)
declare(strict_types=1);
session_start();
if (!isset($_SESSION['ingreso']) || $_SESSION['ingreso'] !== 'YES') {
  header("Location: ../../views/index.php"); exit();
}
$usuario = $_SESSION['usuario'] ?? 'Invitado';
$permiso = $_SESSION['permiso'] ?? 0;
require_once __DIR__ . '/conex.php';
?>
<!doctype html>
<html lang="es">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>SGO · Medición de Baterías</title>

  <!-- CSS -->
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css">
  <link rel="stylesheet" href="https://cdn.datatables.net/1.13.8/css/dataTables.bootstrap5.min.css">
  <link rel="stylesheet" href="https://cdn.datatables.net/buttons/2.4.2/css/buttons.bootstrap5.min.css">
  <link rel="stylesheet" href="https://cdn.datatables.net/responsive/2.5.0/css/responsive.bootstrap5.min.css">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css">
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css">

  <style>
    body{ font-size:.95rem; }
    #asigna { font-size: 12px; }
    #asigna thead th { font-size: 12px; }
    #asigna tbody td { font-size: 11px; }
    #checkAll{ vertical-align: middle; }
    /* Semáforo por FILA (según días) */
    .row-green { background-color: #e6ffe6 !important; }  /* =15 */
    .row-amber { background-color: #fff4d6 !important; }  /* <15 */
    .row-red   { background-color: #ffe1e1 !important; }  /* >15 o sin medición */
    /* Miniatura en historial */
    .thumb-bateria { width:44px; height:44px; object-fit:cover; border-radius:6px; cursor:zoom-in; }
  </style>
</head>
<body>
<div class="container-fluid">
  <div class="row">
    <?php include 'partials/menu.php'; ?>
    <main class="col-md-9 ms-sm-auto col-lg-10 px-0">
      <?php include 'partials/topbar.php'; ?>

      <div class="row g-0">
        <div class="col-12 p-3">

          <!-- Filtros -->
          <div class="row g-2 mb-3">
            <div class="col-12 col-md-2">
              <label class="form-label mb-1">Desde</label>
              <input type="date" id="filtroDesde" class="form-control form-control-sm">
            </div>
            <div class="col-12 col-md-2">
              <label class="form-label mb-1">Hasta</label>
              <input type="date" id="filtroHasta" class="form-control form-control-sm">
            </div>
            <div class="col-12 col-md-3">
              <label class="form-label mb-1">Rango de días</label>
              <select id="filtroRango" class="form-select form-select-sm">
                <option value="todos">Todos</option>
                <option value="sin">Sin medición</option>
                <option value="lt7">Menor a 15</option>
                <option value="b7_14">Igual a 15</option>
                <option value="ge15">Mayor a 15</option>
              </select>
            </div>
            <div class="col-12 col-md-3">
              <label class="form-label mb-1">Chasis (múltiple)</label>
              <input type="text" id="filtroChasis" class="form-control form-control-sm" placeholder="ABC123, DEF456 (coma/espacio/salto)">
            </div>
            <div class="col-12 col-md-2 d-flex align-items-end gap-2">
              <button id="btnAplicarFiltros" class="btn btn-primary btn-sm w-50"><i class="fa-solid fa-magnifying-glass me-1"></i>Buscar</button>
              <button id="btnLimpiarFiltros" class="btn btn-outline-secondary btn-sm w-50">Limpiar</button>
            </div>
          </div>

          <div class="table-responsive">
            <table id="asigna" class="table table-hover table-sm w-100">
              <thead>
                <tr>
                  
                  <th>Marca</th>
                  <th>Modelo</th>
                  <th>Chasis</th>
                  <th>Color</th>
                  <th class="text-center">Última medición</th>
                  <th class="text-end">Días</th>
                  <th class="text-center">Estado batería</th>
                  <th class="text-center">Observación</th>
                  <th style="width:90px;" class="text-center">Mediciones</th>
                </tr>
              </thead>
              <tbody><!-- dinámico --></tbody>
            </table>
          </div>

          <!-- Leyenda rápida -->
          <div class="small text-muted mt-2">
            Días: <span class="text-warning fw-bold">Ámbar</span> &lt; 15 ·
            <span class="text-success fw-bold">Verde</span> = 15 ·
            <span class="text-danger fw-bold">Rojo</span> &gt; 15 ·
            <span class="text-primary fw-bold">Azul</span> sin medición (muestra días desde ingreso).
          </div>

        </div>
      </div>
    </main>
  </div>
</div>

<!-- Modal Historial (sin "Usuario") -->
<div class="modal fade" id="modalHistorial" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-xl modal-dialog-scrollable">
    <div class="modal-content">
      <div class="modal-header py-2">
        <h6 class="modal-title">Historial de mediciones — <span id="histChasis">—</span></h6>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
      </div>
      <div class="modal-body p-2">
        <div id="histLoading" class="text-center my-3" style="display:none;">
          <div class="spinner-border" role="status" aria-hidden="true"></div>
          <div class="small mt-2">Cargando...</div>
        </div>
        <div class="table-responsive">
          <table class="table table-sm table-bordered mb-0" id="tablaHistorial">
            <thead class="table-light">
              <tr>
                <th class="text-center">Fecha</th>
                <th class="text-center">Hora</th>
                <th class="text-center">Estado</th>
                <th class="text-center">Valor</th>
                <th class="text-center">Resistencia</th>
                <th class="text-center">Vida</th>
                <th class="text-center">Foto</th>
              </tr>
            </thead>
            <tbody><!-- filas dinámicas --></tbody>
          </table>
        </div>
      </div>
      <div class="modal-footer py-2">
        <button type="button" class="btn btn-sm btn-secondary" data-bs-dismiss="modal">Cerrar</button>
      </div>
    </div>
  </div>
</div>

<!-- Visor de Foto -->
<div class="modal fade" id="fotoViewer" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-centered">
    <div class="modal-content bg-dark">
      <div class="modal-body p-0">
        <img id="fotoViewerImg" src="" alt="Foto batería" class="img-fluid w-100 d-block">
      </div>
      <div class="modal-footer py-2">
        <button type="button" class="btn btn-light btn-sm" data-bs-dismiss="modal">Cerrar</button>
      </div>
    </div>
  </div>
</div>

<!-- JS -->
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>

<script src="https://cdn.datatables.net/1.13.8/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.8/js/dataTables.bootstrap5.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.2/js/dataTables.buttons.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.2/js/buttons.bootstrap5.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/jszip/3.10.1/jszip.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/pdfmake/0.2.10/pdfmake.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/pdfmake/0.2.10/vfs_fonts.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.2/js/buttons.html5.min.js"></script>
<script src="https://cdn.datatables.net/responsive/2.5.0/js/dataTables.responsive.min.js"></script>
<script src="https://cdn.datatables.net/responsive/2.5.0/js/responsive.bootstrap5.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

<script>
$(function(){

  const API_LISTADO   = 'api/bateria.php';      // server-side (debe devolver: tiene_medicion, dias, observacion, etc.)
  const API_HISTORIAL = 'api/bateria_historial.php';
  const RUTA_FOTOS    = '../uploads/baterias/';

  const parseMultiChasis = (str)=> (str||'').split(/[\s,;|\n\r]+/).map(s=>s.trim()).filter(Boolean);

  const escapeHtml = (str)=> String(str ?? '')
    .replace(/&/g,'&amp;').replace(/</g,'&lt;')
    .replace(/>/g,'&gt;').replace(/"/g,'&quot;')
    .replace(/'/g,'&#039;');

  // ===== DataTable principal (SERVER-SIDE) =====
  const dt = $('#asigna').DataTable({
    processing: true,
    serverSide: true,
    responsive: true,
    deferRender: true,
    order: [[6,'desc'],[5,'desc']], // 6=días, 5=última_fecha (coincide con encabezados)
    ajax: {
      url: API_LISTADO,
      type: 'POST',
      data: function(d){
        d.desde  = $('#filtroDesde').val() || '';
        d.hasta  = $('#filtroHasta').val() || '';
        d.rango  = $('#filtroRango').val() || '';
        d.chasis = parseMultiChasis($('#filtroChasis').val()).join(',');
      },
      dataSrc: function(json){
        if (!json || typeof json !== 'object') {
          console.error('Respuesta no parseable:', json);
          return [];
        }
        if (json.error) console.warn('API error:', json.error, json.debug || '');
        return Array.isArray(json.data) ? json.data : [];
      },
      error: function(xhr){
        console.error('Respuesta cruda del servidor:\n', xhr.responseText);
        Swal.fire({icon:'error', title:'Error', text:'El servidor no devolvió JSON válido. Revisa la consola (F12).'});
      }
    },
    columns: [
      
      { data:'marca' },
      { data:'modelo' },
      { data:'chasis' },
      { data:'color' },

      { data:'ultima_fecha', className:'text-center',
        render:(d,t)=> t==='display' ? (d&&String(d).trim()?escapeHtml(d):'—') : d
      },

      /* ==== DÍAS: <15 ámbar; =15 verde; >15 rojo; SIN MEDICIÓN: azul y muestra días desde fecha_ingreso ==== */
      { data:'dias', className:'text-end', searchable:false,
        render:(data,type,row)=>{
          if (type!=='display') return data;
          const n = Number(data || 0);

          // SIN medición (flag enviado por API) => AZUL con días desde ingreso
          if (!row.tiene_medicion) {
             return `<span class="text-primary fw-bold">${n}<i class="fa-solid fa-circle-info ms-1"
                         data-bs-toggle="tooltip" data-bs-title="No hay medición registrada. Días desde el ingreso."></i></span>`;
          }

          // Con medición: reglas de color
          if (n === 15) {
            return `<span class="text-success fw-bold">${n}</span>`;
          }
          if (n < 15) {
            return `<span class="text-warning fw-bold">${n}</span>`;
          }
          // n > 15
          return `<span class="text-danger fw-bold">
                    ${n}
                    <i class="fa-solid fa-triangle-exclamation ms-1"
                       data-bs-toggle="tooltip" data-bs-title="Atención: ${n} día(s) desde la última medición"></i>
                  </span>`;
        }
      },

      /* ==== Estado batería: AZUL sin medición; VERDE bueno; ROJO malo; AMARILLO regular ==== */
      { data:'estado_bateria', className:'text-center',
        render:(data,type,row)=>{
          if (type!=='display') return data;

          // Si no hay medición, badge azul fijo
          if (!row.tiene_medicion) {
            return `<span class="badge bg-primary">Sin medición</span>`;
          }

          const raw = (data ?? '').toString().trim();
          const v   = raw.toLowerCase();

          let cls = 'bg-secondary';
          let txt = raw || '—';

          // Verde: bueno/buena/good/ok/normal
          if (/\b(bueno|buena|good|ok|normal)\b/.test(v)) {
            cls = 'bg-success';
          }
          // Rojo: malo/mala/bad/low/crítica/baja
          else if (/\b(malo|mala|bad|low|cr[ií]tica|baja)\b/.test(v)) {
            cls = 'bg-danger';
          }
          // Amarillo: regular/media/medio/fair
          else if (/\b(regular|media|medio|fair)\b/.test(v)) {
            cls = 'bg-warning text-dark';
          }

          return `<span class="badge ${cls}">${escapeHtml(txt)}</span>`;
        }
      },

      /* ==== Observación (texto del último registro) ==== */
      { data:'observacion', className:'text-center',
        render:(d,t)=> t==='display' ? (d ? escapeHtml(d) : '—') : d
      },

      /* ==== Botón historial (deshabilitado si no hay medición) ==== */
      { data:null, orderable:false, searchable:false, className:'text-center',
        render:(data,type,row)=>{
          const id = row.id_datadai || row.id || '';
          const chasis = row.chasis ? String(row.chasis).replace(/"/g,'&quot;') : '';
          const sinMed = !row.tiene_medicion || row.tiene_medicion == 0;

          const btnClass  = sinMed ? 'btn-outline-secondary' : 'btn-outline-primary';
          const disabled  = sinMed ? 'disabled aria-disabled="true"' : '';
          const titleText = sinMed ? 'Sin medición' : 'Ver mediciones';

          return `<button class="btn btn-sm ${btnClass} btn-historial"
                          data-id="${id}" data-chasis="${chasis}"
                          ${disabled} title="${titleText}">
                    <i class="fa fa-list"></i>
                  </button>`;
        }
      }
    ],

    // Sin buscador nativo (solo filtros personalizados de arriba)
    dom: '<"top d-flex flex-wrap align-items-center gap-2"B><"bottom d-flex justify-content-between align-items-center"l>rtip',
    searching: false,
    lengthMenu: [[10,25,50,100,-1],[10,25,50,100,"Todos"]],
    buttons: [
      { extend:'copy', text:'Copiar' },
      { extend:'excel', text:'Excel' },
      { extend:'pdfHtml5', text:'PDF', orientation:'landscape', pageSize:'A4',
        customize: doc => { doc.defaultStyle.fontSize = 9; }
      }
    ],
    rowCallback:(row,data)=>{
      $(row).removeClass('row-green row-amber row-red');
      const n = Number(data.dias);

      if (!data.tiene_medicion) {     // sin medición -> fila roja (criterio actual)
        $(row).addClass('row-red');
        return;
      }
      if (n === 15)      $(row).addClass('row-green');
      else if (n < 15)   $(row).addClass('row-amber');
      else               $(row).addClass('row-red');
    },
    language: { url:'https://cdn.datatables.net/plug-ins/1.13.8/i18n/es-ES.json' }
  });

  // Botones filtros -> recargar
  const reload = ()=>{ dt.ajax.reload(null, true); initTooltips(); };
  $('#btnAplicarFiltros').on('click', reload);
  $('#btnLimpiarFiltros').on('click', ()=>{
    $('#filtroDesde').val('');
    $('#filtroHasta').val('');
    $('#filtroRango').val('todos');
    $('#filtroChasis').val('');
    reload();
  });

  // CheckAll
  $('#checkAll').on('change', function(){
    $('#asigna tbody input.row-check').prop('checked', this.checked);
  });
  $('#asigna tbody').on('change', 'input.row-check', function(){
    if (!this.checked) $('#checkAll').prop('checked', false);
  });

  // Tooltips
  function initTooltips(){
    document.querySelectorAll('[data-bs-toggle="tooltip"]').forEach(el=>{
      const t = bootstrap.Tooltip.getInstance(el); if (t) t.dispose();
      new bootstrap.Tooltip(el, { container: 'body' });
    });
  }
  dt.on('draw', initTooltips);
  initTooltips();

  // ===== Modal Historial =====
  const modalEl = document.getElementById('modalHistorial');
  const modal   = new bootstrap.Modal(modalEl);
  const $tbodyHist  = $('#tablaHistorial tbody');
  const $loading    = $('#histLoading');
  const $histChasis = $('#histChasis');

  const celdaValorUnidad = (valor, unidad)=>{
    if (!valor) return '—';
    return (escapeHtml(valor)+' '+escapeHtml(unidad||'')).trim();
  };
  const celdaFoto = (foto)=>{
    if (!foto) return '—';
    const src = (foto.startsWith('http') || foto.startsWith('/')) ? foto : (RUTA_FOTOS + encodeURIComponent(foto));
    return `<img src="${src}" class="thumb-bateria" data-full="${src}" alt="foto">`;
  };

  function renderHistorial(rows){
    if ($.fn.DataTable.isDataTable('#tablaHistorial')) {
      $('#tablaHistorial').DataTable().clear().destroy();
    }
    const html = rows.length ? rows.map(r=>`
      <tr>
        <td class="text-center">${escapeHtml(r.fecha) || '—'}</td>
        <td class="text-center">${escapeHtml(r.hora)  || '—'}</td>
        <td class="text-center">${escapeHtml(r.estado) || '—'}</td>
        <td class="text-center">${celdaValorUnidad(r.valor, r.unidad)}</td>
        <td class="text-center">${r.resistencia ?? '—'}</td>
        <td class="text-center">${r.vida ?? '—'}</td>
        <td class="text-center">${celdaFoto(r.foto)}</td>
      </tr>
    `).join('') : `<tr><td colspan="7" class="text-center text-muted">Sin registros</td></tr>`;
    $tbodyHist.html(html);

    $('#tablaHistorial').DataTable({
      paging:true, searching:false, info:false, lengthChange:false, pageLength:10,
      order:[[0,'desc'],[1,'desc']],
      language:{ url:'https://cdn.datatables.net/plug-ins/1.13.8/i18n/es-ES.json' }
    });
  }

  // Visor de foto
  $(document).on('click', '#tablaHistorial img.thumb-bateria', function(){
    const full = this.getAttribute('data-full');
    if (!full) return;
    document.getElementById('fotoViewerImg').src = full;
    new bootstrap.Modal(document.getElementById('fotoViewer')).show();
  });

  // Abrir modal (no hacer nada si el botón está deshabilitado)
  $(document).on('click', '.btn-historial', function(){
    if (this.disabled) return; // bloquea cuando no hay medición

    const id     = $(this).data('id');
    const chasis = $(this).data('chasis') || '—';
    if (!id) return;

    $histChasis.text(chasis);
    $tbodyHist.empty();
    $loading.show();
    modal.show();

    fetch(`${API_HISTORIAL}?id=${encodeURIComponent(id)}`)
      .then(r => {
        if (!r.ok) throw new Error('HTTP '+r.status);
        return r.text();
      })
      .then(txt => {
        try { return JSON.parse(txt); }
        catch (e) { throw new Error('JSON inválido: '+txt.slice(0,300)); }
      })
      .then(json=>{
        $loading.hide();
        if (json.error) {
          $tbodyHist.html('<tr><td colspan="7" class="text-center text-danger">Servidor: '+escapeHtml(json.error)+'</td></tr>');
          return;
        }
        const rows = Array.isArray(json.data) ? json.data : [];
        renderHistorial(rows);
      })
      .catch(err=>{
        $loading.hide();
        console.error('Historial ERROR:', err);
        $tbodyHist.html('<tr><td colspan="7" class="text-center text-danger">Error al cargar historial</td></tr>');
      });
  });

  // Limpieza al cerrar
  document.getElementById('modalHistorial').addEventListener('hidden.bs.modal', ()=>{
    if ($.fn.DataTable.isDataTable('#tablaHistorial')) {
      $('#tablaHistorial').DataTable().destroy();
    }
    $tbodyHist.empty();
  });

});
</script>
</body>
</html>
