<?php
// views/pages/api_app/vehiculos_ingresados.php
// API: Vehículos SIN impronta (por defecto) con id_estado >= 6 y condicion = 1
// Métodos: GET, OPTIONS
// Parámetros opcionales:
//   - q, chasis, motor, marca, modelo, color
//   - estado_min (int, default 6)
//   - exact=1 -> igualdad en chasis/motor; exact=0 -> LIKE (default 0)
//   - trim=1  -> TRIM en comparaciones exactas (default 0)
//   - cond=all -> ignora d.condicion=1 (default '1')
//   - impronta=all -> desactiva filtro “sin impronta” (default: faltante)
//   - order_by: chasis,motor,marca,modelo,color,id_estado
//   - order_dir: ASC|DESC (default ASC)
//   - _debug=1 -> info de depuración

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }

require __DIR__ . '/../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'Error de conexión a BD']); exit; }
if (function_exists('mysqli_report')) mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
mysqli_set_charset($db, 'utf8mb4');

/* Helpers */
function qp(string $k, $d=null){ return $_GET[$k] ?? $d; }
function likeWrap(?string $s): ?string { return $s === null ? null : ('%'.$s.'%'); }

try {
  $DEBUG = ['meta'=>['time'=>date('Y-m-d H:i:s'),'tz'=>date_default_timezone_get()]];

  $estadoMin = (int)(qp('estado_min', 6));
  $orderByWhitelist = [
    'chasis'=>'d.chasis','motor'=>'d.motor','marca'=>'d.marca',
    'modelo'=>'d.modelo','color'=>'d.color','id_estado'=>'d.id_estado'
  ];
  $orderByKey  = strtolower((string)qp('order_by', 'chasis'));
  $orderBySql  = $orderByWhitelist[$orderByKey] ?? 'd.chasis';
  $orderDir    = strtoupper((string)qp('order_dir', 'ASC'));
  $orderDirSql = ($orderDir === 'DESC') ? 'DESC' : 'ASC';

  $exact   = (int)qp('exact', 0) === 1;
  $trimCmp = (int)qp('trim', 0) === 1;
  $condCtl = strtolower((string)qp('cond', '1'));       // '1' | 'all'
  $impCtl  = strtolower((string)qp('impronta', 'faltante')); // 'faltante' | 'all'

  $q       = qp('q');
  $chasis  = qp('chasis');
  $motor   = qp('motor');
  $marca   = qp('marca');
  $modelo  = qp('modelo');
  $color   = qp('color');

  $where  = ['d.id_estado >= ?'];
  $params = [$estadoMin];
  $types  = 'i';

  if ($condCtl !== 'all') {
    $where[] = 'd.condicion = 1';
  }

  if ($q) {
    $where[] = '(d.chasis LIKE ? OR d.motor LIKE ?)';
    $params[] = likeWrap($q);
    $params[] = likeWrap($q);
    $types   .= 'ss';
  }

  if ($chasis) {
    if ($exact) {
      $where[] = $trimCmp ? 'TRIM(d.chasis) = TRIM(?)' : 'd.chasis = ?';
      $params[] = $chasis;
    } else {
      $where[] = 'd.chasis LIKE ?';
      $params[] = likeWrap($chasis);
    }
    $types .= 's';
  }

  if ($motor) {
    if ($exact) {
      $where[] = $trimCmp ? 'TRIM(d.motor) = TRIM(?)' : 'd.motor = ?';
      $params[] = $motor;
    } else {
      $where[] = 'd.motor LIKE ?';
      $params[] = likeWrap($motor);
    }
    $types .= 's';
  }

  if ($marca)  { $where[]='d.marca LIKE ?';  $params[]=likeWrap($marca);  $types.='s'; }
  if ($modelo) { $where[]='d.modelo LIKE ?'; $params[]=likeWrap($modelo); $types.='s'; }
  if ($color)  { $where[]='d.color LIKE ?';  $params[]=likeWrap($color);  $types.='s'; }

  // Filtro: SIN impronta (por defecto)
  $improntaSql = '';
  if ($impCtl !== 'all') {
    // Opción 1: NOT EXISTS (más robusto)
    $improntaSql = " AND NOT EXISTS (
      SELECT 1 FROM tb_impronta i
      WHERE i.id_datadai = d.iddatadai
        AND i.condicion = 1
    )";
  }

  $whereSql = implode(' AND ', $where) . $improntaSql;

  $sql = "
    SELECT
      d.iddatadai,
      d.chasis,
      d.motor,
      d.marca,
      d.modelo,
      d.color,
      d.id_estado
    FROM tb_datadai d
    WHERE $whereSql
    ORDER BY $orderBySql $orderDirSql
  ";

  $DEBUG['query'] = [
    'where'  => $whereSql,
    'types'  => $types,
    'params' => $params,
    'order'  => "$orderBySql $orderDirSql",
    'sql'    => $sql
  ];

  $stmt = $db->prepare($sql);
  $stmt->bind_param($types, ...$params);
  $stmt->execute();
  $result = $stmt->get_result();

  $items = [];
  while ($row = $result->fetch_assoc()) {
    $items[] = [
      'iddatadai' => (int)$row['iddatadai'],
      'chasis'    => $row['chasis'],
      'motor'     => $row['motor'],
      'marca'     => $row['marca'],
      'modelo'    => $row['modelo'],
      'color'     => $row['color'],
      'id_estado' => (int)$row['id_estado'],
    ];
  }
  $stmt->close();

  $resp = ['success'=>true, 'total'=>count($items), 'items'=>$items];
  if ((int)qp('_debug', 0) === 1) { $resp['_debug'] = $DEBUG; }

  echo json_encode($resp, JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Error interno','error'=>$e->getMessage()], JSON_UNESCAPED_UNICODE);
}
