<?php
// vehiculos_con_novedades.php — Lista vehículos con novedades + detalle de incidentes (opcional)
// Métodos: GET, OPTIONS
//
// Parámetros GET opcionales:
//   - dai=Simple              (por defecto 'Simple')
//   - chasis, marca, modelo, color  (filtros; exact=1 -> igualdad; default LIKE)
//   - q                       (busca en chasis/marca/modelo/color vía LIKE)
//   - exact=0|1               (default 0)
//   - limit=50                (1..500)
//   - offset=0                (>=0)
//   - order_by                (fecha_programada|chasis|marca|modelo|color|total_incidentes)
//   - order_dir               (ASC|DESC)
//   - incidentes=0|1          (0 por defecto; 1 -> agrega array de incidentes por vehículo de la página)
//   - _debug=1                (agrega info de depuración)
//
// Respuesta JSON (ejemplo):
// {
//  "success": true,
//  "total": 12,
//  "items": [
//    {
//      "marca": "...", "modelo": "...", "chasis": "...", "color": "...",
//      "fecha_programada": "2025-10-31",    // o null
//      "total_incidentes": 3,
//      "incidentes": [                      // solo si incidentes=1
//        {
//          "incident_id": 101,
//          "detallezona": "Parachoques",
//          "seccion": null,
//          "detalleparte": "Frontal",
//          "detallenovedad": "Golpe leve",
//          "observacion": "Abolladura lado derecho",
//          "foto": "inc_20251015_223012_a1b2c3d4.jpg",
//          "severidad": "Golpe leve",       // alias pedido
//          "nombre_medida_catalogo": "Reparar pintura",
//          "medida_tomada": "Pendiente"
//        }
//      ]
//    }
//  ]
// }

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS / CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== Conexión ===== */
require_once __DIR__ . '/../conex.php'; // ajusta la ruta si es necesario
$db = $conn ?? ($conex ?? null);
if (!$db) { echo json_encode(['success'=>false,'message'=>'Sin conexión a la base de datos.']); exit; }
if (function_exists('mysqli_report')) {
  mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
}
mysqli_set_charset($db, 'utf8mb4');

/* ===== Parámetros ===== */
$GET        = $_GET + [];
$dai        = isset($GET['dai']) ? trim((string)$GET['dai']) : 'Simple';
$chasis     = isset($GET['chasis']) ? trim((string)$GET['chasis']) : '';
$marca      = isset($GET['marca']) ? trim((string)$GET['marca']) : '';
$modelo     = isset($GET['modelo']) ? trim((string)$GET['modelo']) : '';
$color      = isset($GET['color']) ? trim((string)$GET['color']) : '';
$q          = isset($GET['q']) ? trim((string)$GET['q']) : '';
$exact      = isset($GET['exact']) ? (int)$GET['exact'] : 0;
$limit      = isset($GET['limit']) ? max(1, min(500, (int)$GET['limit'])) : 50;
$offset     = isset($GET['offset']) ? max(0, (int)$GET['offset']) : 0;
$order_by   = isset($GET['order_by']) ? trim((string)$GET['order_by']) : '';
$order_dir  = strtoupper(isset($GET['order_dir']) ? trim((string)$GET['order_dir']) : '');
$withInc    = isset($GET['incidentes']) ? (int)$GET['incidentes'] : 0;

/* ===== Orden seguro ===== */
$allowed_order = [
  'fecha_programada' => 'rr.fecha_programada',
  'chasis'           => 'd.chasis',
  'marca'            => 'd.marca',
  'modelo'           => 'd.modelo',
  'color'            => 'd.color',
  'total_incidentes' => 'inc.total_incidentes'
];
$ob = $allowed_order[$order_by] ?? null;
$od = ($order_dir === 'ASC' || $order_dir === 'DESC') ? $order_dir : 'ASC';

/* ===== WHERE dinámico ===== */
$where   = [];
$params  = [];
$ptypes  = '';

$where[] = 'd.condicion = 1';
$where[] = 'd.dai = ?';          $ptypes .= 's'; $params[] = $dai;
// Debe tener al menos una novedad activa
$where[] = 'EXISTS (SELECT 1 FROM tb_incidente i WHERE i.id_datadai = d.iddatadai AND i.condicion = 1)';

if ($chasis !== '') {
  if ($exact) { $where[] = 'd.chasis = ?'; $ptypes .= 's'; $params[] = $chasis; }
  else        { $where[] = 'd.chasis LIKE ?'; $ptypes .= 's'; $params[] = "%$chasis%"; }
}
if ($marca !== '') {
  if ($exact) { $where[] = 'd.marca = ?'; $ptypes .= 's'; $params[] = $marca; }
  else        { $where[] = 'd.marca LIKE ?'; $ptypes .= 's'; $params[] = "%$marca%"; }
}
if ($modelo !== '') {
  if ($exact) { $where[] = 'd.modelo = ?'; $ptypes .= 's'; $params[] = $modelo; }
  else        { $where[] = 'd.modelo LIKE ?'; $ptypes .= 's'; $params[] = "%$modelo%"; }
}
if ($color !== '') {
  if ($exact) { $where[] = 'd.color = ?'; $ptypes .= 's'; $params[] = $color; }
  else        { $where[] = 'd.color LIKE ?'; $ptypes .= 's'; $params[] = "%$color%"; }
}
if ($q !== '') {
  $where[]  = '(d.chasis LIKE ? OR d.marca LIKE ? OR d.modelo LIKE ? OR d.color LIKE ?)';
  $ptypes  .= 'ssss';
  $likeq    = "%$q%";
  array_push($params, $likeq, $likeq, $likeq, $likeq);
}
$where_sql = $where ? 'WHERE '.implode(' AND ', $where) : '';

/* ===== Subconsultas ===== */
$sub_rr = "
  SELECT id_datadai, MAX(fecha_programada) AS fecha_programada
  FROM tb_reparacion
  GROUP BY id_datadai
";
$sub_inc = "
  SELECT id_datadai, COUNT(*) AS total_incidentes
  FROM tb_incidente
  WHERE condicion = 1
  GROUP BY id_datadai
";

/* ===== SQL base ===== */
$sql_base = "
  FROM tb_datadai d
  LEFT JOIN ($sub_rr) rr   ON rr.id_datadai = d.iddatadai
  LEFT JOIN ($sub_inc) inc ON inc.id_datadai = d.iddatadai
  $where_sql
";

/* ===== COUNT ===== */
$sql_count = "SELECT COUNT(*) c $sql_base";

/* ===== ORDER por defecto (como tu consulta) ===== */
if (!$ob) {
  $order_sql = "ORDER BY (rr.fecha_programada IS NULL), rr.fecha_programada DESC, d.chasis ASC";
} else {
  if ($ob === 'rr.fecha_programada') {
    $order_sql = "ORDER BY (rr.fecha_programada IS NULL) ASC, rr.fecha_programada $od, d.chasis ASC";
  } else {
    $order_sql = "ORDER BY $ob $od, d.chasis ASC";
  }
}

/* ===== DATA ===== */
$sql_data = "
  SELECT
    d.iddatadai,
    d.marca,
    d.modelo,
    d.chasis,
    d.color,
    rr.fecha_programada,
    COALESCE(inc.total_incidentes,0) AS total_incidentes
  $sql_base
  $order_sql
  LIMIT ? OFFSET ?
";

try {
  /* COUNT */
  $stmt = mysqli_prepare($db, $sql_count);
  if ($ptypes !== '') { mysqli_stmt_bind_param($stmt, $ptypes, ...$params); }
  mysqli_stmt_execute($stmt);
  $res  = mysqli_stmt_get_result($stmt);
  $row  = $res->fetch_assoc();
  $total = (int)($row['c'] ?? 0);
  mysqli_stmt_close($stmt);

  /* DATA */
  $stmt = mysqli_prepare($db, $sql_data);
  $ptypes2 = $ptypes . 'ii';
  $params2 = $params;
  $params2[] = $limit;
  $params2[] = $offset;
  mysqli_stmt_bind_param($stmt, $ptypes2, ...$params2);
  mysqli_stmt_execute($stmt);
  $res = mysqli_stmt_get_result($stmt);

  $items = [];
  $idmap = []; // iddatadai => index en $items
  while ($r = $res->fetch_assoc()) {
    $idx = count($items);
    $items[] = [
      'marca'            => $r['marca'],
      'modelo'           => $r['modelo'],
      'chasis'           => $r['chasis'],
      'color'            => $r['color'],
      'fecha_programada' => $r['fecha_programada'] ? substr($r['fecha_programada'], 0, 10) : null,
      'total_incidentes' => (int)$r['total_incidentes'],
      // se añadirá 'incidentes' si se solicita
    ];
    $idmap[(int)$r['iddatadai']] = $idx;
  }
  mysqli_stmt_close($stmt);

  /* ===== Incidentes (opcional) =====
     Si incidentes=1, traemos para los id_datadai de la página actual.
     Campos pedidos:
       - i.idincidente AS incident_id
       - z.detallezona, seccion=NULL, p.detalleparte, n.detallenovedad
       - i.observacion, i.foto
       - n.detallenovedad AS severidad (alias solicitado)
       - m.medida AS nombre_medida_catalogo
       - i.medida AS medida_tomada
  */
  if ($withInc && !empty($idmap)) {
    $ids = array_keys($idmap);
    // armamos placeholders y tipos
    $in_placeholders = implode(',', array_fill(0, count($ids), '?'));
    $in_types = str_repeat('i', count($ids));

    $sql_inc = "
      SELECT
        i.idincidente                           AS incident_id,
        i.id_datadai,
        z.detallezona,
        NULL                                    AS seccion,
        p.detalleparte,
        n.detallenovedad,
        i.observacion,
        i.foto,
        n.detallenovedad                        AS severidad,
        m.medida                                AS nombre_medida_catalogo,
        i.medida                                AS medida_tomada
      FROM tb_incidente i
      LEFT JOIN tb_zona    z ON z.idzona    = i.id_zona
      LEFT JOIN tb_parte   p ON p.idparte   = i.id_parte
      LEFT JOIN tb_novedad n ON n.idnovedad = i.id_novedad
      LEFT JOIN tb_medida  m ON m.id        = i.id_medida
      WHERE i.condicion = 1
        AND i.id_datadai IN ($in_placeholders)
      ORDER BY i.id_datadai, i.idincidente
    ";

    $stmt = mysqli_prepare($db, $sql_inc);
    mysqli_stmt_bind_param($stmt, $in_types, ...$ids);
    mysqli_stmt_execute($stmt);
    $res = mysqli_stmt_get_result($stmt);

    // inicializa array por item
    foreach ($idmap as $idx) { $items[$idx]['incidentes'] = []; }

    while ($row = $res->fetch_assoc()) {
      $id_d = (int)$row['id_datadai'];
      if (!isset($idmap[$id_d])) continue;
      $idx = $idmap[$id_d];
      $items[$idx]['incidentes'][] = [
        'incident_id'            => (int)$row['incident_id'],
        'detallezona'            => $row['detallezona'],
        'seccion'                => null, // según tu SQL original
        'detalleparte'           => $row['detalleparte'],
        'detallenovedad'         => $row['detallenovedad'],
        'observacion'            => $row['observacion'],
        'foto'                   => $row['foto'],
        'severidad'              => $row['severidad'],
        'nombre_medida_catalogo' => $row['nombre_medida_catalogo'],
        'medida_tomada'          => $row['medida_tomada'],
      ];
    }
    mysqli_stmt_close($stmt);
  }

  $out = ['success'=>true, 'total'=>$total, 'items'=>$items];
  if (isset($GET['_debug'])) {
    $out['_debug'] = [
      'order_sql' => $order_sql,
      'where'     => $where,
      'limit'     => $limit,
      'offset'    => $offset,
      'incidentes'=> $withInc
    ];
  }
  echo json_encode($out, JSON_UNESCAPED_UNICODE);
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'success'=>false,
    'message'=>'Error interno.',
    'error'=>$e->getMessage()
  ], JSON_UNESCAPED_UNICODE);
}
