<?php
// views/pages/api_app/ubicacion_select.php
// Devuelve ubicaciones activas por bodega, mostrando SOLO el texto concatenado:
//   text = CONCAT_WS(' - ', u.detalle_ubicacion, u.letra)
// Útil para llenar <select> o Select2.
//
// Parámetros (GET/POST):
//   - idusuario : int   -> si no llega, intenta $_SESSION['idusuario']
//   - id_bodega : int   -> si llega, NO consulta tb_usuario
//   - q         : str   -> filtro (busca en detalle_ubicacion y letra)
//   - limit     : 1..500 (default 200)
//   - offset    : >=0    (default 0)
//   - format    : 'json' (default) | 'html'  -> salida
//   - _debug    : 1      -> info de depuración

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS / CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== Conexión ===== */
$pathConex = realpath(__DIR__.'/../conex.php') ?: realpath(__DIR__.'/../../conex.php') ?: realpath(__DIR__.'/conex.php');
if (!$pathConex) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'No se encontró conex.php']); exit; }
require_once $pathConex;

$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'Sin conexión a BD']); exit; }
if (function_exists('mysqli_report')) mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
$db->set_charset('utf8mb4');
@$db->query("SET time_zone='-05:00'");

/* ===== Sesión (opcional) ===== */
if (session_status() === PHP_SESSION_NONE) { @session_start(); }

/* ===== Helpers ===== */
function qp(string $k, $def=null) {
  if (isset($_POST[$k])) return $_POST[$k];
  if (isset($_GET[$k]))  return $_GET[$k];
  return $def;
}
function fail(int $code, string $msg, array $extra=[]): void {
  http_response_code($code);
  echo json_encode(['success'=>false,'message'=>$msg,'data'=>$extra], JSON_UNESCAPED_UNICODE); exit;
}

/* ===== Input ===== */
$format    = strtolower((string)qp('format','json')) === 'html' ? 'html' : 'json';
$q         = trim((string)qp('q',''));
$limit     = filter_var(qp('limit', 200), FILTER_VALIDATE_INT);  if (!$limit || $limit<1) $limit=200; if ($limit>500) $limit=500;
$offset    = filter_var(qp('offset', 0),   FILTER_VALIDATE_INT);  if ($offset===false || $offset<0) $offset=0;
$debug     = (string)qp('_debug','0') === '1';

$id_bodega = (int)(qp('id_bodega', 0));
$idusuario = (int)(qp('idusuario', 0));

/* ===== Resolver id_bodega ===== */
if ($id_bodega <= 0) {
  if ($idusuario <= 0 && !empty($_SESSION['idusuario'])) {
    $idusuario = (int)$_SESSION['idusuario'];
  }
  if ($idusuario <= 0) {
    fail(422, 'Falta idusuario o id_bodega');
  }

  $sqlUB = "SELECT id_bodega FROM tb_usuario WHERE idusuario = ? LIMIT 1";
  $stUB = $db->prepare($sqlUB);
  $stUB->bind_param('i', $idusuario);
  $stUB->execute();
  $resUB = $stUB->get_result();
  $rowUB = $resUB->fetch_assoc();
  $stUB->close();

  if (!$rowUB || (int)$rowUB['id_bodega'] <= 0) {
    fail(404, 'No se encontró bodega para el usuario', ['idusuario'=>$idusuario]);
  }
  $id_bodega = (int)$rowUB['id_bodega'];
}

/* ===== WHERE =====
   Solo ubicaciones activas de esa bodega, con búsqueda opcional en detalle_ubicacion y letra
*/
$where = "WHERE u.activo = 1 AND u.id_bodega = ?";
$params = [$id_bodega];
$types  = 'i';

if ($q !== '') {
  $where .= " AND (UPPER(u.detalle_ubicacion) LIKE UPPER(?) OR UPPER(u.letra) LIKE UPPER(?))";
  $like = '%'.$q.'%';
  $params[] = $like; $params[] = $like;
  $types   .= 'ss';
}

/* ===== TOTAL ===== */
$sqlCount = "SELECT COUNT(*) AS total FROM tb_ubicacion u $where";
$stC = $db->prepare($sqlCount);
$refs = [ &$types ];
foreach ($params as $k => &$v) { $refs[] = &$v; }
call_user_func_array([$stC,'bind_param'], $refs);
$stC->execute();
$resC = $stC->get_result();
$rowC = $resC->fetch_assoc();
$total = (int)($rowC['total'] ?? 0);
$stC->close();

if ($total === 0) {
  if ($format === 'html') {
    header('Content-Type: text/html; charset=utf-8');
    echo '<option value="">(Sin ubicaciones)</option>';
    exit;
  }
  echo json_encode([
    'success'=>true,
    'total'=>0,
    'items'=>[],
    '_debug'=>$debug ? ['id_bodega'=>$id_bodega,'q'=>$q] : null
  ], JSON_UNESCAPED_UNICODE);
  exit;
}

/* ===== Página ===== */
$sql = "SELECT
          u.idubicacion,
          /* Texto concatenado: detalle_ubicacion - letra */
          TRIM(CONCAT_WS(' - ', NULLIF(u.detalle_ubicacion,''), NULLIF(u.letra,''))) AS texto
        FROM tb_ubicacion u
        $where
        ORDER BY u.idubicacion ASC, u.letra ASC, u.idubicacion ASC
        LIMIT ? OFFSET ?";
$st = $db->prepare($sql);

$types2 = $types.'ii';
$args = $params;
$lim = (int)$limit; $off = (int)$offset;
$args[] = $lim; $args[] = $off;

$refs = [ &$types2 ];
foreach ($args as $k => &$v) { $refs[] = &$v; }
call_user_func_array([$st,'bind_param'], $refs);

$st->execute();
$res = $st->get_result();

/* ===== Salida ===== */
if ($format === 'html') {
  header('Content-Type: text/html; charset=utf-8');
  $html = '';
  while ($r = $res->fetch_assoc()) {
    $id   = (int)$r['idubicacion'];
    $text = htmlspecialchars((string)($r['texto'] ?? ''), ENT_QUOTES, 'UTF-8');
    $html .= '<option value="'.$id.'">'.$text.'</option>';
  }
  $st->close();
  echo $html;
  exit;
}

$items = [];
while ($r = $res->fetch_assoc()) {
  $items[] = [
    'id'   => (int)$r['idubicacion'],
    'text' => (string)($r['texto'] ?? ''),
  ];
}
$st->close();

echo json_encode([
  'success'=>true,
  'total'=>$total,
  'items'=>$items,                // listo para Select2: muestra SOLO "detalle_ubicacion - letra"
  'bodega'=>$id_bodega,
  '_debug'=>$debug ? compact('q','limit','offset') : null
], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
