<?php
// views/pages/api_app/transporte_por_llegar.php
// API (POST JSON): SOLO lista transportes "por llegar" con status = 'COORDINADO'
// Definición "por llegar": fecha_descarga IS NULL O fecha_descarga >= (ayer 00:00) (zona Guayaquil)
//
// Body JSON opcional:
//   - limit: 1..200 (default 50)
//   - offset: >=0   (default 0)
//   - _debug: 1     (opcional) -> incluir info de depuración

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Método no permitido. Use POST JSON.']); exit;
}

/* ===== CONEXIÓN ===== */
$pathConex = realpath(__DIR__.'/../conex.php') ?: realpath(__DIR__.'/../../conex.php') ?: null;
if (!$pathConex) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'No se encontró conex.php']); exit; }
require_once $pathConex;

$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'Error de conexión a BD']); exit; }
mysqli_set_charset($db, 'utf8mb4');
if (function_exists('mysqli_report')) mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
/* Alinear NOW()/CURRENT_DATE con Guayaquil */
@$db->query("SET time_zone='-05:00'");

/* ===== INPUT ===== */
$raw = file_get_contents('php://input');
$in  = json_decode($raw, true);
if (!is_array($in)) $in = $_POST;

$limit  = filter_var($in['limit']  ?? 50, FILTER_VALIDATE_INT);
$offset = filter_var($in['offset'] ?? 0,  FILTER_VALIDATE_INT);
if ($limit === false || $limit < 1) $limit = 50; if ($limit > 200) $limit = 200;
if ($offset === false || $offset < 0) $offset = 0;
$debug = (string)($in['_debug'] ?? '0') === '1';

/* ===== WHERE fijo =====
   - a.condicion = 1
   - a.status = 'COORDINADO'
   - a.fecha_descarga IS NULL OR a.fecha_descarga >= (CURRENT_DATE - INTERVAL 1 DAY)
*/
$where = "WHERE a.condicion = 1
          AND a.status = 'COORDINADO'
          AND (a.fecha_descarga IS NULL OR a.fecha_descarga >= (CURRENT_DATE - INTERVAL 1 DAY))";

try {
  /* ---- TOTAL ---- */
  $sqlCount = "
    SELECT COUNT(*) AS total
      FROM tb_asignar a
      INNER JOIN tb_transporte t ON t.id = a.transportista
      $where
  ";
  $st = $db->prepare($sqlCount);
  $st->execute();
  $res = $st->get_result();
  $row = $res->fetch_assoc();
  $total = (int)($row['total'] ?? 0);
  $st->close();

  if ($total === 0) {
    echo json_encode([
      'success'=>true,
      'total'=>0,
      'items'=>[],
      '_debug'=>$debug ? ['where'=>$where] : null
    ], JSON_UNESCAPED_UNICODE);
    exit;
  }

  /* ---- 1) IDS paginados ---- */
  $sqlIds = "
    SELECT a.id AS id_asignar
      FROM tb_asignar a
      INNER JOIN tb_transporte t ON t.id = a.transportista
      $where
      ORDER BY a.fecha_carga DESC, a.id DESC
      LIMIT ? OFFSET ?
  ";
  $st = $db->prepare($sqlIds);
  $st->bind_param('ii', $limit, $offset);
  $st->execute();
  $res = $st->get_result();

  $ids = [];
  while ($r = $res->fetch_assoc()) $ids[] = (int)$r['id_asignar'];
  $st->close();

  if (!$ids) {
    echo json_encode([
      'success'=>true,
      'total'=>$total,
      'items'=>[],
      '_debug'=>$debug ? ['nota'=>'Sin IDs en esta página'] : null
    ], JSON_UNESCAPED_UNICODE);
    exit;
  }

  /* ---- 2) Datos + vehículos (vínculos activos) ---- */
  $idsIn = implode(',', array_map('intval', $ids));
  $sqlData = "
    SELECT
      a.id                AS id_asignar,
      t.id                AS id_transportista,
      t.nombre            AS transportista_nombre,
      a.numero_doc,
      a.conductor,
      a.placa,
      DATE_FORMAT(a.fecha_carga,    '%Y-%m-%d %H:%i') AS fecha_carga,
      DATE_FORMAT(a.fecha_descarga, '%Y-%m-%d %H:%i') AS fecha_descarga,
      a.guia,
      d.iddatadai         AS id_datadai,
      d.chasis,
      d.marca,
      d.modelo,
      d.color,
      d.id_estado
    FROM tb_asignar a
    INNER JOIN tb_transporte t      ON t.id = a.transportista
    /* Solo vínculos ACTIVOS (ad.condicion = 1) pero permitimos asignaciones sin vehículos aún */
    LEFT JOIN tb_asignar_datadai ad ON ad.id_asignar = a.id AND ad.condicion = 1
    LEFT JOIN tb_datadai d          ON d.iddatadai = ad.id_datadai
    WHERE a.id IN ($idsIn)
    ORDER BY a.fecha_carga DESC, a.id DESC, d.chasis ASC
  ";
  $rs = $db->query($sqlData);

  $items = [];
  while ($r = $rs->fetch_assoc()) {
    $ida = (int)$r['id_asignar'];
    if (!isset($items[$ida])) {
      $items[$ida] = [
        'id_asignar'    => $ida,
        'transportista' => [
          'id'     => (int)$r['id_transportista'],
          'nombre' => (string)$r['transportista_nombre'],
        ],
        'numero_doc'     => (string)($r['numero_doc'] ?? ''),
        'conductor'      => (string)($r['conductor'] ?? ''),
        'placa'          => (string)($r['placa'] ?? ''),
        'fecha_carga'    => $r['fecha_carga'] ?? null,
        'fecha_descarga' => $r['fecha_descarga'] ?? null,
        'guia'           => (string)($r['guia'] ?? ''),
        'vehiculos'      => [],
      ];
    }
    if (!empty($r['id_datadai'])) {
      $items[$ida]['vehiculos'][] = [
        'id_datadai' => (int)$r['id_datadai'],
        'chasis'     => (string)$r['chasis'],
        'marca'      => (string)($r['marca'] ?? ''),
        'modelo'     => (string)($r['modelo'] ?? ''),
        'color'      => (string)($r['color'] ?? ''),
        'id_estado'      => (string)($r['id_estado'] ?? ''),
      ];
    }
  }

  echo json_encode([
    'success'=>true,
    'total'=>$total,
    'items'=>array_values($items),
    '_debug'=>$debug ? ['where'=>$where,'ids'=>$ids] : null
  ], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Error en consulta','error'=>$e->getMessage()]);
}
