<?php
// views/pages/api_app/transporte_por_llegar.php
// API: Transportes por llegar + búsqueda por placa + vehículos por transporte
// Métodos: GET, OPTIONS
// Parámetros GET:
//   - placa        (opcional)  -> filtro por placa; si lo omites, devuelve TODOS los "por llegar"
//   - exact=1      (opcional)  -> igualdad exacta de placa (default 0 -> LIKE)
//   - por_llegar   1|0         -> 1 = fecha_descarga IS NULL o > NOW() (default 1)
//   - limit        1..200      -> default 50
//   - offset       >=0         -> default 0
//   - _debug=1     (opcional)  -> añade info de depuración a la respuesta

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== CORS / HEADERS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Access-Control-Allow-Methods: GET, OPTIONS');

/* 🔒 Anti-cache: navegador + proxies + preflight */
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Cache-Control: post-check=0, pre-check=0', false); // compat IE
header('Pragma: no-cache');                // HTTP/1.0
header('Expires: 0');                      // expirar inmediatamente
header('Surrogate-Control: no-store');     // proxies/CDN
header('Vary: Origin, Accept-Encoding');   // CORS/compresión
header('Access-Control-Max-Age: 0');       // no cachear preflight

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== CONEXIÓN ===== */
require_once __DIR__ . '/../conex.php';
if (!isset($conn) || !$conn instanceof mysqli) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Error de conexión a BD']); exit;
}
mysqli_set_charset($conn, 'utf8mb4');
if (function_exists('mysqli_report')) { mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); }

/* ===== HELPERS ===== */
function qp(string $k, $d=null){ return isset($_GET[$k]) ? $_GET[$k] : $d; }
function clamp_int($v, int $min, int $max): int {
  $n = filter_var($v, FILTER_VALIDATE_INT);
  if ($n === false) $n = $min;
  return max($min, min($max, $n));
}

/* ===== PARÁMETROS ===== */
$placa      = trim((string)qp('placa',''));
$exact      = (string)qp('exact','0') === '1' ? 1 : 0;
$porLlegar  = (string)qp('por_llegar','1') === '0' ? 0 : 1; // default: solo "por llegar"
$limit      = clamp_int(qp('limit',50), 1, 200);
$offset     = clamp_int(qp('offset',0), 0, PHP_INT_MAX);
$debug      = (string)qp('_debug','0') === '1';

/* ===== WHERE DINÁMICO =====
   Tablas esperadas:
   - tb_asignar a   (id, transportista, numero_doc, conductor, placa, fecha_carga, fecha_descarga, guia, condicion)
   - tb_transporte t(id, nombre)
   - tb_asignar_datadai ad (id_asignar, id_datadai)
   - tb_datadai d   (iddatadai, chasis, marca, modelo, color)
*/
$where  = [];
$params = [];
$types  = '';

$where[] = 'a.condicion = 1';
if ($porLlegar === 1) {
  // "Por llegar": sin descarga aún o con descarga futura
  $where[] = '(a.fecha_descarga IS NULL OR a.fecha_descarga > NOW())';
}
if ($placa !== '') {
  if ($exact === 1) {
    $where[]   = 'UPPER(a.placa) = UPPER(?)';
    $params[]  = $placa;
  } else {
    $where[]   = 'UPPER(a.placa) LIKE UPPER(?)';
    $params[]  = '%'.$placa.'%';
  }
  $types .= 's';
}
$whereSql = count($where) ? 'WHERE '.implode(' AND ', $where) : '';

try {
  /* ---- TOTAL DE TRANSPORTES (por criterio) ---- */
  $sqlCount = "
    SELECT COUNT(*) AS total
    FROM tb_asignar a
    INNER JOIN tb_transporte t ON t.id = a.transportista
    $whereSql
  ";
  $stmt = $conn->prepare($sqlCount);
  if ($types !== '') {
    $refs = [ &$types ];
    foreach ($params as $k => &$v) { $refs[] = &$v; }
    call_user_func_array([$stmt,'bind_param'], $refs);
  }
  $stmt->execute();
  $res   = $stmt->get_result();
  $row   = $res->fetch_assoc();
  $total = (int)($row['total'] ?? 0);
  $stmt->close();

  if ($total === 0) {
    echo json_encode([
      'success'=>true, 'total'=>0, 'items'=>[],
      '_debug'=>$debug ? ['where'=>$where,'params'=>$params,'types'=>$types] : null
    ], JSON_UNESCAPED_UNICODE);
    exit;
  }

  /* ---- 1) IDS de asignación paginados ---- */
  $sqlIds = "
    SELECT a.id AS id_asignar
    FROM tb_asignar a
    INNER JOIN tb_transporte t ON t.id = a.transportista
    $whereSql
    ORDER BY a.fecha_carga DESC, a.id DESC
    LIMIT ? OFFSET ?
  ";
  $stmt = $conn->prepare($sqlIds);

  $typesIds = $types.'ii';
  $args = $params;
  $args[] = $limit;
  $args[] = $offset;
  $args[count($args)-2] = (int)$args[count($args)-2];
  $args[count($args)-1] = (int)$args[count($args)-1];

  $refs = [ &$typesIds ];
  foreach ($args as $k => &$v) { $refs[] = &$v; }
  call_user_func_array([$stmt,'bind_param'], $refs);

  $stmt->execute();
  $res = $stmt->get_result();

  $ids = [];
  while ($r = $res->fetch_assoc()) { $ids[] = (int)$r['id_asignar']; }
  $stmt->close();

  if (empty($ids)) {
    echo json_encode([
      'success'=>true, 'total'=>$total, 'items'=>[],
      '_debug'=>$debug ? ['nota'=>'Sin IDs en la página actual','where'=>$where] : null
    ], JSON_UNESCAPED_UNICODE);
    exit;
  }

  /* ---- 2) Datos + vehículos (LEFT JOIN) para esos IDs ---- */
  $idsSafe = array_map('intval', $ids);
  $inList  = implode(',', $idsSafe);

  $sqlData = "
    SELECT
      a.id                AS id_asignar,
      t.id                AS id_transportista,
      t.nombre            AS transportista_nombre,
      a.numero_doc,
      a.conductor,
      a.placa,
      DATE_FORMAT(a.fecha_carga,    '%Y-%m-%d %H:%i') AS fecha_carga,
      DATE_FORMAT(a.fecha_descarga, '%Y-%m-%d %H:%i') AS fecha_descarga,
      a.guia,
      d.iddatadai         AS id_datadai,
      d.chasis,
      d.marca,
      d.modelo,
      d.color
    FROM tb_asignar a
    INNER JOIN tb_transporte t     ON t.id = a.transportista
    LEFT JOIN tb_asignar_datadai ad ON ad.id_asignar = a.id
    LEFT JOIN tb_datadai d         ON d.iddatadai = ad.id_datadai
    WHERE a.id IN ($inList)
    ORDER BY a.fecha_carga DESC, a.id DESC, d.chasis ASC
  ";

  $rs = $conn->query($sqlData);

  $asignaciones = [];
  while ($r = $rs->fetch_assoc()) {
    $ida = (int)$r['id_asignar'];
    if (!isset($asignaciones[$ida])) {
      $asignaciones[$ida] = [
        'id_asignar'    => $ida,
        'transportista' => [
          'id'     => (int)$r['id_transportista'],
          'nombre' => (string)$r['transportista_nombre']
        ],
        'numero_doc'     => (string)($r['numero_doc'] ?? ''),
        'conductor'      => (string)($r['conductor'] ?? ''),
        'placa'          => (string)($r['placa'] ?? ''),
        'fecha_carga'    => $r['fecha_carga'] ?? null,
        'fecha_descarga' => $r['fecha_descarga'] ?? null,
        'guia'           => (string)($r['guia'] ?? ''),
        'vehiculos'      => []
      ];
    }
    // Adjuntar vehículo si existe
    if (!empty($r['id_datadai'])) {
      $asignaciones[$ida]['vehiculos'][] = [
        'id_datadai' => (int)$r['id_datadai'],
        'chasis'     => (string)$r['chasis'],
        'marca'      => (string)($r['marca'] ?? ''),
        'modelo'     => (string)($r['modelo'] ?? ''),
        'color'      => (string)($r['color'] ?? ''),
      ];
    }
  }

  echo json_encode([
    'success' => true,
    'total'   => $total,
    'items'   => array_values($asignaciones),
    '_debug'  => $debug ? ['where'=>$where,'params'=>$params,'types'=>$types,'ids'=>$idsSafe] : null
  ], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Error en consulta','error'=>$e->getMessage()]);
}
