<?php
// views/pages/api_app/lavada_marcar_min.php
// Solo registra INICIO o FIN en tb_lavada tocando ÚNICAMENTE:
// fecha_inicio, hora_inicio, fecha_fin, hora_fin, estado_lavado, id_usuario_l
// Sin autenticación.
// Modo: por chasis (resuelve id_datadai y toma el último registro activo) o por idlavada.

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== BD ===== */
require_once __DIR__ . '/../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'Sin conexión']); exit; }
if (function_exists('mysqli_report')) { mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); }
mysqli_set_charset($db, 'utf8mb4');
@$db->query("SET time_zone='-05:00'");

/* ===== INPUT ===== */
$raw = file_get_contents('php://input') ?: '';
$in  = json_decode($raw, true);
if (!is_array($in)) { $in = $_POST ?: []; }

// Puedes enviar: { chasis:"...", id_usuario_l:123 }  o  { idlavada: 999, id_usuario_l:123 }
$chasis     = strtoupper(trim((string)($in['chasis'] ?? '')));
$idlavadaIn = isset($in['idlavada']) ? (int)$in['idlavada'] : 0;
$idUsuarioL = isset($in['id_usuario_l']) ? (int)$in['id_usuario_l'] : 0;

if ($idUsuarioL <= 0) { http_response_code(400); echo json_encode(['success'=>false,'message'=>'id_usuario_l inválido']); exit; }
if ($chasis === '' && $idlavadaIn <= 0) {
  http_response_code(400);
  echo json_encode(['success'=>false,'message'=>'Envía chasis o idlavada']); exit;
}

try {
  $db->begin_transaction();
  $db->query("SET SESSION TRANSACTION ISOLATION LEVEL READ COMMITTED");

  // 1) Resolver idlavada objetivo
  if ($idlavadaIn > 0) {
    // Bloquea ese idlavada si está activo
    $q = $db->prepare("
      SELECT idlavada, id_datadai, fecha_inicio, hora_inicio, fecha_fin, hora_fin, estado_lavado, id_usuario_l, condicion
      FROM tb_lavada
      WHERE idlavada=? AND condicion=1
      LIMIT 1
      FOR UPDATE
    ");
    $q->bind_param('i', $idlavadaIn);
    $q->execute();
    $row = $q->get_result()->fetch_assoc();
    $q->close();
    if (!$row) { $db->rollback(); http_response_code(404); echo json_encode(['success'=>false,'message'=>'idlavada no encontrado o inactivo']); exit; }
  } else {
    // Por CHASIS: resuelve id_datadai y toma el ÚLTIMO activo
    $qd = $db->prepare("SELECT iddatadai FROM tb_datadai WHERE UPPER(chasis)=? LIMIT 1");
    $qd->bind_param('s', $chasis);
    $qd->execute();
    $rd = $qd->get_result()->fetch_assoc();
    $qd->close();
    if (!$rd) { $db->rollback(); http_response_code(404); echo json_encode(['success'=>false,'message'=>'Chasis no encontrado']); exit; }
    $iddatadai = (int)$rd['iddatadai'];

    $q = $db->prepare("
      SELECT idlavada, id_datadai, fecha_inicio, hora_inicio, fecha_fin, hora_fin, estado_lavado, id_usuario_l, condicion
      FROM tb_lavada
      WHERE id_datadai=? AND condicion=1
      ORDER BY fecha_registro DESC, idlavada DESC
      LIMIT 1
      FOR UPDATE
    ");
    $q->bind_param('i', $iddatadai);
    $q->execute();
    $row = $q->get_result()->fetch_assoc();
    $q->close();
    if (!$row) { $db->rollback(); http_response_code(404); echo json_encode(['success'=>false,'message'=>'No hay lavada activa para ese chasis']); exit; }
  }

  $idlavada = (int)$row['idlavada'];
  $fIni = $row['fecha_inicio']; $hIni = $row['hora_inicio'];
  $fFin = $row['fecha_fin'];    $hFin = $row['hora_fin'];
  $estadoActual = (string)$row['estado_lavado'];

  $hoy  = date('Y-m-d');
  $hora = date('H:i:s');

  $action = 'noop';
  $estadoNuevo = $estadoActual;

  // 2) Lógica mínima:
  // Si no tiene INICIO -> setear INICIO y estado='EN_PROCESO'
  if ((is_null($fIni) || $fIni === '') && (is_null($hIni) || $hIni === '')) {
    $u = $db->prepare("
      UPDATE tb_lavada
      SET fecha_inicio=?, hora_inicio=?, estado_lavado='EN_PROCESO', id_usuario_l=?
      WHERE idlavada=?
    ");
    $u->bind_param('ssii', $hoy, $hora, $idUsuarioL, $idlavada);
    $u->execute(); $u->close();
    $action = 'start';
    $estadoNuevo = 'EN_PROCESO';

  // Si ya tiene INICIO y no tiene FIN -> setear FIN y estado='TERMINADO'
  } elseif ((is_null($fFin) || $fFin === '') && (is_null($hFin) || $hFin === '')) {
    $u = $db->prepare("
      UPDATE tb_lavada
      SET fecha_fin=?, hora_fin=?, estado_lavado='TERMINADO', id_usuario_l=?
      WHERE idlavada=?
    ");
    $u->bind_param('ssii', $hoy, $hora, $idUsuarioL, $idlavada);
    $u->execute(); $u->close();
    $action = 'finish';
    $estadoNuevo = 'TERMINADO';
  }

  // 3) Leer valores finales para responder
  $c = $db->prepare("
    SELECT idlavada, fecha_inicio, hora_inicio, fecha_fin, hora_fin, estado_lavado, id_usuario_l
    FROM tb_lavada
    WHERE idlavada=? LIMIT 1
  ");
  $c->bind_param('i', $idlavada);
  $c->execute();
  $fin = $c->get_result()->fetch_assoc();
  $c->close();

  $db->commit();

  echo json_encode([
    'success' => true,
    'action'  => $action, // start | finish | noop
    'idlavada'=> $idlavada,
    'estado'  => (string)($fin['estado_lavado'] ?? $estadoNuevo),
    'data'    => [
      'fecha_inicio' => $fin['fecha_inicio'],
      'hora_inicio'  => $fin['hora_inicio'],
      'fecha_fin'    => $fin['fecha_fin'],
      'hora_fin'     => $fin['hora_fin'],
      'id_usuario_l' => (int)($fin['id_usuario_l'] ?? $idUsuarioL),
    ]
  ]);
  exit;

} catch (Throwable $e) {
  try { $db->rollback(); } catch (Throwable $ign) {}
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Error en operación','error'=>$e->getMessage()]);
  exit;
}
