<?php
// inventario_terminar.php
// Termina un inventario: estado -> 'TERMINADO', fecha_fin/hora_fin del sistema.
// Reglas:
// - Si NO recibes id_inventario, toma el último por fecha_inicio con estado='EN EJECUCION' y condicion=1.
// - Requiere al menos 1 detalle (tb_inventario_datadai) con condicion=1.
// - Responde JSON.

// === Suposición de columnas en tb_inventario_vin ===
// id (PK), estado, condicion, fecha_inicio, fecha_fin, hora_fin
// (si tus nombres difieren, ajusta el UPDATE y SELECT)

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS / CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Método no permitido. Usa POST.']); exit;
}

/* ===== Conexión ===== */
require __DIR__ . '/../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Error de conexión a BD']); exit;
}
if (function_exists('mysqli_report')) { mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); }
mysqli_set_charset($db, 'utf8mb4');

/* ===== Leer JSON (id_inventario opcional) ===== */
$raw  = file_get_contents('php://input');
$body = json_decode($raw, true) ?: [];
$id_inventario = isset($body['id_inventario']) ? (int)$body['id_inventario'] : 0;

try {
  $db->begin_transaction();

  // 1) Seleccionar el inventario a terminar y bloquearlo
  if ($id_inventario > 0) {
    // Específico
    $sqlInv = "
      SELECT id, estado, fecha_inicio, IFNULL(condicion,1) AS condicion
      FROM tb_inventario_vin
      WHERE id = ?
      FOR UPDATE
    ";
    $stInv = $db->prepare($sqlInv);
    $stInv->bind_param('i', $id_inventario);
    $stInv->execute();
    $inv = $stInv->get_result()->fetch_assoc();
    $stInv->close();
  } else {
    // Último EN EJECUCION
    $sqlInv = "
      SELECT id, estado, fecha_inicio, IFNULL(condicion,1) AS condicion
      FROM tb_inventario_vin
      WHERE condicion = 1 AND estado = 'EN EJECUCION'
      ORDER BY fecha_inicio DESC, id DESC
      LIMIT 1
      FOR UPDATE
    ";
    $rsInv = $db->query($sqlInv);
    $inv   = $rsInv->fetch_assoc();
  }

  if (!$inv) {
    $db->rollback();
    http_response_code(404);
    echo json_encode(['success'=>false,'message'=>"No se encontró inventario 'EN EJECUCION' o el ID no existe."]); exit;
  }

  if ((int)$inv['condicion'] !== 1) {
    $db->rollback();
    http_response_code(400);
    echo json_encode(['success'=>false,'message'=>'Inventario inactivo (condicion=0).']); exit;
  }

  if (strtoupper((string)$inv['estado']) !== 'EN EJECUCION') {
    $db->rollback();
    http_response_code(400);
    echo json_encode(['success'=>false,'message'=>"El inventario no está en 'EN EJECUCION' (estado actual: {$inv['estado']})."]); exit;
  }

  $id_inventario = (int)$inv['id'];

  // 2) Debe tener al menos 1 detalle activo
  $sqlCnt = "
    SELECT COUNT(*) AS c
    FROM tb_inventario_datadai
    WHERE id_inventario = ? AND IFNULL(condicion,1) = 1
    FOR UPDATE
  ";
  $stCnt = $db->prepare($sqlCnt);
  $stCnt->bind_param('i', $id_inventario);
  $stCnt->execute();
  $rowCnt = $stCnt->get_result()->fetch_assoc();
  $stCnt->close();

  $items_count = (int)($rowCnt['c'] ?? 0);
  if ($items_count <= 0) {
    $db->rollback();
    http_response_code(400);
    echo json_encode(['success'=>false,'message'=>'No puedes terminar: el inventario no tiene detalles activos.']); exit;
  }

  // 3) Terminar (estado, fecha_fin, hora_fin)
  $fecha_fin = date('Y-m-d');
  $hora_fin  = date('H:i:s');

  $sqlUp = "
    UPDATE tb_inventario_vin
    SET estado = 'TERMINADO',
        fecha_fin = ?,
        hora_fin  = ?
    WHERE id = ? AND condicion = 1
  ";
  $stUp = $db->prepare($sqlUp);
  $stUp->bind_param('ssi', $fecha_fin, $hora_fin, $id_inventario);
  $stUp->execute();
  $affected = $stUp->affected_rows;
  $stUp->close();

  if ($affected <= 0) {
    // Nada cambió (posible carrera o ya estaba terminado)
    $db->rollback();
    http_response_code(409);
    echo json_encode(['success'=>false,'message'=>'No se pudo terminar (posible estado cambiado por otra operación).']); exit;
  }

  $db->commit();

  echo json_encode([
    'success' => true,
    'message' => 'Inventario terminado correctamente',
    'data'    => [
      'id_inventario' => $id_inventario,
      'fecha_fin'     => $fecha_fin,
      'hora_fin'      => $hora_fin,
      'items'         => $items_count
    ]
  ], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
  if ($db->errno) { $db->rollback(); }
  http_response_code(500);
  echo json_encode([
    'success'=>false,
    'message'=>'Excepción',
    'error'=>$e->getMessage()
  ], JSON_UNESCAPED_UNICODE);
}
