<?php
// views/pages/api_app/datadai_set_estado6.php
// Busca un chasis en tb_datadai y cambia id_estado a 6 si existe.
// Body (POST JSON o form-data):
//   - chasis : string (obligatorio)
//   - _debug : '1'    (opcional)

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS / CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'stage'=>'method','message'=>'Método no permitido; use POST']); exit;
}

/* ===== Conexión BD ===== */
$pathConex = realpath(__DIR__.'/../conex.php') ?: realpath(__DIR__.'/../../conex.php') ?: realpath(__DIR__.'/conex.php');
if (!$pathConex) { http_response_code(500); echo json_encode(['success'=>false,'stage'=>'conn','message'=>'No se encontró conex.php']); exit; }
require_once $pathConex;

$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) { http_response_code(500); echo json_encode(['success'=>false,'stage'=>'conn','message'=>'Sin conexión a BD']); exit; }

if (function_exists('mysqli_report')) { mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); }
$db->set_charset('utf8mb4');
@$db->query("SET time_zone='-05:00'"); // Alinear NOW()/CURRENT_DATE con Guayaquil

/* ===== Helpers ===== */
function fail(int $code, string $msg, array $extra=[]): void {
  http_response_code($code);
  echo json_encode(['success'=>false,'stage'=>'validate','message'=>$msg,'data'=>$extra], JSON_UNESCAPED_UNICODE); exit;
}

/* ===== Input ===== */
$raw = file_get_contents('php://input') ?: '';
$in  = json_decode($raw, true);
if (!is_array($in)) { $in = $_POST; }

$chasis = isset($in['chasis']) ? trim((string)$in['chasis']) : '';
$debug  = (string)($in['_debug'] ?? '0') === '1';

if ($chasis === '') {
  fail(422, 'Falta chasis');
}

try {
  /* 1) Buscar chasis */
  $sqlFind = "SELECT iddatadai, id_estado FROM tb_datadai WHERE chasis = ? LIMIT 1";
  $st = $db->prepare($sqlFind);
  $st->bind_param('s', $chasis);
  $st->execute();
  $res = $st->get_result();
  $row = $res->fetch_assoc();
  $st->close();

  if (!$row) {
    http_response_code(404);
    echo json_encode([
      'success'=>false,
      'stage'=>'not_found',
      'message'=>'Chasis no encontrado',
      'chasis'=>$chasis
    ], JSON_UNESCAPED_UNICODE);
    exit;
  }

  $id_datadai     = (int)$row['iddatadai'];
  $estadoAnterior = isset($row['id_estado']) ? (int)$row['id_estado'] : null;

  /* 2) Actualizar a 6 solo si no lo está ya */
  $sqlUpd = "UPDATE tb_datadai
                SET id_estado = 6
              WHERE iddatadai = ?
                AND (id_estado IS NULL OR id_estado <> 6)";
  $st = $db->prepare($sqlUpd);
  $st->bind_param('i', $id_datadai);
  $st->execute();
  $cambiado = ($st->affected_rows > 0);
  $st->close();

  echo json_encode([
    'success'          => true,
    'message'          => $cambiado ? 'Estado actualizado a 6' : 'Sin cambios (ya estaba en 6)',
    'id_datadai'       => $id_datadai,
    'chasis'           => $chasis,
    'estado_anterior'  => $estadoAnterior,
    'estado_cambiado'  => $cambiado,
    'nuevo_estado'     => 6,
    '_debug'           => $debug ? ['raw'=>$raw] : null
  ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'success'=>false,
    'stage'=>'error',
    'message'=>'Error en procesamiento',
    'error'=>$e->getMessage()
  ], JSON_UNESCAPED_UNICODE);
}
