<?php
// chasis_ultima_pos_anden.php
// Devuelve la ÚLTIMA posición de andén por vehículo (1 fila por iddatadai)
// Criterio de "última": MAX(tb_datadai_anden.id) por id_datadai
//
// GET params:
//   - chasis=...        (LIKE; exact=1 para igualdad exacta)
//   - exact=0|1         (default 0)
//   - marca=..., modelo=..., color=...   (LIKE)
//   - id_ubicacion=int  (filtra andenes de una ubicación específica)
//   - estado_min=int    (default 6; exige d.id_estado >= estado_min)
//   - limit=1..1000     (default 200)
//   - offset>=0         (default 0)
//   - _debug=1          (muestra SQL/params)
//
// WHERE fijos: d.condicion=1 AND a.activo=1

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS / CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== Conexión ===== */
require __DIR__ . '/../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Error de conexión a BD']); exit;
}
if (function_exists('mysqli_report')) { mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); }
mysqli_set_charset($db, 'utf8mb4');

/* ===== Helpers ===== */
function qp(string $k, $d=null){ return $_GET[$k] ?? $d; }
function clamp_int($v, int $min, int $max, int $def): int {
  if (!is_numeric($v)) return $def; $v=(int)$v; return max($min, min($max, $v));
}
function like_escape(string $s): string {
  return strtr($s, ['\\'=>'\\\\','%'=>'\\%','_'=>'\\_']);
}

try {
  $debug       = (int)qp('_debug',0) === 1;
  $exact       = (int)qp('exact',0) === 1 ? 1 : 0;
  $limit       = clamp_int(qp('limit',200), 1, 1000, 200);
  $offset      = max(0, (int)qp('offset',0));
  $estado_min  = is_numeric(qp('estado_min', 6)) ? (int)qp('estado_min', 6) : 6;

  $chasis      = trim((string)qp('chasis',''));
  $marca       = trim((string)qp('marca',''));
  $modelo      = trim((string)qp('modelo',''));
  $color       = trim((string)qp('color',''));
  $id_ubicacion= qp('id_ubicacion', null);

  $where = [
    "d.condicion = 1",
    "a.activo = 1",
    "d.id_estado >= ?"
  ];
  $types = "i";
  $vals  = [$estado_min];

  if ($chasis !== "") {
    if ($exact) { $where[] = "d.chasis = ?"; $types .= "s"; $vals[] = $chasis; }
    else        { $where[] = "d.chasis LIKE ? ESCAPE '\\\\'"; $types .= "s"; $vals[] = '%'.like_escape($chasis).'%'; }
  }
  if ($marca !== "")  { $where[] = "d.marca  LIKE ? ESCAPE '\\\\'";  $types.="s"; $vals[]='%'.like_escape($marca).'%' ; }
  if ($modelo !== "") { $where[] = "d.modelo LIKE ? ESCAPE '\\\\'"; $types.="s"; $vals[]='%'.like_escape($modelo).'%'; }
  if ($color !== "")  { $where[] = "d.color  LIKE ? ESCAPE '\\\\'";  $types.="s"; $vals[]='%'.like_escape($color).'%' ; }
  if ($id_ubicacion !== null && $id_ubicacion !== '') {
    $where[] = "a.id_ubicacion = ?";
    $types  .= "i";
    $vals[]  = (int)$id_ubicacion;
  }

  $whereSql = 'WHERE '.implode(' AND ', $where);

  /* ===== Total (vehículos únicos con última posición) =====
     Usamos una subconsulta que toma el último registro por id_datadai y
     luego contamos los d.iddatadai que cumplen los filtros.
     Criterio de “último”: MAX(da.id)
     --- Si prefieres por fecha: reemplaza MAX(id) por MAX(fecha_i) y ajusta el JOIN por fecha. */
  $sqlCount = "
    SELECT COUNT(*) AS c
    FROM (
      SELECT d.iddatadai
      FROM tb_datadai d
      INNER JOIN (
        SELECT id_datadai, MAX(id) AS max_id
        FROM tb_datadai_anden
        GROUP BY id_datadai
      ) AS last_da ON last_da.id_datadai = d.iddatadai
      INNER JOIN tb_datadai_anden da ON da.id = last_da.max_id
      INNER JOIN tb_anden a ON a.id = da.id_anden
      $whereSql
      GROUP BY d.iddatadai
    ) AS x
  ";
  $stmtC = $db->prepare($sqlCount);
  $stmtC->bind_param($types, ...$vals);
  $stmtC->execute();
  $total = (int)($stmtC->get_result()->fetch_assoc()['c'] ?? 0);
  $stmtC->close();

  /* ===== Lista (última posición) ===== */
  $sql = "
    SELECT
      d.iddatadai,
      d.chasis,
      d.marca,
      d.modelo,
      d.color,
      a.id        AS id_anden,
      a.etiqueta  AS etiqueta,
      a.id_ubicacion,
      a.columna,
      a.fila
    FROM tb_datadai d
    INNER JOIN (
      SELECT id_datadai, MAX(id) AS max_id
      FROM tb_datadai_anden
      GROUP BY id_datadai
    ) AS last_da ON last_da.id_datadai = d.iddatadai
    INNER JOIN tb_datadai_anden da ON da.id = last_da.max_id
    INNER JOIN tb_anden a ON a.id = da.id_anden
    $whereSql
    GROUP BY
      d.iddatadai, d.chasis, d.marca, d.modelo, d.color,
      a.id, a.etiqueta, a.id_ubicacion, a.columna, a.fila
    ORDER BY a.etiqueta ASC, d.chasis ASC
    LIMIT ? OFFSET ?
  ";
  $stmt = $db->prepare($sql);
  $types2 = $types . "ii";
  $vals2  = $vals; $vals2[] = $limit; $vals2[] = $offset;
  $stmt->bind_param($types2, ...$vals2);
  $stmt->execute();
  $res = $stmt->get_result();

  $items = [];
  while ($r = $res->fetch_assoc()) {
    $items[] = [
      'iddatadai'   => (int)$r['iddatadai'],
      'chasis'      => (string)$r['chasis'],
      'marca'       => (string)$r['marca'],
      'modelo'      => (string)$r['modelo'],
      'color'       => (string)$r['color'],
      //'id_anden'    => (int)$r['id_anden'],
      'etiqueta'    => (string)$r['etiqueta'],
      //'id_ubicacion'=> isset($r['id_ubicacion']) ? (int)$r['id_ubicacion'] : null,
      //'columna'     => isset($r['columna']) ? (int)$r['columna'] : null,
      //'fila'        => isset($r['fila']) ? (int)$r['fila'] : null,
    ];
  }
  $stmt->close();

  $out = ['success'=>true,'total'=>$total,'items'=>$items];
  if ($debug) {
    $out['params'] = compact('chasis','exact','marca','modelo','color','id_ubicacion','estado_min','limit','offset');
    $out['sql'] = ['count'=>$sqlCount,'list'=>$sql,'types_count'=>$types,'types_list'=>$types2];
  }

  echo json_encode($out, JSON_UNESCAPED_UNICODE);
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'success'=>false,
    'message'=>'Excepción en la API',
    'error'=>$e->getMessage()
  ], JSON_UNESCAPED_UNICODE);
}
