<?php
// views/pages/api_app/chasis_bateria_list.php
// Lista chasis con última medición de batería y días desde la última.
// Filtro: ?chasis=... (LIKE o exacto con exact=1)
// Sin autorización — devuelve JSON con resultados.

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== CONEXIÓN BD ===== */
require_once __DIR__ . '/../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Sin conexión a base de datos']); exit;
}
if (function_exists('mysqli_report')) { mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); }
mysqli_set_charset($db, 'utf8mb4');
@$db->query("SET time_zone='-05:00'");

/* ===== INPUT GET ===== */
$chasis    = strtoupper(trim((string)($_GET['chasis'] ?? '')));
$exact     = (int)($_GET['exact'] ?? 0) === 1 ? 1 : 0;
//$limit     = (int)($_GET['limit'] ?? 50);
$offset    = (int)($_GET['offset'] ?? 0);
$diasMin   = (int)($_GET['dias_min'] ?? 15);
$estadoMin = (int)($_GET['estado_min'] ?? 6);
$dai       = trim((string)($_GET['dai'] ?? 'Simple'));

if ($limit <= 0)   $limit = 50;
if ($limit > 200)  $limit = 200;
if ($offset < 0)   $offset = 0;
if ($diasMin < 0)  $diasMin = 0;

$cutoffDate = date('Y-m-d', strtotime("-{$diasMin} days"));

/* ===== WHERE dinámico ===== */
$where = [];
$params = [];
$types  = '';

$where[] = "d.condicion = 1";
$where[] = "d.id_estado >= ?";
$types  .= 'i';
$params[] = $estadoMin;

$where[] = "d.dai = ?";
$types  .= 's';
$params[] = $dai;

$where[] = "(b.fecha IS NULL OR DATE(b.fecha) <= ?)";
$types  .= 's';
$params[] = $cutoffDate;

if ($chasis !== '') {
  if ($exact === 1) {
    $where[] = "UPPER(d.chasis) = ?";
    $types  .= 's';
    $params[] = $chasis;
  } else {
    $where[] = "UPPER(d.chasis) LIKE ?";
    $types  .= 's';
    $params[] = "%{$chasis}%";
  }
}

$whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

/* ===== FROM + JOIN ===== */
$fromJoin = "
  FROM tb_datadai AS d
  LEFT JOIN (
    SELECT bb.*
    FROM tb_bateria AS bb
    JOIN (
      SELECT id_datadai, MAX(fecha) AS max_fecha
      FROM tb_bateria
      GROUP BY id_datadai
    ) AS t
      ON t.id_datadai = bb.id_datadai
     AND t.max_fecha  = bb.fecha
  ) AS b
    ON b.id_datadai = d.iddatadai
";

/* ===== TOTAL ===== */
$sqlCount = "SELECT COUNT(*) AS total {$fromJoin} {$whereSql}";
$stC = $db->prepare($sqlCount);
if ($types !== '') { $stC->bind_param($types, ...$params); }
$stC->execute();
$total = (int)($stC->get_result()->fetch_assoc()['total'] ?? 0);
$stC->close();

/* ===== DATA ===== */
$sqlData = "
  SELECT
    d.marca,
    d.modelo,
    d.chasis,
    d.motor,
    d.color,
    b.fecha AS ultima_fecha,
    DATEDIFF(CURDATE(), DATE(b.fecha)) AS dias_desde_ultima
  {$fromJoin}
  {$whereSql}
  ORDER BY (b.fecha IS NULL) DESC, b.fecha ASC, d.chasis
  LIMIT ? OFFSET ?
";
$stD = $db->prepare($sqlData);

$typesData = $types . 'ii';
$paramsData = $params;
$paramsData[] = $limit;
$paramsData[] = $offset;
$stD->bind_param($typesData, ...$paramsData);

$stD->execute();
$res = $stD->get_result();

$items = [];
while ($row = $res->fetch_assoc()) {
  $items[] = [
    'marca'              => $row['marca'],
    'modelo'             => $row['modelo'],
    'chasis'             => $row['chasis'],
    'motor'              => $row['motor'],
    'color'              => $row['color'],
    'ultima_fecha'       => $row['ultima_fecha'] ? substr($row['ultima_fecha'], 0, 10) : null,
    'dias_desde_ultima'  => isset($row['dias_desde_ultima']) ? (int)$row['dias_desde_ultima'] : null,
  ];
}
$stD->close();

/* ===== RESPUESTA ===== */
echo json_encode([
  'success' => true,
  'total'   => $total,
  
  'items'   => $items
], JSON_UNESCAPED_UNICODE);
