<?php
// views/pages/api_app/bateria_insert.php
// Inserta en tb_bateria resolviendo id_datadai por CHASIS.
// Sin autenticación. Soporta JSON y form-data.
// Dinámico: arma columnas/placeholders/tipos para evitar desajustes.

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS / CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== BD ===== */
require_once __DIR__ . '/../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'Sin conexión a BD']); exit; }
if (function_exists('mysqli_report')) { mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); }
mysqli_set_charset($db, 'utf8mb4');
@$db->query("SET time_zone='-05:00'");

/* ===== INPUT ===== */
$raw = file_get_contents('php://input') ?: '';
$in  = json_decode($raw, true);
if (!is_array($in)) { $in = $_POST ?: []; }

$debug = isset($_GET['_debug']) ? 1 : 0;

$chasis = strtoupper(trim((string)($in['chasis'] ?? '')));
if ($chasis === '') { http_response_code(400); echo json_encode(['success'=>false,'message'=>'El parámetro "chasis" es requerido']); exit; }

/* Helpers */
$S = static function($k,$def=null) use($in){ return isset($in[$k]) && $in[$k] !== '' ? trim((string)$in[$k]) : $def; };
$I = static function($k,$def=null) use($in){ return isset($in[$k]) && $in[$k] !== '' ? (int)$in[$k] : $def; };

/* Defaults fecha/hora EC */
$fecha = $S('fecha', date('Y-m-d'));
$hora  = $S('hora',  date('H:i:s'));
if (strlen($hora) === 5) $hora .= ':00';

/* Resolver id_datadai por chasis */
try {
  $qD = $db->prepare("SELECT iddatadai FROM tb_datadai WHERE UPPER(chasis)=? LIMIT 1");
  $qD->bind_param('s', $chasis);
  $qD->execute();
  $rowD = $qD->get_result()->fetch_assoc();
  $qD->close();
  if (!$rowD) { http_response_code(404); echo json_encode(['success'=>false,'message'=>'Chasis no encontrado']); exit; }
  $id_datadai = (int)$rowD['iddatadai'];

  /* ===== Construcción dinámica del INSERT =====
     Campos admitidos (ajusta si tu tabla varía):
       id_datadai (int) *obligatorio, lo ponemos nosotros
       id_usuario (int)
       fecha (date)
       hora (time)
       estado (varchar)
       valor (varchar)
       id_unidad (int)
       resistencia (varchar)
       vida (varchar)
       foto (varchar)
       accion (varchar)
       observacion (varchar)
       tiempo (varchar)
       condicion (int, default 1)
  */
  $cols = [];
  $vals = [];
  $types = '';

  // Siempre id_datadai
  $cols[] = 'id_datadai';   $vals[] = $id_datadai; $types .= 'i';

  // id_usuario (int, opcional)
  $id_usuario = $I('id_usuario', 0); // usa 0 si no llega; ajusta si tu BD requiere NULL
  $cols[] = 'id_usuario';    $vals[] = $id_usuario; $types .= 'i';

  // fecha / hora (siempre ponemos)
  $cols[] = 'fecha';         $vals[] = $fecha;      $types .= 's';
  $cols[] = 'hora';          $vals[] = $hora;       $types .= 's';

  // varchar opcionales: normalizamos ''->NULL
  $varcharKeys = ['estado','valor','resistencia','vida','foto','accion','observacion','tiempo'];
  foreach ($varcharKeys as $k) {
    $v = $S($k, null);
    if ($v === '') $v = null;
    $cols[] = $k;            $vals[] = $v;          $types .= 's';
  }

  // id_unidad (int opcional)
  $id_unidad = $I('id_unidad', null);
  $cols[] = 'id_unidad';     $vals[] = $id_unidad;  $types .= 'i';

  // condicion (int, default 1)
  $condicion = $I('condicion', 1);
  $cols[] = 'condicion';     $vals[] = $condicion;  $types .= 'i';

  // Armar placeholders
  $placeholders = implode(',', array_fill(0, count($cols), '?'));
  $colnames     = implode(',', $cols);

  $sql = "INSERT INTO tb_bateria ($colnames) VALUES ($placeholders)";
  $st  = $db->prepare($sql);

  // bind_param con variables por referencia
  $bind = [$types];
  foreach ($vals as $i => $v) { $bind[] = &$vals[$i]; }
  call_user_func_array([$st, 'bind_param'], $bind);

  $st->execute();
  $idbateria = $st->insert_id;
  $st->close();

  $resp = [
    'success'   => true,
    'message'   => 'Batería registrada',
    'idbateria' => $idbateria,
    'data'      => [
      'id_datadai'  => $id_datadai,
      'chasis'      => $chasis,
      'fecha'       => $fecha,
      'hora'        => $hora,
      'estado'      => $vals[array_search('estado', $cols, true)],
      'valor'       => $vals[array_search('valor', $cols, true)],
      'id_unidad'   => $id_unidad,
      'resistencia' => $vals[array_search('resistencia', $cols, true)],
      'vida'        => $vals[array_search('vida', $cols, true)],
      'foto'        => $vals[array_search('foto', $cols, true)],
      'accion'      => $vals[array_search('accion', $cols, true)],
      'observacion' => $vals[array_search('observacion', $cols, true)],
      'tiempo'      => $vals[array_search('tiempo', $cols, true)],
      'condicion'   => $condicion
    ]
  ];
  if ($debug) { $resp['_debug'] = ['sql'=>$sql,'types'=>$types,'cols'=>$cols,'vals'=>$vals]; }
  echo json_encode($resp, JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'success'=>false,
    'message'=>'No se pudo guardar',
    'error'=>$e->getMessage()
  ]);
}
