<?php 
/**
 * asignar_datadai_link.php
 * Vincula (id_asignar, id_datadai, condicion=1) en tb_asignar_datadai,
 * evitando duplicar el mismo id_datadai con condicion=1.
 * Actualiza tb_datadai.id_estado = 6 para el chasis.
 *
 * Entrada: POST JSON / form-data / x-www-form-urlencoded
 * Parámetros: chasis (string), id_asignar (int)
 */

declare(strict_types=1);

/* ===== Encabezados / CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== Errores como JSON ===== */
ini_set('display_errors', '0');
mysqli_report(MYSQLI_REPORT_OFF);
set_error_handler(function($n,$s,$f,$l){
  http_response_code(400);
  echo json_encode(['success'=>false,'stage'=>'php_error','message'=>"PHP:$n $s @ $f:$l"]); exit;
});
set_exception_handler(function($e){
  http_response_code(400);
  echo json_encode(['success'=>false,'stage'=>'exception','message'=>$e->getMessage()]); exit;
});

/* ===== Conexión BD ===== */
$pathConex = realpath(__DIR__ . '/../conex.php') ?: realpath(__DIR__ . '/../../conex.php') ?: null;
if (!$pathConex) { http_response_code(500); echo json_encode(['success'=>false,'stage'=>'conn','message'=>'No se encontró conex.php']); exit; }
require_once $pathConex;

$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) { http_response_code(500); echo json_encode(['success'=>false,'stage'=>'conn','message'=>'Sin conexión BD']); exit; }
mysqli_set_charset($db, 'utf8mb4');
@$db->query("SET time_zone='-05:00'"); // America/Guayaquil

/* ===== Método ===== */
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'stage'=>'method','message'=>'Método no permitido; usa POST']); exit;
}

/* ===== Utils ===== */
function t(string $v): string { return trim($v); }

/* Cargar JSON si existe */
$RAW = file_get_contents('php://input');
$IN_JSON = json_decode($RAW ?: '[]', true);
if (!is_array($IN_JSON)) $IN_JSON = [];

/* param(): JSON -> POST -> GET */
function param($key, $default=null) {
  global $IN_JSON;
  if (array_key_exists($key, $IN_JSON)) return $IN_JSON[$key];
  if (isset($_POST[$key])) return $_POST[$key];
  if (isset($_GET[$key]))  return $_GET[$key];
  return $default;
}

/* ===== Params ===== */
$chasis     = t((string)param('chasis', ''));
$id_asignar = (int) param('id_asignar', 0);

if ($chasis === '') {
  echo json_encode(['success'=>false,'stage'=>'validate','message'=>'Falta chasis']); exit;
}
if ($id_asignar <= 0) {
  echo json_encode(['success'=>false,'stage'=>'validate','message'=>'Falta id_asignar válido']); exit;
}

/* ===== Buscar iddatadai por chasis ===== */
$sqlFind = "SELECT iddatadai, id_estado FROM tb_datadai WHERE chasis = ? LIMIT 1";
$stFind = $db->prepare($sqlFind);
if (!$stFind) { http_response_code(500); echo json_encode(['success'=>false,'stage'=>'prepare_find','message'=>$db->error]); exit; }
$stFind->bind_param('s', $chasis);
$stFind->execute();
$res = $stFind->get_result();
$row = $res ? $res->fetch_assoc() : null;
$stFind->close();

if (!$row) {
  http_response_code(404);
  echo json_encode(['success'=>false,'stage'=>'not_found','message'=>'Chasis no encontrado en tb_datadai','chasis'=>$chasis]); exit;
}
$id_datadai    = (int)$row['iddatadai'];
$estado_actual = isset($row['id_estado']) ? (int)$row['id_estado'] : null;

/* ===== Transacción ===== */
$db->begin_transaction();

try {
  /* 1) Lock de vínculos activos para este id_datadai */
  $sqlLock = "SELECT id_asignar FROM tb_asignar_datadai WHERE id_datadai = ? AND condicion = 1 LIMIT 1 FOR UPDATE";
  $stLock = $db->prepare($sqlLock);
  if (!$stLock) { throw new RuntimeException('prepare_lock: '.$db->error); }
  $stLock->bind_param('i', $id_datadai);
  $stLock->execute();
  $resLock = $stLock->get_result();
  $rowLock = $resLock ? $resLock->fetch_assoc() : null;
  $stLock->close();

  if ($rowLock) {
    $id_asignar_existente = (int)$rowLock['id_asignar'];
    if ($id_asignar_existente !== $id_asignar) {
      /* Ya existe un vínculo activo con OTRO id_asignar -> RECHAZAR */
      $db->rollback();
      http_response_code(409); // Conflict
      echo json_encode([
        'success'  => false,
        'stage'    => 'duplicate_active',
        'message'  => 'id_datadai ya está vinculado con condicion=1 a otro id_asignar.',
        'id_datadai' => $id_datadai,
        'id_asignar_actual' => $id_asignar_existente,
        'id_asignar_intentado' => $id_asignar
      ], JSON_UNESCAPED_UNICODE);
      exit;
    }
    /* Si es el mismo id_asignar, consideramos que el vínculo ya existe y no insertamos */
    $already_active_same = true;
  } else {
    $already_active_same = false;
  }

  /* 2) Verificar si el par exacto ya existe (cualquier condicion) */
  $sqlPair = "SELECT condicion FROM tb_asignar_datadai WHERE id_asignar = ? AND id_datadai = ? LIMIT 1";
  $stPair = $db->prepare($sqlPair);
  if (!$stPair) { throw new RuntimeException('prepare_pair: '.$db->error); }
  $stPair->bind_param('ii', $id_asignar, $id_datadai);
  $stPair->execute();
  $resPair = $stPair->get_result();
  $rowPair = $resPair ? $resPair->fetch_assoc() : null;
  $stPair->close();

  $created_link_id = null;
  $created_link    = false;
  $cond_anterior   = $rowPair ? (int)$rowPair['condicion'] : null;

  if ($already_active_same) {
    /* Ya había condicion=1 con el mismo id_asignar: no insertamos */
    $created_link = false;
  } elseif ($rowPair) {
    /* Ya existe el par con otra condicion (p.ej., 0): lo actualizamos a 1 */
    $sqlUpdCond = "UPDATE tb_asignar_datadai SET condicion = 1 WHERE id_asignar = ? AND id_datadai = ? AND condicion <> 1";
    $stUpdCond = $db->prepare($sqlUpdCond);
    if (!$stUpdCond) { throw new RuntimeException('prepare_update_cond: '.$db->error); }
    $stUpdCond->bind_param('ii', $id_asignar, $id_datadai);
    if (!$stUpdCond->execute()) { throw new RuntimeException('execute_update_cond: '.$stUpdCond->error); }
    $created_link = ($stUpdCond->affected_rows > 0);
    $stUpdCond->close();
  } else {
    /* No existe el par: insertamos condicion=1 */
    $sqlIns = "INSERT INTO tb_asignar_datadai (id_asignar, id_datadai, condicion) VALUES (?, ?, 1)";
    $stIns = $db->prepare($sqlIns);
    if (!$stIns) { throw new RuntimeException('prepare_insert: '.$db->error); }
    $stIns->bind_param('ii', $id_asignar, $id_datadai);
    if (!$stIns->execute()) { throw new RuntimeException('execute_insert: '.$stIns->error); }
    $created_link     = true;
    $created_link_id  = $db->insert_id ?: null;
    $stIns->close();
  }

  /* 3) Actualizar estado a 6 solo si no lo está ya */
  $sqlUpd = "UPDATE tb_datadai SET id_estado = 6 WHERE iddatadai = ? AND (id_estado IS NULL OR id_estado <> 6)";
  $stUpd = $db->prepare($sqlUpd);
  if (!$stUpd) { throw new RuntimeException('prepare_update_estado: '.$db->error); }
  $stUpd->bind_param('i', $id_datadai);
  if (!$stUpd->execute()) { throw new RuntimeException('execute_update_estado: '.$stUpd->error); }
  $estado_cambiado = ($stUpd->affected_rows > 0);
  $stUpd->close();

  $db->commit();

  http_response_code(201);
  echo json_encode([
    'success'         => true,
    'message'         => $already_active_same
                          ? 'Vínculo ya estaba activo (condicion=1) para este id_asignar.'
                          : ($created_link ? 'Vínculo activado/creado y estado actualizado cuando aplicó.' : 'Sin cambios (ya estaba vinculado).'),
    'id_asignar'      => $id_asignar,
    'id_datadai'      => $id_datadai,
    'estado_anterior' => $estado_actual,
    'estado_cambiado' => $estado_cambiado,
    'nuevo_estado'    => 6,
    'link'            => [
      'ya_activo_mismo_asignar' => $already_active_same,
      'creado_o_activado'       => $created_link,
      'created_id'              => $created_link_id,
      'condicion_anterior_par'  => $cond_anterior
    ]
  ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

} catch (Throwable $e) {
  $db->rollback();
  http_response_code(500);
  echo json_encode(['success'=>false,'stage'=>'transaction','message'=>$e->getMessage()]);
}
