<?php
// views/pages/api_app/anden_free.php
// SOLO libera el andén (sin crear asignaciones)
//
// Flujo:
// 1) Cierra asignación ABIERTA en tb_datadai_anden (condicion=1 y fecha_f IS NULL)
//    usando fecha/hora local (America/Guayaquil) calculada en PHP:
//      - fecha_f = $nowEc (string 'Y-m-d H:i:s')
//      - dia     = diferencia en días (DateTime PHP, TZ=America/Guayaquil)
// 2) Pone tb_anden.activo = 0
//
// Parámetros (GET o POST JSON/form-data):
//   id_anden : int (requerido)
//   chasis   : string (opcional; filtra la asignación por ese vehículo)
//   _debug   : 1 (opcional; devuelve info de diagnóstico de TZ)
//
// NOTA: Este endpoint evita depender de NOW()/TIMESTAMPDIFF de MySQL para
// fechas — todo se hace en PHP con TZ Ecuador para eliminar desfases.

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS / CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== CONEXIÓN ===== */
require_once __DIR__ . '/../conex.php';
$mysqli = $conn ?? ($conex ?? null);
if (!$mysqli instanceof mysqli) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'Error de conexión a BD']); exit; }
if (function_exists('mysqli_report')) { mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); }
mysqli_set_charset($mysqli, 'utf8mb4');

/* ===== INPUT ===== */
function inData(): array {
  $ct = $_SERVER['CONTENT_TYPE'] ?? '';
  if ($_SERVER['REQUEST_METHOD'] === 'POST' && stripos((string)$ct, 'application/json') !== false) {
    $raw = file_get_contents('php://input');
    if ($raw !== false && strlen(trim($raw)) > 0) {
      $data = json_decode($raw, true);
      if (json_last_error() === JSON_ERROR_NONE && is_array($data)) return $data;
    }
  }
  if (!empty($_POST)) return $_POST;
  return $_GET;
}
$in = inData();
$id_anden = isset($in['id_anden']) ? (int)$in['id_anden'] : 0;
$chasis   = isset($in['chasis']) ? trim((string)$in['chasis']) : '';
$wantDbg  = isset($in['_debug']) ? (int)$in['_debug'] === 1 : false;

if ($id_anden <= 0) {
  http_response_code(400);
  echo json_encode(['success'=>false,'message'=>'Parámetro requerido: id_anden > 0']); exit;
}

/* ===== Helpers ===== */
function getVehiculoByChasis(mysqli $db, string $chasis): ?array {
  $sql = "SELECT iddatadai, chasis
          FROM tb_datadai
          WHERE UPPER(TRIM(chasis)) = UPPER(TRIM(?))
          ORDER BY iddatadai DESC
          LIMIT 1";
  $st = $db->prepare($sql);
  $st->bind_param('s', $chasis);
  $st->execute();
  $rs = $st->get_result();
  $row = $rs->fetch_assoc();
  $st->close();
  return $row ?: null;
}

/* ===== TZ y reloj locales (Ecuador) ===== */
$tzEc  = new DateTimeZone('America/Guayaquil');
$nowEcDT = new DateTime('now', $tzEc);
$nowEc = $nowEcDT->format('Y-m-d H:i:s');

try {
  $mysqli->begin_transaction();

  /* 1) Lock del andén */
  $sqlAnden = "SELECT id, id_ubicacion, etiqueta, activo, condicion
               FROM tb_anden
               WHERE id = ?
               FOR UPDATE";
  $st = $mysqli->prepare($sqlAnden);
  $st->bind_param('i', $id_anden);
  $st->execute();
  $rs = $st->get_result();
  $anden = $rs->fetch_assoc();
  $st->close();
  if (!$anden) throw new RuntimeException('El andén especificado no existe.');

  /* 2) Resolver id_datadai opcional */
  $id_datadai = null;
  if ($chasis !== '') {
    $veh = getVehiculoByChasis($mysqli, $chasis);
    if (!$veh) throw new RuntimeException('No se encontró id_datadai para el chasis indicado.');
    $id_datadai = (int)$veh['iddatadai'];
  }

  /* 3) Buscar asignación ABIERTA (condicion=1, fecha_f IS NULL) */
  if ($id_datadai !== null) {
    $sqlAsig = "SELECT id, id_datadai, id_anden, fecha_i, fecha_f, dia, condicion
                FROM tb_datadai_anden
                WHERE id_anden = ? AND id_datadai = ? AND condicion = 1 AND fecha_f IS NULL
                ORDER BY id DESC
                LIMIT 1";
    $st = $mysqli->prepare($sqlAsig);
    $st->bind_param('ii', $id_anden, $id_datadai);
  } else {
    $sqlAsig = "SELECT id, id_datadai, id_anden, fecha_i, fecha_f, dia, condicion
                FROM tb_datadai_anden
                WHERE id_anden = ? AND condicion = 1 AND fecha_f IS NULL
                ORDER BY id DESC
                LIMIT 1";
    $st = $mysqli->prepare($sqlAsig);
    $st->bind_param('i', $id_anden);
  }
  $st->execute();
  $rs = $st->get_result();
  $asig = $rs->fetch_assoc();
  $st->close();

  /* 4) Si hay asignación abierta, cerrar usando PHP (TZ Ecuador) */
  $finalAsig = null;
  if ($asig) {
    // Convertir fecha_i de la BD a DateTime Ecuador para cálculo exacto de días
    // Asumimos que fecha_i viene en el formato 'Y-m-d H:i:s' y representa lo guardado en BD.
    // Si tus columnas son TIMESTAMP y MySQL las devuelve ya ajustadas a una TZ distinta,
    // puedes forzar interpretación como UTC y convertir a Ecuador. Descomenta dos líneas abajo si hace falta:
    // $fi = DateTime::createFromFormat('Y-m-d H:i:s', $asig['fecha_i'], new DateTimeZone('UTC'));
    // $fi->setTimezone($tzEc);

    // Interpretación directa como hora local Ecuador:
    $fi = DateTime::createFromFormat('Y-m-d H:i:s', $asig['fecha_i'], $tzEc);
    if (!$fi) {
      // Fallback: intentar sin TZ y luego asignar Ecuador
      $fi = DateTime::createFromFormat('Y-m-d H:i:s', $asig['fecha_i']);
      if ($fi) $fi->setTimezone($tzEc);
    }

    // Diferencia en días enteros (como TIMESTAMPDIFF(DAY, fecha_i, fecha_f))
    $diffDays = null;
    if ($fi) {
      // Normalizamos copias para asegurar que ambos están en la misma TZ
      $fiEc  = (clone $fi)->setTimezone($tzEc);
      $nowEcCopy = (clone $nowEcDT);
      $interval = $fiEc->diff($nowEcCopy);
      $diffDays = (int)$interval->days; // días completos
      // Si prefieres cualquier fracción como día completo: $diffDays = (int)ceil($interval->h > 0 || $interval->i > 0 || $interval->s > 0 ? $interval->days + 0.0001 : $interval->days);
    }

    // Actualizar (sin NOW): usamos $nowEc y $diffDays calculados en PHP
    $sqlClose = "UPDATE tb_datadai_anden
                 SET fecha_f = ?,
                     dia     = ?
                 WHERE id = ?";
    $st = $mysqli->prepare($sqlClose);
    // Si no pudimos calcular diffDays por algún motivo, enviamos NULL
    if ($diffDays === null) {
      $null = null;
      $st->bind_param('sii', $nowEc, $null, $asig['id']);
    } else {
      $st->bind_param('sii', $nowEc, $diffDays, $asig['id']);
    }
    $st->execute();
    $st->close();

    // Traer registro final
    $sqlGet = "SELECT id, id_datadai, id_anden, fecha_i, fecha_f, dia, condicion
               FROM tb_datadai_anden
               WHERE id = ?";
    $st = $mysqli->prepare($sqlGet);
    $st->bind_param('i', $asig['id']);
    $st->execute();
    $rs = $st->get_result();
    $finalAsig = $rs->fetch_assoc();
    $st->close();
  }

  /* 5) Marcar el andén como libre */
  if ((int)$anden['activo'] !== 0) {
    $sqlFree = "UPDATE tb_anden SET activo = 0 WHERE id = ?";
    $st = $mysqli->prepare($sqlFree);
    $st->bind_param('i', $id_anden);
    $st->execute();
    $st->close();
    $nuevoActivo = 0;
  } else {
    $nuevoActivo = 0;
  }

  $resp = [
    'success' => true,
    'anden' => [
      'id'        => (int)$anden['id'],
      'etiqueta'  => $anden['etiqueta'] ?? null,
      'activo'    => $nuevoActivo,
      'condicion' => isset($anden['condicion']) ? (int)$anden['condicion'] : null
    ],
    'asignacion' => $finalAsig ? [
      'id'         => (int)$finalAsig['id'],
      'id_datadai' => (int)$finalAsig['id_datadai'],
      'id_anden'   => (int)$finalAsig['id_anden'],
      'fecha_i'    => $finalAsig['fecha_i'],
      'fecha_f'    => $finalAsig['fecha_f'], // guardada desde PHP con TZ Ecuador
      'dia'        => isset($finalAsig['dia']) ? (int)$finalAsig['dia'] : null,
      'condicion'  => isset($finalAsig['condicion']) ? (int)$finalAsig['condicion'] : null
    ] : null,
    'nota' => $finalAsig
      ? 'Andén liberado y asignación cerrada (hora local de Ecuador).'
      : 'Andén liberado. No había asignación abierta (condicion=1) para cerrar.'
  ];

  /* ===== Debug opcional ===== */
  if ($wantDbg) {
    // Muestra cómo ve MySQL y cómo ve PHP la hora
    $dbg = ['php_now_ec' => $nowEc];
    try {
      $q = $mysqli->query("SELECT NOW() AS mysql_now, @@session.time_zone AS session_tz, @@system_time_zone AS system_tz");
      if ($q) {
        $row = $q->fetch_assoc();
        $dbg['mysql_now'] = $row['mysql_now'] ?? null;
        $dbg['mysql_session_tz'] = $row['session_tz'] ?? null;
        $dbg['mysql_system_tz'] = $row['system_tz'] ?? null;
        $q->close();
      }
    } catch (Throwable $e) { $dbg['mysql_debug_error'] = $e->getMessage(); }
    $resp['_debug'] = $dbg;
  }

  $mysqli->commit();
  echo json_encode($resp);

} catch (Throwable $e) {
  try { $mysqli->rollback(); } catch (\Throwable $e2) {}
  http_response_code(400);
  echo json_encode(['success'=>false,'message'=>$e->getMessage()]);
}
