<?php
/**
 * asignar_datadai_link.php
 * Crea vínculo (id_asignar, id_datadai, condicion=1) en tb_asignar_datadai
 * y actualiza tb_datadai.id_estado = 6 para el chasis encontrado.
 *
 * Entrada: JSON (body), POST (form) o GET (query string)
 * Parámetros: chasis (string), id_asignar (int)
 * Seguridad: SIN token y SIN sesión (protege por IP/Firewall en producción)
 */

declare(strict_types=1);

// ===== Encabezados / CORS =====
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

// ===== Errores como JSON =====
ini_set('display_errors', '0');
mysqli_report(MYSQLI_REPORT_OFF);
set_error_handler(function($n,$s,$f,$l){
  http_response_code(400);
  echo json_encode(['success'=>false,'stage'=>'php_error','message'=>"PHP:$n $s @ $f:$l"]); exit;
});
set_exception_handler(function($e){
  http_response_code(400);
  echo json_encode(['success'=>false,'stage'=>'exception','message'=>$e->getMessage()]); exit;
});

// ===== Conexión BD =====
$pathConex = realpath(__DIR__ . '/../conex.php') ?: realpath(__DIR__ . '/../../conex.php') ?: 'conex.php';
if (!$pathConex) { http_response_code(500); echo json_encode(['success'=>false,'stage'=>'conn','message'=>'No se encontró conex.php']); exit; }
require_once $pathConex;

$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) { http_response_code(500); echo json_encode(['success'=>false,'stage'=>'conn','message'=>'Sin conexión BD']); exit; }
mysqli_set_charset($db, 'utf8mb4');
@$db->query("SET time_zone='-05:00'"); // America/Guayaquil

// ===== Método =====
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'stage'=>'method','message'=>'Método no permitido; usa POST']); exit;
}

// ===== Utils =====
function t(string $v): string { return trim($v); }

// Cargar JSON si existe
$RAW = file_get_contents('php://input');
$IN_JSON = json_decode($RAW ?: '[]', true);
if (!is_array($IN_JSON)) $IN_JSON = [];

// param(): JSON -> POST -> GET
function param($key, $default=null) {
  global $IN_JSON;
  if (array_key_exists($key, $IN_JSON)) return $IN_JSON[$key];
  if (isset($_POST[$key])) return $_POST[$key];
  if (isset($_GET[$key]))  return $_GET[$key];
  return $default;
}

// ===== Params =====
$chasis     = t((string)param('chasis', ''));
$id_asignar = (int) param('id_asignar', 0);

if ($chasis === '') {
  echo json_encode(['success'=>false,'stage'=>'validate','message'=>'Falta chasis']); exit;
}
if ($id_asignar <= 0) {
  echo json_encode(['success'=>false,'stage'=>'validate','message'=>'Falta id_asignar válido']); exit;
}

// ===== Buscar iddatadai por chasis =====
$sqlFind = "SELECT iddatadai, id_estado FROM tb_datadai WHERE chasis = ? LIMIT 1";
$stFind = $db->prepare($sqlFind);
if (!$stFind) { http_response_code(500); echo json_encode(['success'=>false,'stage'=>'prepare_find','message'=>$db->error]); exit; }
$stFind->bind_param('s', $chasis);
$stFind->execute();
$res = $stFind->get_result();
$row = $res ? $res->fetch_assoc() : null;
$stFind->close();

if (!$row) {
  http_response_code(404);
  echo json_encode(['success'=>false,'stage'=>'not_found','message'=>'Chasis no encontrado en tb_datadai','chasis'=>$chasis]); exit;
}
$id_datadai    = (int)$row['iddatadai'];
$estado_actual = isset($row['id_estado']) ? (int)$row['id_estado'] : null;

// ===== Transacción =====
$db->begin_transaction();

try {
  // Evitar duplicados del vínculo
  $sqlExists = "SELECT 1 FROM tb_asignar_datadai WHERE id_asignar = ? AND id_datadai = ? LIMIT 1";
  $stEx = $db->prepare($sqlExists);
  if (!$stEx) { throw new RuntimeException('prepare_exists: '.$db->error); }
  $stEx->bind_param('ii', $id_asignar, $id_datadai);
  $stEx->execute();
  $stEx->store_result();
  $already = $stEx->num_rows > 0;
  $stEx->close();

  $created_link_id = null;
  $created_link    = false;

  if (!$already) {
    // Insertar vínculo con condicion = 1
    $sqlIns = "INSERT INTO tb_asignar_datadai (id_asignar, id_datadai, condicion) VALUES (?, ?, 1)";
    $stIns = $db->prepare($sqlIns);
    if (!$stIns) { throw new RuntimeException('prepare_insert: '.$db->error); }
    $stIns->bind_param('ii', $id_asignar, $id_datadai);
    if (!$stIns->execute()) { throw new RuntimeException('execute_insert: '.$stIns->error); }
    $created_link     = true;
    $created_link_id  = $db->insert_id ?: null; // puede ser null si no hay autoincrement
    $stIns->close();
  }

  // Actualizar estado a 6 solo si no lo está ya
  $sqlUpd = "UPDATE tb_datadai SET id_estado = 6 WHERE iddatadai = ? AND (id_estado IS NULL OR id_estado <> 6)";
  $stUpd = $db->prepare($sqlUpd);
  if (!$stUpd) { throw new RuntimeException('prepare_update: '.$db->error); }
  $stUpd->bind_param('i', $id_datadai);
  if (!$stUpd->execute()) { throw new RuntimeException('execute_update: '.$stUpd->error); }
  $estado_cambiado = ($stUpd->affected_rows > 0);
  $stUpd->close();

  $db->commit();

  http_response_code(201);
  echo json_encode([
    'success'        => true,
    'message'        => 'Vínculo creado/confirmado y estado de datadai actualizado cuando aplicó.',
    'id_asignar'     => $id_asignar,
    'id_datadai'     => $id_datadai,
    'estado_anterior'=> $estado_actual,
    'estado_cambiado'=> $estado_cambiado,
    'nuevo_estado'   => 6,
    'link'           => [
      'ya_existia'   => $already,
      'creado'       => $created_link,
      'created_id'   => $created_link_id
    ]
  ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

} catch (Throwable $e) {
  $db->rollback();
  http_response_code(500);
  echo json_encode(['success'=>false,'stage'=>'transaction','message'=>$e->getMessage()]);
}
