<?php
/**
 * asignar_create.php — API para CREAR (INSERT) registros en tb_asignar
 * (SIN TOKEN, SIN SESIÓN, SIN links/agrup_movimiento/fecha_carga)
 *
 * Entrada: JSON (body), POST (form) o GET (query string).
 *
 * Obligatorios:
 *  - tipo_transporte, placa, nombre (conductor), fecha_descarga (YYYY-MM-DD [HH:MM[:SS]])
 *
 * Opcionales:
 *  - cedula, guias, compania (-> transportista), destino, observaciones | observacion
 *
 * Lote:
 *  - ids: [int,...] | id_gestion: "1,2,3"  (si vienen, inserta uno por cada ID; si no, inserta uno solo)
 *
 * Inserta:
 *  INSERT INTO tb_asignar
 *    (transportista, destino, status, numero_doc, conductor, placa, fecha_descarga, tipo, guia, observaciones, condicion)
 *  VALUES
 *    (?,            ?,      'COORDINADO',   ?,         ?,        ?,        ?,             ?,    ?,    ?,            1)
 */

declare(strict_types=1);

// ===== Encabezados / CORS =====
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

// ===== Errores como JSON =====
ini_set('display_errors', '0');
mysqli_report(MYSQLI_REPORT_OFF);
set_error_handler(function($n,$s,$f,$l){
  http_response_code(400);
  echo json_encode(['success'=>false,'stage'=>'php_error','message'=>"PHP:$n $s @ $f:$l"]); exit;
});
set_exception_handler(function($e){
  http_response_code(400);
  echo json_encode(['success'=>false,'stage'=>'exception','message'=>$e->getMessage()]); exit;
});

// ===== Conexión BD =====
$pathConex = realpath(__DIR__ . '/../conex.php') ?: realpath(__DIR__ . '/../../conex.php') ?: 'conex.php';
if (!$pathConex) { http_response_code(500); echo json_encode(['success'=>false,'stage'=>'conn','message'=>'No se encontró conex.php']); exit; }
require_once $pathConex;

$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) { http_response_code(500); echo json_encode(['success'=>false,'stage'=>'conn','message'=>'Sin conexión BD']); exit; }
mysqli_set_charset($db, 'utf8mb4');
@$db->query("SET time_zone='-05:00'"); // America/Guayaquil

// ===== Método =====
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'stage'=>'method','message'=>'Método no permitido; usa POST']); exit;
}

// ===== Utilidades =====
function str_t(string $v): string { return trim($v); }
function normalizeDate(?string $v): ?string {
  $v = is_null($v) ? null : trim($v);
  if ($v === null || $v === '') return null;
  if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $v)) $v .= ' 00:00:00';
  if (preg_match('/^\d{4}-\d{2}-\d{2}\s+\d{2}:\d{2}$/', $v)) $v .= ':00';
  $dt = DateTime::createFromFormat('Y-m-d H:i:s', $v);
  return $dt ? $dt->format('Y-m-d H:i:s') : null;
}
function ints_from_csv_or_array($src): array {
  if (is_array($src)) return array_values(array_filter(array_map('intval', $src), fn($i)=>$i>0));
  if (is_string($src) || is_numeric($src)) {
    $txt = (string)$src;
    if (strpos($txt, ',') !== false) return array_values(array_filter(array_map('intval', explode(',', $txt)), fn($i)=>$i>0));
    $val = (int)$txt; return $val>0 ? [$val] : [];
  }
  return [];
}

// ===== Leer JSON (si hay) =====
$RAW = file_get_contents('php://input');
$IN_JSON = json_decode($RAW ?: '[]', true);
if (!is_array($IN_JSON)) $IN_JSON = [];

// ===== param() toma de JSON -> POST -> GET =====
function param($key, $default=null) {
  global $IN_JSON;
  if (array_key_exists($key, $IN_JSON)) return $IN_JSON[$key];
  if (isset($_POST[$key])) return $_POST[$key];
  if (isset($_GET[$key]))  return $_GET[$key];
  return $default;
}

// ===== Lote (opcional) =====
$ids_lote = [];
$ids_lote = array_merge($ids_lote, ints_from_csv_or_array(param('ids', [])));
$ids_lote = array_merge($ids_lote, ints_from_csv_or_array(param('id_gestion', '')));
$ids_lote = array_values(array_unique($ids_lote));

// ===== Campos =====
$tipo  = str_t((string)param('tipo_transporte',''));
$placa = str_t((string)param('placa',''));
$cedu  = str_t((string)param('cedula',''));
$nom   = str_t((string)param('nombre',''));
$fd    = normalizeDate((string)param('fecha_descarga',''));
$guia  = str_t((string)param('guias',''));

$compania = str_t((string)param('compania',''));  // va a columna transportista
$destino  = str_t((string)param('destino',''));
$obs      = param('observaciones', null);
if ($obs === null) $obs = param('observacion', '');
$obs = str_t((string)$obs);

// ===== Validaciones mínimas =====
if ($tipo === '')  { echo json_encode(['success'=>false,'stage'=>'validate','message'=>'Falta tipo_transporte']); exit; }
if ($placa === '') { echo json_encode(['success'=>false,'stage'=>'validate','message'=>'Falta placa']); exit; }
if ($nom === '')   { echo json_encode(['success'=>false,'stage'=>'validate','message'=>'Falta nombre (conductor)']); exit; }
if (!$fd)          { echo json_encode(['success'=>false,'stage'=>'validate','message'=>'fecha_descarga inválida o ausente']); exit; }

// ===== SQL (INSERT) =====
// Asegúrate de que tb_asignar tenga estas columnas y que id sea AUTO_INCREMENT.
$sql = "INSERT INTO tb_asignar
          (transportista, destino, status, numero_doc, conductor,  placa, fecha_descarga, tipo, guia, observaciones, condicion)
        VALUES
          (            ?,       ?,  'COORDINADO',        ?,         ?,       ?,             ?,    ?,    ?,             ?,          1)";

$stmt = $db->prepare($sql);
if (!$stmt) { http_response_code(500); echo json_encode(['success'=>false,'stage'=>'prepare','message'=>$db->error]); exit; }

// ===== Transacción e inserts =====
$db->begin_transaction();
$inserted_ids = [];
$failed       = [];

// Si no vienen IDs de lote, insertamos uno solo.
$vueltas = !empty($ids_lote) ? $ids_lote : [null];

try {
  foreach ($vueltas as $dummy) {
    // Orden coincide con los placeholders del INSERT:
    // transportista, destino, numero_doc, conductor, placa, fecha_descarga, tipo, guia, observaciones
    if (!$stmt->bind_param('sssssssss', $compania, $destino, $cedu, $nom, $placa, $fd, $tipo, $guia, $obs)) {
      throw new RuntimeException('bind_param falló');
    }
    if (!$stmt->execute()) {
      $failed[] = ['error'=>$stmt->error];
      continue;
    }
    $inserted_ids[] = $stmt->insert_id;
  }

  if (!empty($failed)) {
    $db->rollback();
    http_response_code(500);
    echo json_encode([
      'success' => false,
      'stage'   => 'execute',
      'message' => 'Fallo al insertar uno o más registros',
      'failed'  => $failed,
      'inserted_ids_partial' => $inserted_ids
    ]); exit;
  }

  $db->commit();

  // Si es un solo insert, devolvemos también "id" directo para comodidad
  $resp = [
    'success'      => true,
    'message'      => 'Registros creados correctamente.',
    'inserted'     => count($inserted_ids),
    'inserted_ids' => $inserted_ids
  ];
  if (count($inserted_ids) === 1) {
    $resp['id'] = $inserted_ids[0];
  }

  http_response_code(201);
  echo json_encode($resp, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

} catch (Throwable $e) {
  $db->rollback();
  http_response_code(500);
  echo json_encode(['success'=>false,'stage'=>'transaction','message'=>$e->getMessage()]);
} finally {
  $stmt->close();
}
