<?php
// incidentes_trabajo_taller_novedades.php — Vehículo (único) + Incidentes + Tipos/Talleres (si existen)
// - MUESTRA los chasis aunque el incidente NO tenga taller/tipo asignado.
// Métodos: GET, OPTIONS
//
// Parámetros GET opcionales:
//   - dai=Simple         (por defecto 'Simple')
//   - chasis             (string; si se envía, filtra por chasis; exact=1 usa igualdad)
//   - exact=0|1          (default 0)
//   - incident_id        (int; filtra por un incidente específico)
//   - limit=200          (1..1000)  -> se aplica sobre la lista base de incidentes; el resultado final está agrupado por vehículo
//   - offset=0           (>=0)
//
// Respuesta (agrupada por vehículo; sin repetir chasis/marca/modelo/color):
// {
//   "success": true,
//   "total": 1,                    // vehículos
//   "total_incidentes": 2,         // suma de incidentes en todos los vehículos devueltos
//   "items": [
//     {
//       "chasis": "8AJKB3CD3T1717945",
//       "marca": "Toyota",
//       "modelo": "Hilux",
//       "color": "Blanco",
//       "incidentes": [
//         {
//           "incident_id": 123,
//           "novedad": { ... },
//           "tipos_asignados": [               // <- puede venir VACÍO si no hay taller/tipo asignado
//             { "id_tipo": 7, "tipo": "Enderezada", "id_taller": 5, "taller": "Carrocerías XYZ" }
//           ]
//         }
//       ]
//     }
//   ]
// }

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS / CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== Conexión ===== */
$pathConex = realpath(__DIR__ . '/../conex.php')
         ?: realpath(__DIR__ . '/../../conex.php')
         ?: realpath(__DIR__ . '/conex.php');
if (!$pathConex) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'No se encontró conex.php']); exit;
}
require_once $pathConex;

$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Sin conexión a la base de datos.']); exit;
}
if (function_exists('mysqli_report')) { mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); }
mysqli_set_charset($db, 'utf8mb4');

/* ===== Parámetros ===== */
$GET        = $_GET + [];
$dai        = isset($GET['dai']) ? trim((string)$GET['dai']) : 'Simple';
$chasis     = isset($GET['chasis']) ? trim((string)$GET['chasis']) : '';
$exact      = isset($GET['exact']) ? (int)$GET['exact'] : 0;
$incidentId = isset($GET['incident_id']) ? (int)$GET['incident_id'] : 0;
$limit      = isset($GET['limit']) ? max(1, min(1000, (int)$GET['limit'])) : 200;
$offset     = isset($GET['offset']) ? max(0, (int)$GET['offset']) : 0;

/* ===== WHERE base (incidentes activos; SIN exigir tipos) ===== */
$where  = [];
$params = [];
$types  = '';

$where[] = 'd.condicion = 1';
$where[] = 'd.dai = ?'; $types .= 's'; $params[] = $dai;
$where[] = 'i.condicion = 1';

if ($chasis !== '') {
  if ($exact) { $where[] = 'd.chasis = ?'; $types .= 's'; $params[] = $chasis; }
  else        { $where[] = 'd.chasis LIKE ?'; $types .= 's'; $params[] = "%$chasis%"; }
}
if ($incidentId > 0) {
  $where[] = 'i.idincidente = ?'; $types .= 'i'; $params[] = $incidentId;
}

$whereSql = 'WHERE '.implode(' AND ', $where);

/* ===== Query de INCIDENTES (sin tipos) =====
   - Trae datos de vehículo (marca, modelo, color) + novedades del incidente
   - Se pagina aquí (por incidente), luego se agrupa por vehículo en PHP.
*/
$sqlInc = "
  SELECT
    d.iddatadai,
    d.chasis,
    d.marca,
    d.modelo,
    d.color,
    i.idincidente  AS incident_id,
    z.detallezona,
    p.detalleparte,
    n.detallenovedad,
    i.observacion,
    i.foto,
    n.detallenovedad        AS severidad,
    m.medida                AS nombre_medida_catalogo,
    i.medida                AS medida_tomada
  FROM tb_incidente i
  JOIN tb_datadai d ON d.iddatadai = i.id_datadai
  LEFT JOIN tb_zona    z ON z.idzona    = i.id_zona
  LEFT JOIN tb_parte   p ON p.idparte   = i.id_parte
  LEFT JOIN tb_novedad n ON n.idnovedad = i.id_novedad
  LEFT JOIN tb_medida  m ON m.id        = i.id_medida
  $whereSql
  ORDER BY d.chasis ASC, i.idincidente ASC
  LIMIT ? OFFSET ?
";

/* ===== Query de TIPOS (por los incidentes obtenidos) =====
   - NO se exige que existan (si no hay, simplemente no habrá filas para ese incidente).
*/
$sqlTipos = "
  SELECT
    it.incident_id,
    tr.id         AS id_tipo,
    tr.tipo,
    tr.id_taller,
    COALESCE(t.taller,'') AS taller
  FROM tb_incidente_tipo it
  LEFT JOIN tb_tipo_reparacion tr ON tr.id = it.id_tipo
  LEFT JOIN tb_taller t           ON t.idtaller = tr.id_taller
  WHERE it.incident_id IN (/*PLACEHOLDER*/)
  ORDER BY t.taller ASC, tr.tipo ASC
";

try {
  /* 1) Obtener incidentes base (paginados) */
  $stmt = mysqli_prepare($db, $sqlInc);
  $types1 = $types.'ii';
  $params1 = $params;
  $params1[] = $limit;
  $params1[] = $offset;
  mysqli_stmt_bind_param($stmt, $types1, ...$params1);
  mysqli_stmt_execute($stmt);
  $res = mysqli_stmt_get_result($stmt);

  $incidentes = [];
  $idsInc = [];
  while ($r = $res->fetch_assoc()) {
    $incidentes[] = $r;
    $idsInc[] = (int)$r['incident_id'];
  }
  mysqli_stmt_close($stmt);

  /* 2) Si no hay incidentes, respuesta vacía */
  if (empty($incidentes)) {
    echo json_encode(['success'=>true, 'total'=>0, 'total_incidentes'=>0, 'items'=>[]], JSON_UNESCAPED_UNICODE);
    exit;
  }

  /* 3) Obtener tipos/taller por esos incidentes (si hay) */
  $tiposByInc = [];
  if (!empty($idsInc)) {
    $ph = implode(',', array_fill(0, count($idsInc), '?'));
    $sqlT = str_replace('/*PLACEHOLDER*/', $ph, $sqlTipos);
    $stmtT = mysqli_prepare($db, $sqlT);
    $typesT = str_repeat('i', count($idsInc));
    mysqli_stmt_bind_param($stmtT, $typesT, ...$idsInc);
    mysqli_stmt_execute($stmtT);
    $resT = mysqli_stmt_get_result($stmtT);
    while ($rowT = $resT->fetch_assoc()) {
      $iid = (int)$rowT['incident_id'];
      if (!isset($tiposByInc[$iid])) $tiposByInc[$iid] = [];
      $tiposByInc[$iid][] = [
        'id_tipo'   => isset($rowT['id_tipo']) ? (int)$rowT['id_tipo'] : null,
        'tipo'      => $rowT['tipo'],
        'id_taller' => isset($rowT['id_taller']) ? (int)$rowT['id_taller'] : null,
        'taller'    => $rowT['taller'],
      ];
    }
    mysqli_stmt_close($stmtT);
  }

  /* 4) Agrupar por vehículo (iddatadai) para NO repetir chasis/marca/modelo/color */
  $vehiculos = [];         // iddatadai => { chasis, marca, modelo, color, incidentes: [] }
  $vehIndex  = [];         // iddatadai => index en $items
  $totalInc  = 0;

  foreach ($incidentes as $row) {
    $idd  = (int)$row['iddatadai'];
    if (!isset($vehIndex[$idd])) {
      $vehIndex[$idd] = count($vehiculos);
      $vehiculos[] = [
        'chasis'  => $row['chasis'],
        'marca'   => $row['marca'],
        'modelo'  => $row['modelo'],
        'color'   => $row['color'],
        'incidentes' => []
      ];
    }

    $inc = [
      'incident_id' => (int)$row['incident_id'],
      'novedad' => [
        'detallezona'            => $row['detallezona'],
        'detalleparte'           => $row['detalleparte'],
        'detallenovedad'         => $row['detallenovedad'],
        'observacion'            => $row['observacion'],
        'foto'                   => $row['foto'],
        'severidad'              => $row['severidad'],
        'nombre_medida_catalogo' => $row['nombre_medida_catalogo'],
        'medida_tomada'          => $row['medida_tomada'],
      ],
      // Si no hay tipos para este incidente, quedará como [] (no forzamos nulls)
      'tipos_asignados' => $tiposByInc[(int)$row['incident_id']] ?? []
    ];

    $vehiculos[$vehIndex[$idd]]['incidentes'][] = $inc;
    $totalInc++;
  }

  /* 5) Total de vehículos (agrupados) y salida */
  $totalVehiculos = count($vehiculos);

  echo json_encode([
    'success'           => true,
    'total'             => $totalVehiculos,  // vehículos
    'total_incidentes'  => $totalInc,        // incidentes en los vehículos devueltos
    'items'             => array_values($vehiculos)
  ], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'success'=>false,
    'message'=>'Error interno.',
    'error'=>$e->getMessage()
  ], JSON_UNESCAPED_UNICODE);
}
