<?php
declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/**
 * validar_tag_directo.php
 * Valida par (idusuario, tag_code) contra tb_usuario (sin depender de sesión).
 *
 * Entrada (POST JSON o form-data):
 *   - idusuario   (int)
 *   - tag_code    (string)
 *
 * Respuesta JSON:
 *   { success: bool, message: string, data?: { idusuario:int, tag_code:string, activo:bool, match:bool } }
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Access-Control-Allow-Methods: POST, OPTIONS');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

// 1) Leer entrada (JSON o form)
$input = json_decode(file_get_contents('php://input'), true);
if (!is_array($input)) $input = $_POST;

$idusuario = isset($input['idusuario']) ? (int)$input['idusuario'] : 0;
$tag_code  = trim((string)($input['tag_code'] ?? $input['tag'] ?? ''));

if ($idusuario <= 0 || $tag_code === '') {
  http_response_code(422);
  echo json_encode([
    'success' => false,
    'message' => 'Faltan parámetros: idusuario y/o tag_code.'
  ]);
  exit;
}

// Normalizar tag (opcional: mayúsculas, sin espacios)
$tag_code = strtoupper($tag_code);

// 2) Conexión BD
require_once __DIR__ . '/../conex.php'; // ajusta ruta si es necesario
if (!($conn instanceof mysqli)) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Error de conexión a BD.']);
  exit;
}
mysqli_set_charset($conn, 'utf8mb4');

// 3) Consultar usuario y validar
try {
  $sql = "SELECT idusuario, activo, tag_codigo
          FROM tb_usuario
          WHERE idusuario = ?
          LIMIT 1";
  $stmt = $conn->prepare($sql);
  if (!$stmt) throw new Exception('Prepare failed: '.$conn->error);
  $stmt->bind_param('i', $idusuario);
  $stmt->execute();
  $rs = $stmt->get_result();
  $user = $rs->fetch_assoc();
  $stmt->close();

  if (!$user) {
    http_response_code(404);
    echo json_encode([
      'success'=>false,
      'message'=>'Usuario no encontrado.',
      'data'=>['idusuario'=>$idusuario]
    ]);
    exit;
  }

  $isActive   = (int)$user['activo'] === 1;
  $dbTag      = (string)($user['tag_codigo'] ?? '');
  $dbTagUpper = strtoupper(trim($dbTag));
  $match      = ($dbTagUpper !== '' && $dbTagUpper === $tag_code);

  if (!$isActive) {
    http_response_code(403);
    echo json_encode([
      'success'=>false,
      'message'=>'El usuario está inactivo.',
      'data'=>[
        'idusuario' => (int)$user['idusuario'],
        'tag_code'  => $tag_code,
        'activo'    => false,
        'match'     => $match
      ]
    ]);
    exit;
  }

  if (!$match) {
    http_response_code(403);
    echo json_encode([
      'success'=>false,
      'message'=>'El tag no coincide con el usuario.',
      'data'=>[
        'idusuario' => (int)$user['idusuario'],
        'tag_code'  => $tag_code,
        'activo'    => true,
        'match'     => false
      ]
    ]);
    exit;
  }

  // OK
  echo json_encode([
    'success'=>true,
    'message'=>'Validación correcta: usuario activo y tag coincide.',
    'data'=>[
      'idusuario' => (int)$user['idusuario'],
      'tag_code'  => $tag_code,
      'activo'    => true,
      'match'     => true
    ]
  ]);
  exit;

} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Error interno: '.$e->getMessage()]);
  exit;
}
