<?php
// views/pages/api_app/usuario_info.php
// Devuelve datos de usuario por idusuario, filtrando campos por permiso.
// Con fallback si columnas/tablas de bodega/empresa no existen.
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== Conexión ===== */
$pathConex = realpath(__DIR__ . '/../conex.php') ?: realpath(__DIR__ . '/../../conex.php') ?: realpath(__DIR__ . '/conex.php');
if (!$pathConex) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'No se encontró conex.php']); exit; }
require_once $pathConex;

$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'Conexión BD no disponible']); exit; }
mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone = '-05:00'");

/* ===== Entrada ===== */
$raw = file_get_contents('php://input');
$in  = json_decode($raw, true) ?: [];
$idusuario = 0;
if (isset($in['idusuario']))    $idusuario = (int)$in['idusuario'];
if (isset($_POST['idusuario'])) $idusuario = (int)$_POST['idusuario'];
if (isset($_GET['idusuario']))  $idusuario = (int)$_GET['idusuario'];
$debug = isset($_GET['debug']) && $_GET['debug'] == '1';

if ($idusuario <= 0) {
  http_response_code(400);
  echo json_encode(['success'=>false,'message'=>'Parámetro idusuario requerido (>0)']); exit;
}

/* ===== SQL principal con JOINs (esquema real provisto) =====
   Tablas/columnas:
   tb_usuario: idusuario, usuario, permiso, id_registro, id_bodega, id_empresa, condicion, activo, tag_codigo
   tb_bodega:  idbodega, detalle
   tb_empresa: idempresa, razon_social
*/
$sqlJoin = "SELECT u.idusuario, u.usuario, u.permiso, u.id_registro, u.id_bodega, u.id_empresa,
                   u.condicion, u.activo, u.tag_codigo,
                   b.detalle AS bodega, e.razon_social AS empresa
            FROM tb_usuario AS u
            LEFT JOIN tb_bodega  AS b ON b.idbodega  = u.id_bodega
            LEFT JOIN tb_empresa AS e ON e.idempresa = u.id_empresa
            WHERE u.idusuario = ?
            LIMIT 1";

/* ===== Fallback sin JOINs (sólo devuelve IDs) ===== */
$sqlNoJoin = "SELECT u.idusuario, u.usuario, u.permiso, u.id_registro, u.id_bodega, u.id_empresa,
                     u.condicion, u.activo, u.tag_codigo
              FROM tb_usuario AS u
              WHERE u.idusuario = ?
              LIMIT 1";

/* ===== Ejecutar con intento JOIN y fallback ===== */
function runOne(mysqli $db, string $sql, int $id): array {
  $stmt = mysqli_prepare($db, $sql);
  if (!$stmt) return ['ok'=>false, 'err'=>mysqli_error($db), 'sql'=>$sql, 'row'=>null];
  mysqli_stmt_bind_param($stmt, 'i', $id);
  if (!mysqli_stmt_execute($stmt)) return ['ok'=>false, 'err'=>mysqli_error($db), 'sql'=>$sql, 'row'=>null];
  $res = mysqli_stmt_get_result($stmt);
  $row = $res ? mysqli_fetch_assoc($res) : null;
  return ['ok'=>true, 'err'=>null, 'sql'=>$sql, 'row'=>$row];
}

$r = runOne($db, $sqlJoin, $idusuario);
$usedSql = $sqlJoin;

if (!$r['ok']) {
  // Si falló (por columnas/tablas), usamos fallback sin JOIN
  $r = runOne($db, $sqlNoJoin, $idusuario);
  $usedSql = $sqlNoJoin;
}

if (!$r['ok']) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Error de consulta','error'=>$r['err'], 'debug_sql'=>$debug ? $r['sql'] : null]);
  exit;
}

$user = $r['row'];
if (!$user) { http_response_code(404); echo json_encode(['success'=>false,'message'=>'Usuario no encontrado']); exit; }

/* ===== Esquemas por permiso (incluye siempre id_bodega, id_empresa, condicion, activo) ===== */
$schemas = [
  1 => ['idusuario','usuario','permiso','id_registro','id_bodega','bodega','id_empresa','empresa','condicion','activo','tag_codigo'],
  2 => ['idusuario','usuario','permiso','id_bodega','bodega','id_empresa','empresa','condicion','activo','tag_codigo'],
  5 => ['idusuario','usuario','permiso','id_bodega','bodega','id_empresa','condicion','activo','tag_codigo'],
];

// Si el permiso no está mapeado, al menos devuelve estos (incluye ids, condicion y activo)
$permiso = (int)($user['permiso'] ?? 0);
$allowed = $schemas[$permiso] ?? ['idusuario','usuario','permiso','id_bodega','id_empresa','condicion','activo','tag_codigo'];

/* Si venimos del fallback sin JOIN, no habrá bodega/empresa de texto; las removemos del whitelist */
if (!array_key_exists('bodega', $user))  $allowed = array_values(array_filter($allowed, fn($f) => $f !== 'bodega'));
if (!array_key_exists('empresa', $user)) $allowed = array_values(array_filter($allowed, fn($f) => $f !== 'empresa'));

/* ===== (Opcional) Bloquear usuarios inactivos =====
   Descomenta para responder 403 si el usuario está inactivo o deshabilitado:
*/
// if ((int)$user['condicion'] !== 1) { http_response_code(403); echo json_encode(['success'=>false,'message'=>'Usuario inactivo (condicion=0)']); exit; }
// if ((int)$user['activo']    !== 1) { http_response_code(403); echo json_encode(['success'=>false,'message'=>'Usuario deshabilitado (activo=0)']); exit; }

/* ===== Armar salida ===== */
$data = [];
foreach ($allowed as $f) {
  if (array_key_exists($f, $user)) $data[$f] = $user[$f];
}

echo json_encode([
  'success' => true,
  'message' => 'OK',
  'data'    => $data
  
  
], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
