<?php
// tipos_reparacion_select.php — Lista para <select> de tipos de reparación (+ taller)
// Métodos: GET, OPTIONS
//
// Parámetros GET opcionales:
//   - id_taller (int)     -> filtra por taller
//   - q                  -> búsqueda en tipo/taller (LIKE)
//   - exact=0|1          -> 1 = igualdad exacta en q; default 0 (LIKE)
//   - grouped=0|1        -> 1 = agrupa por taller (para <optgroup>)
//   - _debug=1           -> info de depuración
//
// Respuesta (grouped=0 por defecto):
// { "success": true, "total": N, "items": [ { "id":1,"tipo":"...","id_taller":3,"taller":"...","value":1,"label":"..." }, ... ] }
//
// Respuesta (grouped=1):
// { "success": true, "total": N, "groups": [ { "label":"Taller X", "options":[{"value":1,"label":"Tipo A","id_taller":3}, ...] }, ... ] }

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS / CORS / NO-CACHE ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== Conexión ===== */
require_once __DIR__ . '/../conex.php'; // ajusta si tu ruta es distinta
$db = $conn ?? ($conex ?? null);
if (!$db) { echo json_encode(['success'=>false,'message'=>'Sin conexión a la base de datos.']); exit; }
if (function_exists('mysqli_report')) {
  mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
}
mysqli_set_charset($db, 'utf8mb4');

/* ===== Parámetros ===== */
$GET       = $_GET + [];
$idTaller  = isset($GET['id_taller']) ? (int)$GET['id_taller'] : 0;
$q         = isset($GET['q']) ? trim((string)$GET['q']) : '';
$exact     = isset($GET['exact']) ? (int)$GET['exact'] : 0;
$grouped   = isset($GET['grouped']) ? (int)$GET['grouped'] : 0;

/* ===== WHERE ===== */
$where   = ['tr.condicion = 1'];
$ptypes  = '';
$params  = [];

if ($idTaller > 0) {
  $where[] = 'tr.id_taller = ?';
  $ptypes .= 'i';
  $params[] = $idTaller;
}
if ($q !== '') {
  if ($exact) {
    $where[] = '(tr.tipo = ? OR t.taller = ?)';
    $ptypes .= 'ss';
    $params[] = $q;
    $params[] = $q;
  } else {
    $where[] = '(tr.tipo LIKE ? OR t.taller LIKE ?)';
    $like = "%$q%";
    $ptypes .= 'ss';
    $params[] = $like;
    $params[] = $like;
  }
}

$where_sql = $where ? 'WHERE '.implode(' AND ', $where) : '';

/* ===== SQL ===== */
$sql = "
  SELECT
    tr.id,
    tr.tipo,
    tr.id_taller,
    t.taller
  FROM tb_tipo_reparacion tr
  LEFT JOIN tb_taller t ON t.idtaller = tr.id_taller
  $where_sql
  ORDER BY t.taller ASC, tr.tipo ASC
";

try {
  $stmt = mysqli_prepare($db, $sql);
  if ($ptypes !== '') {
    mysqli_stmt_bind_param($stmt, $ptypes, ...$params);
  }
  mysqli_stmt_execute($stmt);
  $res = mysqli_stmt_get_result($stmt);

  $items = [];
  while ($r = $res->fetch_assoc()) {
    $items[] = [
      'id'         => (int)$r['id'],
      'tipo'       => $r['tipo'],
      'id_taller'  => isset($r['id_taller']) ? (int)$r['id_taller'] : null,
      'taller'     => $r['taller'],
      // conveniencia para <select>
      'value'      => (int)$r['id'],
      'label'      => $r['tipo'],
    ];
  }
  mysqli_stmt_close($stmt);

  $out = ['success'=>true, 'total'=>count($items)];

  if ($grouped) {
    // Agrupar por taller para <optgroup>
    $groups = [];
    foreach ($items as $it) {
      $labelTaller = $it['taller'] ?? 'Sin taller';
      if (!isset($groups[$labelTaller])) $groups[$labelTaller] = [];
      $groups[$labelTaller][] = [
        'value'     => $it['value'],
        'label'     => $it['label'],
        'id_taller' => $it['id_taller'],
      ];
    }
    $out['groups'] = [];
    foreach ($groups as $label => $options) {
      $out['groups'][] = ['label'=>$label, 'options'=>$options];
    }
  } else {
    $out['items'] = $items;
  }

  if (isset($GET['_debug'])) {
    $out['_debug'] = ['where'=>$where_sql];
  }

  echo json_encode($out, JSON_UNESCAPED_UNICODE);
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'success'=>false,
    'message'=>'Error interno.',
    'error'=>$e->getMessage()
  ], JSON_UNESCAPED_UNICODE);
}
