<?php
// views/pages/api_app/tarja_get.php
// API (GET): Devuelve TARJA + INCIDENTES (según tu esquema de tb_incidente).
// Parámetros (GET):
//  - id_tarja        (opcional) -> se mapea a la PK real de tb_tarja
//  - numerotarja     (opcional)
//  - chasis          (opcional) -> busca en tb_datadai.chasis (exacto o LIKE)
//  - exact=1         (opcional) -> para chasis exacto
//  - id_datadai      (opcional) -> última tarja de ese id_datadai
//  - _debug=1        (opcional) -> añade trace

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS / CORS / Anti-cache ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

$DEBUG = isset($_GET['_debug']) && $_GET['_debug'] == '1';

/* ===== Error handler JSON (evita 500) ===== */
set_exception_handler(function(Throwable $e) use ($DEBUG) {
  http_response_code(400);
  $out = ['success'=>false,'message'=>'Error: '.$e->getMessage()];
  if ($DEBUG) $out['trace'] = $e->getTraceAsString();
  echo json_encode($out, JSON_UNESCAPED_UNICODE);
  exit;
});

/* ===== Conexión ===== */
$pathConex = realpath(__DIR__.'/../conex.php') ?: realpath(__DIR__.'/conex.php') ?: realpath(__DIR__.'/../../conex.php');
if (!$pathConex) throw new RuntimeException('No se encontró conex.php');
require_once $pathConex;

$db = null;
if (isset($conn) && $conn instanceof mysqli) $db = $conn;
elseif (isset($db) && $db instanceof mysqli) { /* ok */ }
else $db = $db ?? ($conn ?? null);
if (!$db instanceof mysqli) throw new RuntimeException('Sin conexión a la base de datos');

if (function_exists('mysqli_report')) mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
mysqli_set_charset($db, 'utf8mb4');

/* ===== Helpers ===== */
function qp(string $k, $d=null){ return $_GET[$k] ?? $d; }
function toIntNullable($v): ?int { return ($v === null || $v === '') ? null : (int)$v; }
function is_truthy($v): bool { if ($v === null) return false; $s=strtolower((string)$v); return in_array($s,['1','true','si','sí','yes','y','on'],true); }

/** Columnas de una tabla vía INFORMATION_SCHEMA (mejor en hosting compartido). */
function table_columns(mysqli $db, string $table): array {
  $sql = "SELECT COLUMN_NAME FROM INFORMATION_SCHEMA.COLUMNS
          WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?
          ORDER BY ORDINAL_POSITION";
  $st = $db->prepare($sql);
  $st->bind_param('s', $table);
  $st->execute();
  $res = $st->get_result();
  $cols = [];
  while ($r = $res->fetch_assoc()) $cols[] = $r['COLUMN_NAME'];
  $st->close();
  if (!$cols) throw new RuntimeException("Tabla no existe o sin columnas: {$table}");
  return $cols;
}

/** Detecta PK de tb_tarja (por si no es id_tarja). */
function detect_pk_tarja(mysqli $db): string {
  $cols = table_columns($db, 'tb_tarja');
  $candidatos = ['id_tarja','idtarja','idTarja','id','pk','idt'];
  foreach ($candidatos as $c) if (in_array($c,$cols,true)) return $c;
  foreach ($cols as $c) { if (preg_match('/^id(_|$)/i',$c)) return $c; }
  return $cols[0];
}

/** Incidentes según tu esquema EXACTO:
 * tb_incidente (
 *   idincidente, id_zona, id_parte, id_novedad, id_tarja, id_datadai,
 *   observacion, id_medida, medida, foto, reponsabilidad, validar, condicion, grave
 * )
 */
function fetch_incidentes_exact(mysqli $db, int $idTarja, int $idDatadai): array {
  // Preferimos vincular por id_tarja; si no hay incidentes así, probamos por id_datadai (por si tu app los guardó así)
  $rows = [];

  // 1) Por id_tarja
  $sql1 = "SELECT
            idincidente AS id_incidente,
            id_zona, id_parte, id_novedad,
            id_tarja, id_datadai,
            observacion, id_medida, medida, foto,
            reponsabilidad AS responsabilidad,
            validar, condicion, grave
          FROM tb_incidente
          WHERE id_tarja = ?
          ORDER BY idincidente ASC";
  $st1 = $db->prepare($sql1);
  $st1->bind_param('i', $idTarja);
  $st1->execute();
  $r1 = $st1->get_result();
  while ($row = $r1->fetch_assoc()) $rows[] = $row;
  $st1->close();

  if (!empty($rows)) return $rows;

  // 2) Fallback: por id_datadai (solo si no hubo por id_tarja)
  if ($idDatadai > 0) {
    $sql2 = "SELECT
              idincidente AS id_incidente,
              id_zona, id_parte, id_novedad,
              id_tarja, id_datadai,
              observacion, id_medida, medida, foto,
              reponsabilidad AS responsabilidad,
              validar, condicion, grave
            FROM tb_incidente
            WHERE id_datadai = ?
            ORDER BY idincidente ASC";
    $st2 = $db->prepare($sql2);
    $st2->bind_param('i', $idDatadai);
    $st2->execute();
    $r2 = $st2->get_result();
    $rows = [];
    while ($row = $r2->fetch_assoc()) $rows[] = $row;
    $st2->close();
  }

  return $rows;
}

/* ===== Inputs ===== */
$id_in        = toIntNullable(qp('id_tarja'));  // nombre público; internamente es la PK real
$numerotarja  = ($t = trim((string)qp('numerotarja',''))) !== '' ? $t : null;
$chasis       = ($c = trim((string)qp('chasis',''))) !== '' ? $c : null;
$exact        = is_truthy(qp('exact'));
$id_datadai   = toIntNullable(qp('id_datadai'));

/* ===== Lógica principal ===== */
$pkTarja = detect_pk_tarja($db);

// Selección de tarja por prioridad
if ($id_in) {
  $criteria = "pk:{$pkTarja}";
  $sql = "SELECT * FROM tb_tarja WHERE {$pkTarja} = ? LIMIT 1";
  $st = $db->prepare($sql);
  $st->bind_param('i', $id_in);
} elseif ($numerotarja) {
  $criteria = 'numerotarja';
  $sql = "SELECT * FROM tb_tarja WHERE numerotarja = ? LIMIT 1";
  $st = $db->prepare($sql);
  $st->bind_param('s', $numerotarja);
} elseif ($chasis) {
  $criteria = 'chasis'.($exact?':exact':':like');
  if ($exact) {
    $sql = "SELECT t.*
            FROM tb_tarja t
            INNER JOIN tb_datadai d ON d.iddatadai = t.id_datadai
            WHERE d.chasis = ?
            ORDER BY t.{$pkTarja} DESC
            LIMIT 1";
    $st = $db->prepare($sql);
    $st->bind_param('s', $chasis);
  } else {
    $like = '%'.$chasis.'%';
    $sql = "SELECT t.*
            FROM tb_tarja t
            INNER JOIN tb_datadai d ON d.iddatadai = t.id_datadai
            WHERE d.chasis LIKE ?
            ORDER BY t.{$pkTarja} DESC
            LIMIT 1";
    $st = $db->prepare($sql);
    $st->bind_param('s', $like);
  }
} elseif ($id_datadai) {
  $criteria = 'id_datadai:last';
  $sql = "SELECT * FROM tb_tarja WHERE id_datadai = ? ORDER BY {$pkTarja} DESC LIMIT 1";
  $st = $db->prepare($sql);
  $st->bind_param('i', $id_datadai);
} else {
  throw new InvalidArgumentException('Debe enviar id_tarja (id), numerotarja, chasis o id_datadai');
}

$st->execute();
$res = $st->get_result();
$tarja = $res->fetch_assoc();
$st->close();

if (!$tarja) {
  echo json_encode(['success'=>false,'message'=>'No se encontró la tarja solicitada'], JSON_UNESCAPED_UNICODE);
  exit;
}

$idTarja   = (int)$tarja[$pkTarja];
$idDatadai = (int)($tarja['id_datadai'] ?? 0);

// Chasis para meta
$chasisEncontrado = null;
if ($idDatadai > 0) {
  $stc = $db->prepare("SELECT chasis FROM tb_datadai WHERE iddatadai = ? LIMIT 1");
  $stc->bind_param('i', $idDatadai);
  $stc->execute();
  $rc = $stc->get_result()->fetch_assoc();
  $stc->close();
  $chasisEncontrado = $rc['chasis'] ?? null;
}

// INCIDENTES (exacto a tu esquema)
$incidentes = fetch_incidentes_exact($db, $idTarja, $idDatadai);

// Respuesta
echo json_encode([
  'success' => true,
  'data' => [
    'tarja' => $tarja,
    'incidentes' => $incidentes,
    'meta' => [
      'criterio'    => $criteria,
      'pk_tarja'    => $pkTarja,
      'id_tarja'    => $idTarja,
      'id_datadai'  => $idDatadai,
      'chasis'      => $chasisEncontrado
    ]
  ]
], JSON_UNESCAPED_UNICODE);
