<?php
// lavada_pendiente.php
// API: Lista de carros pendientes por lavar (JOIN tb_lavada, tb_datadai, tb_tipolavada)
// Por defecto filtra "pendiente" como t.lavada = 0
//
// Métodos: GET, OPTIONS
// Parámetros opcionales:
//   - pendiente = 1 | all   (default 1 -> solo pendientes; 'all' -> sin filtro de pendiente)
//   - lavada = 0|1          (si lo envías, fuerza t.lavada = ese valor)
//   - chasis, marca, modelo, color (filtros; exact=1 para igualdad, exact=0 -> LIKE)
//   - q                      (busca en chasis o motor vía LIKE)
//   - fecha_desde=YYYY-MM-DD (filtra l.fecha_registro >= fecha_desde)
//   - fecha_hasta=YYYY-MM-DD (filtra l.fecha_registro < fecha_hasta + 1 día)
//   - cond = 1 | all         (default 1 -> exige d.condicion=1, l.condicion=1 si existen)
//   - estado_min (int)       (opcional, p.ej. 6 -> d.id_estado >= 6)
//   - order_by: fecha_registro, chasis, marca, modelo, color, lavada (default fecha_registro)
//   - order_dir: ASC | DESC  (default DESC)
//   - exact = 0|1            (default 0)
//   - trim  = 0|1            (default 0; aplica TRIM en comparaciones exactas)
//   - _debug = 1             (devuelve SQL, params, etc.)

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') { http_response_code(204); exit; }

/* ===== Conexión ===== */
require __DIR__ . '/../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'Error de conexión a BD']); exit; }
if (function_exists('mysqli_report')) mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
mysqli_set_charset($db, 'utf8mb4');

/* ===== Helpers ===== */
function qp(string $k, $d=null){ return $_GET[$k] ?? $d; }
function likeWrap(?string $s): ?string { return $s === null ? null : ('%'.$s.'%'); }
function nextDay(string $ymd): string { $dt = DateTime::createFromFormat('Y-m-d', $ymd) ?: new DateTime($ymd); $dt->modify('+1 day'); return $dt->format('Y-m-d'); }

try {
  $DEBUG = ['meta'=>['time'=>date('Y-m-d H:i:s'), 'tz'=>date_default_timezone_get()]];

  $pendiente = strtolower((string)qp('pendiente', '1')); // '1' | 'all'
  $lavada    = qp('lavada'); // 0|1 (opcional)
  $condCtl   = strtolower((string)qp('cond', '1')); // '1' | 'all'
  $estadoMin = qp('estado_min', null);
  $estadoMin = ($estadoMin === null || $estadoMin === '') ? null : (int)$estadoMin;

  $q       = qp('q');
  $chasis  = qp('chasis');
  $marca   = qp('marca');
  $modelo  = qp('modelo');
  $color   = qp('color');
  $exact   = (int)qp('exact', 0) === 1;
  $trimCmp = (int)qp('trim', 0) === 1;

  $fDesde  = qp('fecha_desde'); // YYYY-MM-DD
  $fHasta  = qp('fecha_hasta'); // YYYY-MM-DD

  $orderByWhitelist = [
    'fecha_registro' => 'l.fecha_registro',
    'chasis'         => 'd.chasis',
    'marca'          => 'd.marca',
    'modelo'         => 'd.modelo',
    'color'          => 'd.color',
    'lavada'         => 't.lavada'
  ];
  $orderByKey  = strtolower((string)qp('order_by', 'fecha_registro'));
  $orderBySql  = $orderByWhitelist[$orderByKey] ?? 'l.fecha_registro';
  $orderDir    = strtoupper((string)qp('order_dir', 'DESC'));
  $orderDirSql = ($orderDir === 'ASC') ? 'ASC' : 'DESC';

  /* ===== WHERE dinámico ===== */
  $where = [];
  $params = [];
  $types  = '';

  // JOIN base:
  // tb_lavada l
  // tb_datadai d ON d.iddatadai = l.id_datadai
  // tb_tipolavada t ON t.idtipo_lavada = l.id_tipolavada

  // Pendiente control:
  if ($lavada !== null && $lavada !== '') {
    // Fuerza lavada exacto
    $where[] = 't.lavada = ?';
    $params[] = (int)$lavada;
    $types   .= 'i';
  } elseif ($pendiente !== 'all') {
    // Predeterminado: solo pendientes
    $where[] = 't.lavada = 0';
  }

  // Condición (si tus tablas tienen columna condicion)
  if ($condCtl !== 'all') {
    // Estas columnas suelen existir; si no existieran en tu esquema, quítalas.
    $where[] = 'd.condicion = 1';
    $where[] = 'l.condicion = 1';
    // Si tb_tipolavada también tiene condicion, puedes añadir:
    // $where[] = 't.condicion = 1';
  }

  // Estado mínimo (opcional)
  if (is_int($estadoMin)) {
    $where[] = 'd.id_estado >= ?';
    $params[] = $estadoMin;
    $types   .= 'i';
  }

  // Fecha registro (rango)
  if ($fDesde) {
    $where[] = 'l.fecha_registro >= ?';
    $params[] = $fDesde;
    $types   .= 's';
  }
  if ($fHasta) {
    // semiabierto [desde, hasta+1)
    $where[] = 'l.fecha_registro < ?';
    $params[] = nextDay($fHasta);
    $types   .= 's';
  }

  // Búsquedas
  if ($q) {
    $where[]  = '(d.chasis LIKE ? OR d.motor LIKE ?)';
    $params[] = likeWrap($q);
    $params[] = likeWrap($q);
    $types   .= 'ss';
  }
  if ($chasis) {
    if ($exact) {
      $where[]  = $trimCmp ? 'TRIM(d.chasis) = TRIM(?)' : 'd.chasis = ?';
      $params[] = $chasis;
      $types   .= 's';
    } else {
      $where[]  = 'd.chasis LIKE ?';
      $params[] = likeWrap($chasis);
      $types   .= 's';
    }
  }
  if ($marca)  { $where[]='d.marca LIKE ?';  $params[]=likeWrap($marca);  $types.='s'; }
  if ($modelo) { $where[]='d.modelo LIKE ?'; $params[]=likeWrap($modelo); $types.='s'; }
  if ($color)  { $where[]='d.color LIKE ?';  $params[]=likeWrap($color);  $types.='s'; }

  $whereSql = $where ? ('WHERE ' . implode(' AND ', $where)) : '';

  /* ===== SQL ===== */
  $sql = "
    SELECT
      l.idlavada          AS id_lavada,
      l.fecha_registro,
      l.id_datadai,
      l.estado_lavado,
      d.chasis,
      d.motor,
      d.marca,
      d.modelo,
      d.color,
      d.id_estado,
      t.idtipo_lavada,
      t.lavada
    FROM tb_lavada l
    INNER JOIN tb_datadai   d ON d.iddatadai = l.id_datadai
    INNER JOIN tb_tipolavada t ON t.idtipo_lavada = l.id_tipolavada
    $whereSql
    ORDER BY $orderBySql $orderDirSql
  ";

  $DEBUG['query'] = [
    'where'  => $whereSql,
    'types'  => $types,
    'params' => $params,
    'order'  => "$orderBySql $orderDirSql",
    'sql'    => $sql
  ];

  $stmt = $db->prepare($sql);
  if ($types !== '') { $stmt->bind_param($types, ...$params); }
  $stmt->execute();
  $res = $stmt->get_result();

  $items = [];
  while ($row = $res->fetch_assoc()) {
    $items[] = [
      'id_lavada'      => isset($row['id_lavada']) ? (int)$row['id_lavada'] : null,
      'fecha_registro' => $row['fecha_registro'],
      'id_datadai'     => (int)$row['id_datadai'],
      'estado_lavado'  => $row['estado_lavado'],
      'chasis'         => $row['chasis'],
      'motor'          => $row['motor'],
      'marca'          => $row['marca'],
      'modelo'         => $row['modelo'],
      'color'          => $row['color'],
      'id_estado'      => isset($row['id_estado']) ? (int)$row['id_estado'] : null,
      'idtipo_lavada'  => isset($row['idtipo_lavada']) ? (int)$row['idtipo_lavada'] : null,
      'lavada'         => $row['lavada']
    ];
  }
  $stmt->close();

  $resp = ['success'=>true, 'total'=>count($items), 'items'=>$items];
  if ((int)qp('_debug', 0) === 1) { $resp['_debug'] = $DEBUG; }

  echo json_encode($resp, JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Error interno','error'=>$e->getMessage()], JSON_UNESCAPED_UNICODE);
}
