<?php
// inventario_ultima_fecha_flag30.php — Solo indica si han pasado 30 días (o más)
// GET:
//   - estado (default 'Terminado')  -> case-insensitive + TRIM
//   - any=1  : ignora 'estado' y usa el último con condicion=1
//   - dias_umbral (opcional, default 30)

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Guayaquil');

/* HEADERS */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

try {
  if (function_exists('mysqli_report')) {
    mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
  }

  require __DIR__ . '/../conex.php';
  $db = $conn ?? ($conex ?? null);
  if (!$db instanceof mysqli) {
    http_response_code(500);
    echo json_encode(['success'=>false,'message'=>'Error de conexión a BD']); exit;
  }
  mysqli_set_charset($db, 'utf8mb4');

  $estado      = isset($_GET['estado']) && $_GET['estado'] !== '' ? (string)$_GET['estado'] : 'Terminado';
  $any         = isset($_GET['any']) ? (int)$_GET['any'] : 0;
  $dias_umbral = isset($_GET['dias_umbral']) ? max(1, (int)$_GET['dias_umbral']) : 30;

  // Consulta principal (condicion=1, orden por más reciente)
  if ($any === 1) {
    $sql = "
      SELECT id, estado, fecha_inicio, hora_inicio
      FROM tb_inventario_vin
      WHERE condicion = 1
      ORDER BY fecha_inicio DESC, COALESCE(hora_inicio,'00:00:00') DESC, id DESC
      LIMIT 1
    ";
    $stmt = $db->prepare($sql);
  } else {
    $sql = "
      SELECT id, estado, fecha_inicio, hora_inicio
      FROM tb_inventario_vin
      WHERE condicion = 1
        AND TRIM(UPPER(estado)) = TRIM(UPPER(?))
      ORDER BY fecha_inicio DESC, COALESCE(hora_inicio,'00:00:00') DESC, id DESC
      LIMIT 1
    ";
    $stmt = $db->prepare($sql);
    $stmt->bind_param('s', $estado);
  }

  $stmt->execute();
  $res = $stmt->get_result();

  if (!$row = $res->fetch_assoc()) {
    echo json_encode(['success'=>true, 'found'=>false, 'tiene_30_dias'=>0, 'message'=>'No hay inventarios coincidentes.']);
    exit;
  }

  $fecha_inicio = (string)$row['fecha_inicio'];
  $tz  = new DateTimeZone('America/Guayaquil');
  $hoy = new DateTime('today', $tz);
  $ult = DateTime::createFromFormat('Y-m-d', $fecha_inicio, $tz);

  $dias_transcurridos = null;
  $tiene_30 = 0;

  if ($ult instanceof DateTime) {
    $diff = $ult->diff($hoy);                 // valor absoluto
    $dias_transcurridos = (int)$diff->days;
    if ($diff->invert === 1) {                // si fecha inventario > hoy => negativo
      $dias_transcurridos = -$dias_transcurridos;
    }
    if ($dias_transcurridos >= $dias_umbral) {
      $tiene_30 = 1;
    }
  }

  echo json_encode([
    'success'            => true,
    'found'              => true,
    'tiene_30_dias'      => $tiene_30,          // 1 si >= 30 (o >= dias_umbral), si no 0
    'dias_transcurridos' => $dias_transcurridos // solo referencia
  ], JSON_UNESCAPED_UNICODE);

} catch (mysqli_sql_exception $e) {
  http_response_code(500);
  echo json_encode(['success'=>false,'stage'=>'sql','message'=>'Error SQL','error'=>$e->getMessage()]);
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['success'=>false,'stage'=>'exception','message'=>'Error inesperado','error'=>$e->getMessage()]);
}
