<?php
// inventario_create.php — Crea una tarea de inventario (sin fecha_fin, hora_fin, id_usuario)
// Métodos: POST, OPTIONS
// Cuerpo (JSON o form-data/x-www-form-urlencoded):
//   - estado        (string)    [opcional, por defecto 'Pendiente']
//   - condicion     (int)       [opcional, por defecto 1]
//   - fecha_inicio  (YYYY-MM-DD)[opcional, por defecto fecha actual Ecuador]
//   - hora_inicio   (HH:MM:SS)  [opcional, por defecto hora actual Ecuador]
//
// Respuesta JSON: { success, message, id_inventario, item: { ... } }

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS / CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

try {
  if (function_exists('mysqli_report')) {
    mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
  }

  /* ===== Conexión ===== */
  require __DIR__ . '/../conex.php'; // ajusta la ruta si aplica
  $db = $conn ?? ($conex ?? null);
  if (!$db instanceof mysqli) {
    http_response_code(500);
    echo json_encode(['success'=>false,'message'=>'Error de conexión a BD']); exit;
  }
  mysqli_set_charset($db, 'utf8mb4');

  /* ===== Utilidades ===== */
  function body_json(): array {
    $raw = file_get_contents('php://input');
    if (!$raw) return [];
    $j = json_decode($raw, true);
    return is_array($j) ? $j : [];
  }

  $in = body_json();
  if (empty($in) && !empty($_POST)) { $in = $_POST; }

  // Defaults
  $estado       = (string)($in['estado'] ?? 'PENDIENTE'); // ← por defecto Pendiente
  $condicion    = isset($in['condicion']) ? (int)$in['condicion'] : 1;

  $now = new DateTime('now', new DateTimeZone('America/Guayaquil'));
  $fecha_inicio = (string)($in['fecha_inicio'] ?? $now->format('Y-m-d'));
  $hora_inicio  = (string)($in['hora_inicio']  ?? $now->format('H:i:s'));

  // Validación simple de fecha y hora
  $dt_check = DateTime::createFromFormat('Y-m-d H:i:s', $fecha_inicio.' '.$hora_inicio, new DateTimeZone('America/Guayaquil'));
  if (!$dt_check) {
    http_response_code(400);
    echo json_encode(['success'=>false,'message'=>'Formato inválido de fecha/hora de inicio']); exit;
  }

  /* ===== INSERT ===== */
  $sql = "INSERT INTO tb_inventario_vin
            (estado, fecha_inicio, hora_inicio, condicion)
          VALUES (      ?,          ?,            ?,        ?)";
  $stmt = $db->prepare($sql);
  // Tipos: estado(s), fecha_inicio(s), hora_inicio(s), condicion(i)
  $stmt->bind_param('sssi', $estado, $fecha_inicio, $hora_inicio, $condicion);
  $stmt->execute();

  $id_inventario = $stmt->insert_id ?? 0;

  echo json_encode([
    'success'       => true,
    'message'       => 'Tarea de inventario creada correctamente.',
    'id_inventario' => (int)$id_inventario,
    'item' => [
      'id'           => (int)$id_inventario,
      'estado'       => $estado,
      'fecha_inicio' => $fecha_inicio,
      'hora_inicio'  => $hora_inicio,
      'condicion'    => $condicion
    ]
  ], JSON_UNESCAPED_UNICODE);

} catch (mysqli_sql_exception $e) {
  http_response_code(500);
  echo json_encode([
    'success' => false,
    'stage'   => 'sql',
    'message' => 'Error SQL',
    'error'   => $e->getMessage()
  ], JSON_UNESCAPED_UNICODE);
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'success' => false,
    'stage'   => 'exception',
    'message' => 'Error inesperado',
    'error'   => $e->getMessage()
  ], JSON_UNESCAPED_UNICODE);
}
