<?php
// inventario_add_chasis.php
// Inserta chasis al inventario vigente (último por fecha_inicio).
// Reglas:
// - No recibe id_inventario: se toma el último con estado ('PENDIENTE','EN EJECUCION') y condicion=1.
// - Solo inserta si tb_datadai.id_estado >= 6 y condicion=1.
// - Si es el primer registro y el inventario estaba 'PENDIENTE', pasa a 'EN EJECUCION'.
// - Todo con transacción segura.

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Método no permitido. Usa POST.']); exit;
}

/* ===== Conexión ===== */
require __DIR__ . '/../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Error de conexión a BD']); exit;
}
if (function_exists('mysqli_report')) { mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); }
mysqli_set_charset($db, 'utf8mb4');

/* ===== Leer JSON ===== */
$raw  = file_get_contents('php://input');
$body = json_decode($raw, true);

$chasis = isset($body['chasis']) ? trim((string)$body['chasis']) : '';

if ($chasis === '') {
  http_response_code(400);
  echo json_encode(['success'=>false,'message'=>'Parámetros inválidos. Requiere chasis.']); exit;
}

try {
  $db->begin_transaction();

  // 1️⃣ Seleccionar inventario vigente (último por fecha_inicio)
  $sqlInv = "
    SELECT id, estado, fecha_inicio, condicion
    FROM tb_inventario_vin
    WHERE condicion = 1
      AND estado IN ('PENDIENTE','EN EJECUCION')
    ORDER BY fecha_inicio DESC, id DESC
    LIMIT 1
    FOR UPDATE
  ";
  $rsInv = $db->query($sqlInv);
  $inv = $rsInv->fetch_assoc();

  if (!$inv) {
    $db->rollback();
    http_response_code(404);
    echo json_encode(['success'=>false,'message'=>"No hay inventario activo en estado 'PENDIENTE' o 'EN EJECUCION'."]); exit;
  }

  $id_inventario = (int)$inv['id'];
  $estado_inv = (string)$inv['estado'];

  // 2️⃣ Validar chasis
  $sqlD = "SELECT iddatadai, id_estado, condicion FROM tb_datadai WHERE chasis = ? LIMIT 1";
  $stD  = $db->prepare($sqlD);
  $stD->bind_param('s', $chasis);
  $stD->execute();
  $rowD = $stD->get_result()->fetch_assoc();
  $stD->close();

  if (!$rowD) {
    $db->rollback();
    http_response_code(404);
    echo json_encode(['success'=>false,'message'=>'Chasis no encontrado en tb_datadai']); exit;
  }
  if ((int)$rowD['condicion'] !== 1) {
    $db->rollback();
    http_response_code(400);
    echo json_encode(['success'=>false,'message'=>'El chasis está inactivo (condicion=0)']); exit;
  }
  if ((int)$rowD['id_estado'] < 6) {
    $db->rollback();
    http_response_code(400);
    echo json_encode([
      'success'=>false,
      'message'=>'El chasis no cumple id_estado >= 6',
      'id_estado_actual'=>(int)$rowD['id_estado']
    ]);
    exit;
  }

  $id_datadai = (int)$rowD['iddatadai'];

  // 3️⃣ Evitar duplicado
  $sqlE = "SELECT id FROM tb_inventario_datadai WHERE id_inventario = ? AND id_datadai = ? LIMIT 1";
  $stE = $db->prepare($sqlE);
  $stE->bind_param('ii', $id_inventario, $id_datadai);
  $stE->execute();
  $exists = $stE->get_result()->fetch_assoc();
  $stE->close();

  if ($exists) {
    $db->rollback();
    echo json_encode([
      'success'=>true,
      'message'=>'El chasis ya está registrado en el inventario vigente',
      'data'=>[
        'id_inventario'=>$id_inventario,
        'id_datadai'=>$id_datadai,
        'was_first'=>false,
        'inventario_estado_prev'=>$estado_inv,
        'inventario_estado_updated'=>false,
        'fecha'=>date('Y-m-d'),
        'hora'=>date('H:i:s')
      ]
    ], JSON_UNESCAPED_UNICODE);
    exit;
  }

  // 4️⃣ Verificar si es el primer detalle del inventario
  $sqlC = "SELECT COUNT(*) AS c FROM tb_inventario_datadai WHERE id_inventario = ? AND condicion = 1 FOR UPDATE";
  $stC = $db->prepare($sqlC);
  $stC->bind_param('i', $id_inventario);
  $stC->execute();
  $count = (int)($stC->get_result()->fetch_assoc()['c'] ?? 0);
  $stC->close();

  $was_first = ($count === 0);
  $estado_updated = false;

  // 5️⃣ Si es el primero y el inventario está 'PENDIENTE', lo pasa a 'EN EJECUCION'
  if ($was_first && strtoupper($estado_inv) === 'PENDIENTE') {
    $sqlUp = "UPDATE tb_inventario_vin SET estado = 'EN EJECUCION' WHERE id = ? AND condicion = 1";
    $stUp = $db->prepare($sqlUp);
    $stUp->bind_param('i', $id_inventario);
    $stUp->execute();
    $estado_updated = ($stUp->affected_rows >= 0);
    $stUp->close();
    $estado_inv = 'EN EJECUCION';
  }

  // 6️⃣ Insertar nuevo detalle
  $fecha = date('Y-m-d');
  $hora  = date('H:i:s');
  $sqlIns = "INSERT INTO tb_inventario_datadai (id_inventario, id_datadai, fecha, hora, condicion)
             VALUES (?, ?, ?, ?, 1)";
  $stIns = $db->prepare($sqlIns);
  $stIns->bind_param('iiss', $id_inventario, $id_datadai, $fecha, $hora);
  $stIns->execute();
  $stIns->close();

  $db->commit();

  echo json_encode([
    'success'=>true,
    'message'=>'Insertado correctamente en el inventario vigente',
    'data'=>[
      'id_inventario'=>$id_inventario,
      'estado_inventario'=>$estado_inv,
      'id_datadai'=>$id_datadai,
      'was_first'=>$was_first,
      'inventario_estado_updated'=>$estado_updated,
      'fecha'=>$fecha,
      'hora'=>$hora
    ]
  ], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
  if ($db->errno) { $db->rollback(); }
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Excepción','error'=>$e->getMessage()], JSON_UNESCAPED_UNICODE);
}
