<?php
// incidente_tipos.php — GET: lista tipos por incidente | POST: reemplaza set de tipos
// Usa conex.php ($conn o $conex) y MySQLi
declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS / CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== Helpers ===== */
function api_json(array $data, int $code = 200): void {
  http_response_code($code);
  echo json_encode($data, JSON_UNESCAPED_UNICODE);
  exit;
}
function int_array_unique_positive($arr): array {
  $out = [];
  if (is_array($arr)) {
    foreach ($arr as $v) {
      $i = (int)$v;
      if ($i > 0) { $out[] = $i; }
    }
  }
  return array_values(array_unique($out));
}
function placeholders(int $n): string {
  return implode(', ', array_fill(0, $n, '?'));
}

/* ===== Conexión (usa tu patrón) ===== */
$pathConex = realpath(__DIR__ . '/../conex.php')
         ?: realpath(__DIR__ . '/../../conex.php')
         ?: realpath(__DIR__ . '/conex.php');
if (!$pathConex) {
  api_json(['success'=>false,'data'=>[], 'message'=>'No se encontró conex.php'], 500);
}
require_once $pathConex;

$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) {
  api_json(['success'=>false,'data'=>[], 'message'=>'Sin conexión BD'], 500);
}
mysqli_set_charset($db, 'utf8mb4');
if (function_exists('mysqli_report')) {
  mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
}

/* ===== Asegurar tabla pivote ===== */
try {
  $ddl = "CREATE TABLE IF NOT EXISTS tb_incidente_tipo (
            incident_id INT NOT NULL,
            id_tipo     INT NOT NULL,
            PRIMARY KEY (incident_id, id_tipo),
            KEY idx_tipo (id_tipo)
          ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
  $db->query($ddl);
} catch (Throwable $e) {
  api_json(['success'=>false, 'message'=>'Error creando tabla pivote', 'detail'=>$e->getMessage()], 500);
}

/* ===== Lógica ===== */
$method = $_SERVER['REQUEST_METHOD'];

try {
  if ($method === 'GET') {
    $incident_id = isset($_GET['incident_id']) ? (int)$_GET['incident_id'] : 0;
    if ($incident_id <= 0) {
      api_json(['success'=>false,'message'=>'incident_id requerido (GET)'], 422);
    }

    // Validar existencia de incidente
    $stmtChk = $db->prepare("SELECT 1 FROM tb_incidente WHERE idincidente=? LIMIT 1");
    $stmtChk->bind_param('i', $incident_id);
    $stmtChk->execute();
    $resChk = $stmtChk->get_result();
    if ($resChk->num_rows === 0) {
      api_json(['success'=>false,'message'=>'Incidente no existe'], 404);
    }

    // Traer tipos asociados (con nombre tipo y taller)
    $sql = "SELECT 
              it.id_tipo,
              tr.tipo,
              tr.id_taller,
              COALESCE(t.taller, '') AS taller
            FROM tb_incidente_tipo it
            LEFT JOIN tb_tipo_reparacion tr ON tr.id = it.id_tipo
            LEFT JOIN tb_taller t ON t.idtaller = tr.id_taller
            WHERE it.incident_id=?
            ORDER BY t.taller, tr.tipo";
    $stmt = $db->prepare($sql);
    $stmt->bind_param('i', $incident_id);
    $stmt->execute();
    $rows = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

    $ids = [];
    foreach ($rows as $r) { $ids[] = (int)$r['id_tipo']; }

    api_json([
      'success'=>true,
      'incident_id'=>$incident_id,
      'total'=>count($rows),
      'ids'=>$ids,
      'items'=>$rows
    ], 200);
  }

  if ($method === 'POST') {
    $raw = file_get_contents('php://input');
    $in  = json_decode($raw, true);
    if (!is_array($in)) { $in = []; }

    $incident_id = isset($in['incident_id']) ? (int)$in['incident_id'] : 0;
    $tipos_ids   = int_array_unique_positive($in['tipos_ids'] ?? []);
    if ($incident_id <= 0) {
      api_json(['success'=>false,'message'=>'incident_id requerido (POST)'], 422);
    }

    // Validar incidente
    $stmtInc = $db->prepare("SELECT 1 FROM tb_incidente WHERE idincidente=? LIMIT 1");
    $stmtInc->bind_param('i', $incident_id);
    $stmtInc->execute();
    $resInc = $stmtInc->get_result();
    if ($resInc->num_rows === 0) {
      api_json(['success'=>false,'message'=>'Incidente no existe'], 404);
    }

    // Validar que los tipos existan (opcional: condicion=1 si lo usas)
    if (!empty($tipos_ids)) {
      $ph = placeholders(count($tipos_ids));
      $sqlVal = "SELECT id FROM tb_tipo_reparacion WHERE id IN ($ph)";
      // Si quieres solo activos:
      // $sqlVal .= " AND condicion=1";
      $stmtVal = $db->prepare($sqlVal);
      $types = str_repeat('i', count($tipos_ids));
      $stmtVal->bind_param($types, ...$tipos_ids);
      $stmtVal->execute();
      $found = $stmtVal->get_result()->fetch_all(MYSQLI_ASSOC);

      $foundIds = [];
      foreach ($found as $f) { $foundIds[] = (int)$f['id']; }
      $invalid = array_values(array_diff($tipos_ids, $foundIds));
      if (!empty($invalid)) {
        api_json([
          'success'=>false,
          'message'=>'Algunos id_tipo no existen en tb_tipo_reparacion',
          'invalid_ids'=>$invalid
        ], 422);
      }
    }

    // Reemplazo atómico
    $db->begin_transaction();

    $stmtDel = $db->prepare("DELETE FROM tb_incidente_tipo WHERE incident_id=?");
    $stmtDel->bind_param('i', $incident_id);
    $stmtDel->execute();

    if (!empty($tipos_ids)) {
      $stmtIns = $db->prepare("INSERT INTO tb_incidente_tipo (incident_id, id_tipo) VALUES (?, ?)");
      foreach ($tipos_ids as $tid) {
        $stmtIns->bind_param('ii', $incident_id, $tid);
        $stmtIns->execute();
      }
    }

    $db->commit();

    // Devolver set final
    $sqlOut = "SELECT 
                 it.id_tipo,
                 tr.tipo,
                 tr.id_taller,
                 COALESCE(t.taller, '') AS taller
               FROM tb_incidente_tipo it
               LEFT JOIN tb_tipo_reparacion tr ON tr.id = it.id_tipo
               LEFT JOIN tb_taller t ON t.idtaller = tr.id_taller
               WHERE it.incident_id=?
               ORDER BY t.taller, tr.tipo";
    $stmtOut = $db->prepare($sqlOut);
    $stmtOut->bind_param('i', $incident_id);
    $stmtOut->execute();
    $rows = $stmtOut->get_result()->fetch_all(MYSQLI_ASSOC);

    $ids = [];
    foreach ($rows as $r) { $ids[] = (int)$r['id_tipo']; }

    api_json([
      'success'=>true,
      'message'=>'Tipos de reparación actualizados',
      'incident_id'=>$incident_id,
      'total'=>count($rows),
      'ids'=>$ids,
      'items'=>$rows
    ], 200);
  }

  api_json(['success'=>false,'message'=>'Método no permitido'], 405);

} catch (Throwable $e) {
  // Intentar rollback si hay transacción
  if ($db instanceof mysqli) { @mysqli_rollback($db); }
  api_json(['success'=>false,'message'=>'Error del servidor','detail'=>$e->getMessage()], 500);
}
