<?php
// guardar_impronta_por_chasis.php
// Inserta en tb_impronta a partir de chasis (resuelve iddatadai) o id_datadai.
// Campos guardados: (id_datadai, fecha, usuario, foto, observacion, condicion=1)
//
// Métodos: POST, OPTIONS
// Admite:
//   - multipart/form-data: foto (file) | image (file) | foto_base64 (text) | foto (ruta)
//   - application/json:    foto_base64 | foto (ruta)
// Parámetros clave:
//   - chasis            (string)  -> requerido si no envías id_datadai
//   - id_datadai        (int)     -> opcional; si viene, se usa directo
//   - exact=1|0         (int)     -> por defecto 1 (igualdad)
//   - trim=1|0          (int)     -> por defecto 1 (aplica TRIM a comparaciones exactas)
//   - cond=1|all        (string)  -> por defecto '1' (filtra d.condicion = 1)
//   - estado_min=int    (int)     -> opcional; p.ej. 6 para requerir id_estado >= 6
//   - usuario           (int)     -> opcional; 0 si no se envía
//   - observacion       (string)  -> opcional
//
// NOTA: si tu columna tb_impronta.fecha es DATE, cambia el formato a 'Y-m-d'.

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== Cabeceras ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');
if (($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'OPTIONS') { http_response_code(204); exit; }

/* ===== Conexión ===== */
$pathConex = realpath(__DIR__ . '/conex.php') ?: realpath(__DIR__ . '/../conex.php') ?: realpath(__DIR__ . '/../../conex.php');
if (!$pathConex) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'No se encontró conex.php']); exit; }
require_once $pathConex;

$db = null;
if (isset($conn) && $conn instanceof mysqli)      { $db = $conn; }
elseif (isset($conex) && $conex instanceof mysqli){ $db = $conex; }
if (!$db instanceof mysqli) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'Sin conexión a BD']); exit; }
if (function_exists('mysqli_report')) mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
mysqli_set_charset($db, 'utf8mb4');

/* ===== Config subida ===== */
const IMPRONTA_DIR_REL = 'uploads/improntas';
const MAX_BYTES = 10 * 1024 * 1024;
const ALLOWED_MIMES = ['image/jpeg','image/png','image/webp','image/gif'];

/* ===== Helpers ===== */
function fail(int $code, string $msg, array $extra = []): void {
  http_response_code($code);
  echo json_encode(['success'=>false,'message'=>$msg,'data'=>$extra], JSON_UNESCAPED_UNICODE);
  exit;
}
function ensure_dir(string $abs): void {
  if (!is_dir($abs) && !@mkdir($abs, 0775, true) && !is_dir($abs)) {
    fail(500, 'No se pudo crear el directorio de subida', ['dir'=>$abs]);
  }
}
function finfo_mime_safe(string $fileAbs): string {
  $finfo = new finfo(FILEINFO_MIME_TYPE);
  return $finfo->file($fileAbs) ?: 'application/octet-stream';
}
function ext_by_mime(string $mime): string {
  return match ($mime) {
    'image/jpeg' => 'jpg',
    'image/png'  => 'png',
    'image/webp' => 'webp',
    'image/gif'  => 'gif',
    default      => 'bin',
  };
}
function unique_name(string $dirAbs, string $ext): string {
  $ext = ltrim($ext, '.');
  do {
    $n = 'imp_' . date('Ymd_His') . '_' . bin2hex(random_bytes(4)) . '.' . $ext;
  } while (file_exists($dirAbs . DIRECTORY_SEPARATOR . $n));
  return $n;
}
function save_base64_image(string $payload, string $dirAbs, string $dirRel): array {
  // Retorna [abs, rel]
  if (strpos($payload, 'data:') === 0) {
    $payload = preg_replace('#^data:image/\w+;base64,#', '', $payload);
  }
  $bin = base64_decode($payload, true);
  if ($bin === false) { fail(400, 'Base64 inválido'); }
  if (strlen((string)$bin) > MAX_BYTES) { fail(413, 'Imagen demasiado grande'); }

  $tmp = $dirAbs . DIRECTORY_SEPARATOR . 'tmp_' . bin2hex(random_bytes(3));
  file_put_contents($tmp, $bin);

  $realMime = finfo_mime_safe($tmp);
  if (!in_array($realMime, ALLOWED_MIMES, true)) { @unlink($tmp); fail(415, 'Tipo no permitido: '.$realMime); }

  $ext = ext_by_mime($realMime);
  $finalAbs = $dirAbs . DIRECTORY_SEPARATOR . unique_name($dirAbs, $ext);
  if (!rename($tmp, $finalAbs)) { @unlink($tmp); fail(500, 'No se pudo guardar la imagen'); }

  $finalRel = rtrim($dirRel, '/\\') . '/' . basename($finalAbs);
  $finalRel = str_replace('\\', '/', $finalRel);
  return [$finalAbs, $finalRel];
}
function gp(string $k, $d=null){ return $_POST[$k] ?? $_GET[$k] ?? $d; }

/**
 * Resuelve iddatadai desde chasis según parámetros.
 */
function resolve_iddatadai_from_chasis(mysqli $db, string $chasis, bool $exact=true, bool $trimCmp=true, string $condCtl='1', ?int $estadoMin=null): ?array {
  $where = [];
  $types = '';
  $params = [];

  if ($exact) {
    $where[] = $trimCmp ? 'TRIM(d.chasis) = TRIM(?)' : 'd.chasis = ?';
    $types  .= 's';
    $params[] = $chasis;
  } else {
    $where[] = 'd.chasis LIKE ?';
    $types  .= 's';
    $params[] = '%'.$chasis.'%';
  }
  if ($condCtl !== 'all') {
    $where[] = 'd.condicion = 1';
  }
  if (is_int($estadoMin)) {
    $where[] = 'd.id_estado >= ?';
    $types  .= 'i';
    $params[] = $estadoMin;
  }

  $whereSql = implode(' AND ', $where);
  $sql = "SELECT d.iddatadai, d.chasis, d.id_estado, d.condicion
          FROM tb_datadai d
          WHERE $whereSql
          ORDER BY d.iddatadai DESC
          LIMIT 1";

  $stmt = $db->prepare($sql);
  $stmt->bind_param($types, ...$params);
  $stmt->execute();
  $res = $stmt->get_result();
  $row = $res->fetch_assoc();
  $stmt->close();

  if (!$row) return null;
  return [
    'iddatadai' => (int)$row['iddatadai'],
    'chasis'    => $row['chasis'],
    'id_estado' => (int)$row['id_estado'],
    'condicion' => (int)$row['condicion'],
  ];
}

/* ===== Lógica principal ===== */
try {
  if (($_SERVER['REQUEST_METHOD'] ?? 'POST') !== 'POST') { fail(405, 'Usa POST'); }

  // Carpeta destino: /uploads/improntas/YYYY/MM
  $subdir = date('Y') . '/' . date('m');
  $dirRel = rtrim(IMPRONTA_DIR_REL, '/\\') . '/' . $subdir;
  $dirAbs = __DIR__ . DIRECTORY_SEPARATOR . str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $dirRel);
  ensure_dir($dirAbs);

  $contentType = $_SERVER['CONTENT_TYPE'] ?? '';
  $isMultipart = stripos($contentType, 'multipart/form-data') !== false;

  // Parámetros comunes
  $estadoMin = gp('estado_min', null); // si viene string, castear abajo
  $estadoMin = ($estadoMin === null || $estadoMin === '') ? null : (int)$estadoMin;
  $exact     = (int)gp('exact', 1) === 1;
  $trimCmp   = (int)gp('trim', 1) === 1;
  $condCtl   = strtolower((string)gp('cond', '1')); // '1' | 'all'

  $id_datadai = null;
  $usuario = 0;
  $observacion = null;
  $fotoAbs = null;
  $fotoRel = null; // cadena para BD ('' si no se envía)

  if ($isMultipart) {
    $id_datadai  = isset($_POST['id_datadai']) ? (int)$_POST['id_datadai'] : null;
    $usuario     = isset($_POST['usuario']) ? (int)$_POST['usuario'] : 0;
    $observacion = isset($_POST['observacion']) ? trim((string)$_POST['observacion']) : null;
    $chasis      = isset($_POST['chasis']) ? trim((string)$_POST['chasis']) : '';

    // Foto (archivo)
    $fileField = null;
    if (!empty($_FILES['foto']['tmp_name']))   { $fileField = 'foto'; }
    if (!empty($_FILES['image']['tmp_name']))  { $fileField = $fileField ?: 'image'; }

    if ($fileField) {
      $err  = (int)($_FILES[$fileField]['error'] ?? UPLOAD_ERR_NO_FILE);
      if ($err !== UPLOAD_ERR_OK) { fail(400, 'Error de subida (código '.$err.')'); }
      $size = (int)($_FILES[$fileField]['size'] ?? 0);
      if ($size <= 0) { fail(400, 'Archivo vacío'); }
      if ($size > MAX_BYTES) { fail(413, 'Imagen demasiado grande'); }

      $tmp = $_FILES[$fileField]['tmp_name'];
      $realMime = finfo_mime_safe($tmp);
      if (!in_array($realMime, ALLOWED_MIMES, true)) { fail(415, 'Tipo no permitido: '.$realMime); }
      $ext = ext_by_mime($realMime);
      $finalAbs = $dirAbs . DIRECTORY_SEPARATOR . unique_name($dirAbs, $ext);
      if (!move_uploaded_file($tmp, $finalAbs)) { fail(500, 'No se pudo mover la imagen'); }
      $fotoAbs = $finalAbs;
      $fotoRel = rtrim($dirRel, '/\\') . '/' . basename($finalAbs);
    }

    // Foto base64
    if (!$fotoAbs && !empty($_POST['foto_base64'])) {
      [$fotoAbs, $fotoRel] = save_base64_image((string)$_POST['foto_base64'], $dirAbs, $dirRel);
    }

    // Ruta existente
    if (!$fotoAbs && empty($_POST['foto_base64']) && !empty($_POST['foto'])) {
      $ruta = str_replace('\\','/', trim((string)$_POST['foto']));
      if (strpos($ruta, '..') !== false) { fail(400, 'Ruta de foto inválida'); }
      $fotoRel = $ruta;
    }

  } else {
    // JSON
    $raw = file_get_contents('php://input') ?: '';
    $data = json_decode($raw, true);
    if (!is_array($data)) { fail(400, 'JSON inválido'); }

    $id_datadai  = isset($data['id_datadai']) ? (int)$data['id_datadai'] : null;
    $usuario     = isset($data['usuario']) ? (int)$data['usuario'] : 0;
    $observacion = isset($data['observacion']) ? trim((string)$data['observacion']) : null;
    $chasis      = isset($data['chasis']) ? trim((string)$data['chasis']) : '';

    if (!empty($data['foto_base64'])) {
      [$fotoAbs, $fotoRel] = save_base64_image((string)$data['foto_base64'], $dirAbs, $dirRel);
    } elseif (!empty($data['foto'])) {
      $ruta = str_replace('\\','/', trim((string)$data['foto']));
      if (strpos($ruta, '..') !== false) { fail(400, 'Ruta de foto inválida'); }
      $fotoRel = $ruta;
    }
  }

  // Resolver id_datadai si no fue enviado
  if (!$id_datadai) {
    if ($chasis === '') { fail(400, 'Debes enviar chasis o id_datadai'); }
    $found = resolve_iddatadai_from_chasis($db, $chasis, $exact, $trimCmp, $condCtl, $estadoMin);
    if (!$found) { fail(404, 'No se encontró iddatadai para el chasis', ['chasis'=>$chasis, 'estado_min'=>$estadoMin, 'cond'=>$condCtl]); }
    $id_datadai = $found['iddatadai'];
  }

  // Preparar datos de inserción
  $fecha = date('Y-m-d H:i:s'); // usa 'Y-m-d' si tu columna es DATE
  $condicion = 1;
  if ($fotoRel === null) { $fotoRel = ''; } // evita NULL si la columna no lo permite

  // INSERT
  $sql = "INSERT INTO tb_impronta (id_datadai, fecha, usuario, foto, observacion, condicion)
          VALUES (?, ?, ?, ?, ?, ?)";
  $stmt = $db->prepare($sql);
  $stmt->bind_param('isissi', $id_datadai, $fecha, $usuario, $fotoRel, $observacion, $condicion);
  $stmt->execute();
  $idNew = (int)$db->insert_id;
  $rows  = $stmt->affected_rows;
  $stmt->close();

  echo json_encode([
    'success' => true,
    'message' => 'Impronta guardada correctamente',
    'data' => [
      'id' => $idNew,
      'rows_affected' => $rows,
      'id_datadai' => $id_datadai,
      'fecha' => $fecha,
      'usuario' => $usuario,
      'foto' => $fotoRel,
      'observacion' => $observacion,
      'condicion' => $condicion
    ]
  ], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
  fail(400, 'Error: ' . $e->getMessage());
}
