<?php
// views/pages/api_app/destino_list.php
// API: Lista de destinos para poblar selects (GET, OPTIONS)
// Tabla: tb_destino(id, nombre, condicion)
// Filtros: ?activo=1|0 (default 1), ?q=texto, ?id=5, ?ids=1,2,3, ?limit=100, ?offset=0, ?order=nombre|id
// Ejemplo: /api_app/destino_list.php?activo=1&q=guey&limit=50

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== CORS / JSON / No-Cache ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== Manejo de errores como JSON ===== */
mysqli_report(MYSQLI_REPORT_OFF);
set_error_handler(function($n,$s,$f,$l){
  http_response_code(500);
  echo json_encode(['success'=>false,'stage'=>'php_error','message'=>"PHP:$n $s @ $f:$l"]); exit;
});
set_exception_handler(function($e){
  http_response_code(500);
  echo json_encode(['success'=>false,'stage'=>'exception','message'=>$e->getMessage()]); exit;
});

/* ===== Conexión ===== */
$pathConex = realpath(__DIR__ . '/../../conex.php') ?: realpath(__DIR__ . '/../conex.php') ?: null;
if (!$pathConex) { http_response_code(500); echo json_encode(['success'=>false,'stage'=>'conn','message'=>'No se encontró conex.php']); exit; }
require_once $pathConex;

$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) { http_response_code(500); echo json_encode(['success'=>false,'stage'=>'conn','message'=>'Sin conexión BD']); exit; }
mysqli_set_charset($db, 'utf8mb4');
@$db->query("SET time_zone='-05:00'");

/* ===== Helpers ===== */
function gp(string $k, $d=null){ return $_GET[$k] ?? $d; }
function parse_ids($v): array {
  if (is_array($v)) return array_values(array_filter(array_map('intval',$v), fn($x)=>$x>0));
  if (is_string($v) && $v !== '') return array_values(array_filter(array_map('intval', explode(',', $v)), fn($x)=>$x>0));
  if (is_numeric($v)) { $i=(int)$v; return $i>0 ? [$i] : []; }
  return [];
}

/* ===== Parámetros ===== */
$id        = (int)gp('id', 0);
$ids       = parse_ids(gp('ids', ''));
$q         = trim((string)gp('q', ''));
$activoStr = gp('activo', '1');                  // por defecto solo activos
$activo    = ($activoStr === '' ? null : (int)$activoStr); // null = todos
$limit     = max(1, min(500, (int)gp('limit', 100)));
$offset    = max(0, (int)gp('offset', 0));
$orderKey  = strtolower((string)gp('order', 'nombre'));
$orderBy   = $orderKey === 'id' ? 'd.id' : 'd.nombre';      // nombre por defecto

/* ===== WHERE dinámico (prepared) ===== */
$where = [];
$params = [];
$types  = '';

if ($id > 0) {
  $where[] = 'd.id = ?';
  $types  .= 'i';
  $params[] = $id;
}
if (!empty($ids)) {
  // Construye IN seguro
  $in = implode(',', array_fill(0, count($ids), '?'));
  $where[] = "d.id IN ($in)";
  $types  .= str_repeat('i', count($ids));
  $params = array_merge($params, $ids);
}
if ($q !== '') {
  $where[] = 'd.nombre LIKE ?';
  $types  .= 's';
  $params[] = '%'.$q.'%';
}
if ($activo !== null) {
  $where[] = 'd.condicion = ?';
  $types  .= 'i';
  $params[] = $activo;
}

$WHERE_SQL = $where ? ('WHERE '.implode(' AND ', $where)) : '';

/* ===== Conteo ===== */
$sqlCount = "SELECT COUNT(*) AS total FROM tb_destino d $WHERE_SQL";
$stmtC = $db->prepare($sqlCount);
if ($types) $stmtC->bind_param($types, ...$params);
$stmtC->execute();
$total = 0;
if ($resC = $stmtC->get_result()) {
  $rowC = $resC->fetch_assoc();
  $total = (int)($rowC['total'] ?? 0);
  $resC->free();
}
$stmtC->close();

/* ===== Datos ===== */
$sql = "SELECT d.id, d.nombre, d.condicion
        FROM tb_destino d
        $WHERE_SQL
        ORDER BY $orderBy ASC
        LIMIT ? OFFSET ?";
$stmt = $db->prepare($sql);

$types2 = $types . 'ii';
$params2 = $params;
$params2[] = $limit;
$params2[] = $offset;

$stmt->bind_param($types2, ...$params2);
$stmt->execute();
$items = [];
if ($res = $stmt->get_result()) {
  while ($r = $res->fetch_assoc()) {
    $items[] = [
      'id'        => (int)$r['id'],
      'nombre'    => $r['nombre']
      //'condicion' => (int)$r['condicion']
    ];
  }
  $res->free();
}
$stmt->close();

/* ===== Respuesta ===== */
echo json_encode([
  'success' => true,
  //'total'   => $total,
  //'limit'   => $limit,
  //'offset'  => $offset,
  'items'   => $items
], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
