<?php
// views/pages/api_app/anden_cerrar_y_liberar.php
// Libera andén (tb_anden.activo = 0) y cierra asignación en tb_datadai_anden
// - Filtra por id_anden (requerido) y opcionalmente chasis (para amarrar id_datadai)
// - Solo cierra asignación con condicion = 1 y fecha_f IS NULL
// - fecha_f = NOW(); dia = TIMESTAMPDIFF(DAY, fecha_i, NOW())
// - Timezone: America/Guayaquil
//
// Métodos: GET | POST (JSON / form-data)
// Parámetros:
//   id_anden : int  (requerido)
//   chasis   : string (opcional; si viene, se valida/filtra la asignación por ese vehículo)
//
// Respuesta: JSON

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== Headers / CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== Conexión ===== */
require_once __DIR__ . '/../conex.php';
$mysqli = $conn ?? ($conex ?? null);
if (!$mysqli instanceof mysqli) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Error de conexión a BD']); exit;
}
if (function_exists('mysqli_report')) { mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); }
mysqli_set_charset($mysqli, 'utf8mb4');

/* ===== Input ===== */
function inData(): array {
  $ct = $_SERVER['CONTENT_TYPE'] ?? '';
  if ($_SERVER['REQUEST_METHOD'] === 'POST' && stripos($ct, 'application/json') !== false) {
    $raw = file_get_contents('php://input');
    if ($raw !== false && strlen(trim($raw)) > 0) {
      $data = json_decode($raw, true);
      if (json_last_error() === JSON_ERROR_NONE && is_array($data)) return $data;
    }
  }
  if (!empty($_POST)) return $_POST;
  return $_GET;
}
$in = inData();
$id_anden = isset($in['id_anden']) ? (int)$in['id_anden'] : 0;
$chasis   = isset($in['chasis']) ? trim((string)$in['chasis']) : '';

if ($id_anden <= 0) {
  http_response_code(400);
  echo json_encode(['success'=>false,'message'=>'Parámetro requerido: id_anden > 0']); exit;
}

/* ===== Helpers ===== */
function getVehiculoByChasis(mysqli $db, string $chasis): ?array {
  $sql = "SELECT iddatadai, chasis FROM tb_datadai
          WHERE UPPER(TRIM(chasis)) = UPPER(TRIM(?))
          ORDER BY iddatadai DESC
          LIMIT 1";
  $st = $db->prepare($sql);
  $st->bind_param('s', $chasis);
  $st->execute();
  $rs = $st->get_result();
  $row = $rs->fetch_assoc();
  $st->close();
  return $row ?: null;
}

try {
  $mysqli->begin_transaction();

  /* 1) Lock del andén por id (validar existencia) */
  $sqlAnden = "SELECT id, id_ubicacion, etiqueta, activo, condicion
               FROM tb_anden
               WHERE id = ?
               FOR UPDATE";
  $st = $mysqli->prepare($sqlAnden);
  $st->bind_param('i', $id_anden);
  $st->execute();
  $rs = $st->get_result();
  $anden = $rs->fetch_assoc();
  $st->close();

  if (!$anden) throw new RuntimeException('El andén especificado no existe.');

  /* 2) (Opcional) resolver id_datadai por chasis para filtrar asignación */
  $id_datadai = null;
  if ($chasis !== '') {
    $veh = getVehiculoByChasis($mysqli, $chasis);
    if (!$veh) throw new RuntimeException('No se encontró id_datadai para el chasis indicado.');
    $id_datadai = (int)$veh['iddatadai'];
  }

  /* 3) Buscar la asignación ABIERTA (condicion=1 y fecha_f IS NULL) del andén
        Si se envía chasis, además filtrar por id_datadai */
  if ($id_datadai !== null) {
    $sqlAsig = "SELECT id, id_datadai, id_anden, fecha_i, fecha_f, dia, condicion
                FROM tb_datadai_anden
                WHERE id_anden = ? AND id_datadai = ? AND condicion = 1 AND fecha_f IS NULL
                ORDER BY id DESC
                LIMIT 1";
    $st = $mysqli->prepare($sqlAsig);
    $st->bind_param('ii', $id_anden, $id_datadai);
  } else {
    $sqlAsig = "SELECT id, id_datadai, id_anden, fecha_i, fecha_f, dia, condicion
                FROM tb_datadai_anden
                WHERE id_anden = ? AND condicion = 1 AND fecha_f IS NULL
                ORDER BY id DESC
                LIMIT 1";
    $st = $mysqli->prepare($sqlAsig);
    $st->bind_param('i', $id_anden);
  }
  $st->execute();
  $rs = $st->get_result();
  $asig = $rs->fetch_assoc();
  $st->close();

  if (!$asig) {
    // No hay asignación abierta con condicion=1 (igual liberamos el andén si está marcado activo)
    // Si quieres que sea estrictamente obligatorio cerrar asignación, descomenta la línea siguiente:
    // throw new RuntimeException('No existe asignación abierta (condicion=1) para este andén (y chasis, si se envió).');
  } else {
    // 4) Cerrar asignación: fecha_f = NOW(), dia = diff días
    $sqlClose = "UPDATE tb_datadai_anden
                 SET fecha_f = NOW(),
                     dia     = TIMESTAMPDIFF(DAY, fecha_i, NOW())
                 WHERE id = ?";
    $st = $mysqli->prepare($sqlClose);
    $st->bind_param('i', $asig['id']);
    $st->execute();
    $st->close();
  }

  /* 5) Poner el andén en libre (activo = 0) */
  if ((int)$anden['activo'] === 1) {
    $sqlFree = "UPDATE tb_anden SET activo = 0 WHERE id = ?";
    $st = $mysqli->prepare($sqlFree);
    $st->bind_param('i', $id_anden);
    $st->execute();
    $st->close();
    $nuevoActivo = 0;
  } else {
    $nuevoActivo = (int)$anden['activo']; // ya estaba 0 u otro valor
  }

  /* 6) Traer asignación final (si existía) para la respuesta */
  $finalAsig = null;
  if ($asig) {
    $sqlGet = "SELECT id, id_datadai, id_anden, fecha_i, fecha_f, dia, condicion
               FROM tb_datadai_anden
               WHERE id = ?";
    $st = $mysqli->prepare($sqlGet);
    $st->bind_param('i', $asig['id']);
    $st->execute();
    $rs = $st->get_result();
    $finalAsig = $rs->fetch_assoc();
    $st->close();
  }

  $mysqli->commit();

  echo json_encode([
    'success' => true,
    'anden' => [
      'id'          => (int)$anden['id'],
      'etiqueta'    => $anden['etiqueta'] ?? null,
      'activo_prev' => isset($anden['activo']) ? (int)$anden['activo'] : null,
      'activo'      => $nuevoActivo,
      'condicion'   => isset($anden['condicion']) ? (int)$anden['condicion'] : null
    ],
    'asignacion' => $finalAsig ? [
      'id'         => (int)$finalAsig['id'],
      'id_datadai' => (int)$finalAsig['id_datadai'],
      'id_anden'   => (int)$finalAsig['id_anden'],
      'fecha_i'    => $finalAsig['fecha_i'],
      'fecha_f'    => $finalAsig['fecha_f'],
      'dia'        => isset($finalAsig['dia']) ? (int)$finalAsig['dia'] : null,
      'condicion'  => isset($finalAsig['condicion']) ? (int)$finalAsig['condicion'] : null
    ] : null,
    'nota' => $asig ? 'Andén liberado y asignación cerrada.' : 'Andén liberado. No había asignación abierta con condicion=1.'
  ]);
} catch (Throwable $e) {
  try { $mysqli->rollback(); } catch (\Throwable $e2) {}
  http_response_code(400);
  echo json_encode(['success'=>false,'message'=>$e->getMessage()]);
}
