<?php
// views/pages/api_app/bateria_update_cargalenta.php
// UPDATE por CHASIS en tb_bateria, solo si existe un registro con hora_inicio NOT NULL.
// Actualiza dinámicamente: estado, valor, id_unidad, resistencia, vida, foto,
// accion, observacion, tiempo, recomendacion, hora_fin.

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS / CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== BD ===== */
require_once __DIR__ . '/../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'Sin conexión a BD']); exit; }
if (function_exists('mysqli_report')) { mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); }
mysqli_set_charset($db, 'utf8mb4');
@$db->query("SET time_zone='-05:00'");

/* ===== INPUT ===== */
$raw = file_get_contents('php://input') ?: '';
$in  = json_decode($raw, true);
if (!is_array($in)) { $in = $_POST ?: []; }

$debug = isset($_GET['_debug']) ? 1 : 0;
$S = static function($k,$def=null) use($in){ return isset($in[$k]) ? (is_string($in[$k]) ? trim($in[$k]) : $in[$k]) : $def; };
$I = static function($k,$def=null) use($in){ return (isset($in[$k]) && $in[$k] !== '' && $in[$k] !== null) ? (int)$in[$k] : $def; };

/* ===== Requeridos ===== */
$chasis = strtoupper((string)$S('chasis', ''));
if ($chasis === '') { http_response_code(400); echo json_encode(['success'=>false,'message'=>'El parámetro "chasis" es requerido']); exit; }

/* ===== Resolver id_datadai por CHASIS ===== */
try {
  // 1) id_datadai
  $q = $db->prepare("SELECT iddatadai FROM tb_datadai WHERE UPPER(chasis)=? LIMIT 1");
  $q->bind_param('s', $chasis);
  $q->execute();
  $rowD = $q->get_result()->fetch_assoc();
  $q->close();

  if (!$rowD) { http_response_code(404); echo json_encode(['success'=>false,'message'=>'Chasis no encontrado']); exit; }
  $id_datadai = (int)$rowD['iddatadai'];

  // 2) Encontrar el registro objetivo en tb_bateria con hora_inicio NOT NULL (el más reciente)
  $q2 = $db->prepare("
      SELECT idbateria
      FROM tb_bateria
      WHERE id_datadai = ? AND hora_inicio IS NOT NULL
      ORDER BY fecha DESC, hora_inicio DESC, idbateria DESC
      LIMIT 1
  ");
  $q2->bind_param('i', $id_datadai);
  $q2->execute();
  $rowB = $q2->get_result()->fetch_assoc();
  $q2->close();

  if (!$rowB) {
    http_response_code(409);
    echo json_encode([
      'success'=>false,
      'message'=>'No existe registro en tb_bateria con hora_inicio distinto de NULL para este chasis'
    ]);
    exit;
  }
  $idbateria = (int)$rowB['idbateria'];

  /* ===== Campos permitidos a actualizar ===== */
  $fieldsSpec = [
    // nombre_campo => tipo ('s' string, 'i' int, 't' time string) – quedan como 's' para bind
    'estado'         => 's',
    'valor'          => 's',
    'id_unidad'      => 'i',
    'resistencia'    => 's',
    'vida'           => 's',
    'foto'           => 's',
    'accion'         => 's',
    'observacion'    => 's',
    'tiempo'         => 's',
    'recomendacion'  => 's',
    'hora_fin'       => 's', // HH:MM o HH:MM:SS
  ];

  $sets   = [];
  $vals   = [];
  $types  = '';

  foreach ($fieldsSpec as $campo => $tip) {
    if (array_key_exists($campo, $in)) {
      $v = $S($campo, null);
      // Normaliza strings vacíos a NULL
      if ($v === '' || $v === 'null') { $v = null; }
      // Para id_unidad, tipo entero (permitimos null)
      if ($campo === 'id_unidad') {
        $v = $I('id_unidad', null);
        $types .= 'i';
      } else {
        // hora_fin: normaliza HH:MM -> HH:MM:00
        if ($campo === 'hora_fin' && is_string($v) && strlen($v) === 5) { $v .= ':00'; }
        $types .= 's';
      }
      $sets[] = "$campo = ?";
      $vals[] = $v;
    }
  }

  if (empty($sets)) {
    http_response_code(400);
    echo json_encode([
      'success'=>false,
      'message'=>'No se enviaron campos válidos para actualizar'
    ]);
    exit;
  }

  $sql = "UPDATE tb_bateria SET ".implode(', ', $sets)." WHERE idbateria = ?";
  $types .= 'i';
  $vals[] = $idbateria;

  $st = $db->prepare($sql);
  // bind_param requiere referencias
  $bind = [$types];
  foreach ($vals as $i => $v) { $bind[] = &$vals[$i]; }
  call_user_func_array([$st, 'bind_param'], $bind);
  $st->execute();
  $affected = $st->affected_rows;
  $st->close();

  $resp = [
    'success'   => true,
    'message'   => $affected > 0 ? 'Registro actualizado' : 'Sin cambios (valores idénticos)',
    'idbateria' => $idbateria,
    'id_datadai'=> $id_datadai,
    'chasis'    => $chasis,
    'updated'   => array_keys(array_intersect_key($fieldsSpec, $in)),
  ];
  if ($debug) { $resp['_debug'] = ['sql'=>$sql, 'types'=>$types, 'vals'=>$vals]; }
  echo json_encode($resp, JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'success'=>false,
    'message'=>'No se pudo actualizar',
    'error'=>$e->getMessage()
  ], JSON_UNESCAPED_UNICODE);
}
