<?php
// bateria_acciones.php — Lista registros de tb_bateria con acciones específicas
// Métodos: GET, OPTIONS
//
// Parámetros GET opcionales:
//   - chasis, marca, modelo, color           (string; exact=1 -> igualdad; exact=0 -> LIKE)
//   - accion                                 (CSV; por defecto: "CARGA LENTA,ENCENDER VEHICULO")
//   - exact=0|1                              (default 0 -> LIKE)
//   - limit=50                               (1..500)
//   - offset=0                               (>=0)
//   - order                                  ("fecha_desc" | "fecha_asc" | "id_desc" | "id_asc") — default fecha_desc
//   - _debug=1                               (incluye SQL y bindings devueltos)
// Respuesta:
//   { success, total, limit, offset, rows: [ { chasis, marca, modelo, color, idbateria, accion, tiempo, fecha, hora } ] }

declare(strict_types=1);
session_start();
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS / CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== CONEXIÓN ===== */
require_once __DIR__ . '/../conex.php';
if (!isset($conn) || !$conn) {
  echo json_encode(['success'=>false,'message'=>'Error de conexión']); exit;
}

try {
  if (function_exists('mysqli_report')) { mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); }

  /* ===== ENTRADAS ===== */
  $chasis = isset($_GET['chasis']) ? trim((string)$_GET['chasis']) : '';
  $marca  = isset($_GET['marca'])  ? trim((string)$_GET['marca'])  : '';
  $modelo = isset($_GET['modelo']) ? trim((string)$_GET['modelo']) : '';
  $color  = isset($_GET['color'])  ? trim((string)$_GET['color'])  : '';

  $exact  = isset($_GET['exact']) ? (int)$_GET['exact'] : 0;
  $limit  = isset($_GET['limit']) ? (int)$_GET['limit'] : 50;
  $offset = isset($_GET['offset'])? (int)$_GET['offset']: 0;
  $order  = isset($_GET['order']) ? trim((string)$_GET['order']) : 'fecha_desc';
  $debug  = isset($_GET['_debug'])? (int)$_GET['_debug']: 0;

  // Acciones por defecto
  $accionesDefault = ['CARGA LENTA','ENCENDER VEHICULO'];
  if (!empty($_GET['accion'])) {
    // CSV -> arreglo, se limpia y se ignoran vacíos
    $acciones = array_filter(array_map(function($v){ return trim((string)$v); }, explode(',', (string)$_GET['accion'])));
  } else {
    $acciones = $accionesDefault;
  }
  // Normaliza límite
  if ($limit < 1) $limit = 50;
  if ($limit > 500) $limit = 500;
  if ($offset < 0) $offset = 0;

  // Orden
  switch ($order) {
    case 'fecha_asc': $orderBy = 'b.fecha ASC, b.hora ASC, b.idbateria ASC'; break;
    case 'id_asc':    $orderBy = 'b.idbateria ASC'; break;
    case 'id_desc':   $orderBy = 'b.idbateria DESC'; break;
    default:          $orderBy = 'b.fecha DESC, b.hora DESC, b.idbateria DESC'; // fecha_desc
  }

  /* ===== WHERE dinámico ===== */
  $where   = [];
  $params  = [];
  $types   = '';

  // Filtro condición siempre activo: b.condicion = 1
  $where[] = 'b.condicion = 1';

  // Acciones: (b.accion IN (?, ?, ...))
  if (count($acciones) > 0) {
    $in = implode(',', array_fill(0, count($acciones), '?'));
    $where[] = "b.accion IN ($in)";
    foreach ($acciones as $a) { $params[] = $a; $types .= 's'; }
  }

  // Filtros por vehículo (d.*)
  $like = ($exact === 1) ? '=' : 'LIKE';
  $wrap = function(string $v) use ($exact) { return ($exact === 1) ? $v : "%$v%"; };

  if ($chasis !== '') { $where[] = "d.chasis $like ?"; $params[] = $wrap($chasis); $types .= 's'; }
  if ($marca  !== '') { $where[] = "d.marca  $like ?"; $params[] = $wrap($marca);  $types .= 's'; }
  if ($modelo !== '') { $where[] = "d.modelo $like ?"; $params[] = $wrap($modelo); $types .= 's'; }
  if ($color  !== '') { $where[] = "d.color  $like ?"; $params[] = $wrap($color);  $types .= 's'; }

  // WHERE final (con precedencia correcta):
  // b.condicion = 1 AND b.accion IN (...) [AND d.* filtros]
  $whereSql = count($where) ? ('WHERE '.implode(' AND ', $where)) : '';

  /* ===== SQL base ===== */
  $sqlBase = "
    FROM tb_bateria AS b
    INNER JOIN tb_datadai AS d ON d.iddatadai = b.id_datadai
    $whereSql
  ";

  /* ===== TOTAL ===== */
  $sqlCount = "SELECT COUNT(*) AS total $sqlBase";
  $stmtCnt = mysqli_prepare($conn, $sqlCount);
  if ($types !== '') { mysqli_stmt_bind_param($stmtCnt, $types, ...$params); }
  mysqli_stmt_execute($stmtCnt);
  $resCnt = mysqli_stmt_get_result($stmtCnt);
  $rowCnt = mysqli_fetch_assoc($resCnt);
  $total  = (int)($rowCnt['total'] ?? 0);
  mysqli_stmt_close($stmtCnt);

  /* ===== SELECT con paginación ===== */
  $sql = "
    SELECT
      d.chasis, d.marca, d.modelo, d.color,
      b.idbateria, b.accion, b.tiempo,
      b.fecha, b.hora
    $sqlBase
    ORDER BY $orderBy
    LIMIT ? OFFSET ?
  ";

  // Bind + ejecución
  $stmt = mysqli_prepare($conn, $sql);
  $types2 = $types . 'ii';
  $params2 = $params;
  $params2[] = $limit;
  $params2[] = $offset;
  mysqli_stmt_bind_param($stmt, $types2, ...$params2);
  mysqli_stmt_execute($stmt);
  $result = mysqli_stmt_get_result($stmt);

  $rows = [];
  while ($r = mysqli_fetch_assoc($result)) { $rows[] = $r; }
  mysqli_stmt_close($stmt);

  $out = [
    'success' => true,
    'total'   => $total,
    'limit'   => $limit,
    'offset'  => $offset,
    'rows'    => $rows
  ];

  if ($debug === 1) {
    $out['_debug'] = [
      'where' => $where,
      'types' => $types,
      'params'=> $params,
      'sql_count' => $sqlCount,
      'sql_list'  => $sql
    ];
  }

  echo json_encode($out, JSON_UNESCAPED_UNICODE);
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'success' => false,
    'message' => 'Error inesperado',
    'error'   => $e->getMessage()
  ], JSON_UNESCAPED_UNICODE);
}
