<?php
/**
 * CREA en tb_asignar con:
 * - status = "COORDINADO" (fijo)
 * - SIN campo grupo
 * - fecha y fecha_descarga = NOW() (America/Guayaquil)
 * - compania/destino = INT (0 -> NULL)
 * - localidad = NULL (no se usa)
 * - numero_doc: si "" (cadena vacía) -> NULL en BD
 *
 * Tabla: tb_asignar(
 *  id, transportista(INT), localidad, destino(INT), status, fecha, numero_doc, conductor, placa,
 *  fecha_carga, fecha_descarga, tipo, guia, condicion, observaciones
 * )
 */

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== Encabezados ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== Errores controlados ===== */
ini_set('display_errors', '0');
mysqli_report(MYSQLI_REPORT_OFF);
set_error_handler(function($n,$s,$f,$l){
  http_response_code(400);
  echo json_encode(['success'=>false,'stage'=>'php_error','message'=>"PHP:$n $s @ $f:$l"], JSON_UNESCAPED_UNICODE);
  exit;
});
set_exception_handler(function($e){
  http_response_code(400);
  echo json_encode(['success'=>false,'stage'=>'exception','message'=>$e->getMessage()], JSON_UNESCAPED_UNICODE);
  exit;
});

/* ===== Conexión ===== */
$pathConex = realpath(__DIR__.'/../conex.php') ?: realpath(__DIR__.'/../../conex.php') ?: null;
if (!$pathConex) { http_response_code(500); echo json_encode(['success'=>false,'stage'=>'conn','message'=>'No se encontró conex.php']); exit; }
require_once $pathConex;

$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) { http_response_code(500); echo json_encode(['success'=>false,'stage'=>'conn','message'=>'Sin conexión BD']); exit; }
mysqli_set_charset($db, 'utf8mb4');

/* Fija zona horaria en MySQL (corregido) */
@$db->query("SET time_zone='-05:00'"); // Guayaquil

/* ===== Helpers ===== */
function body(): array {
  $raw = file_get_contents('php://input');
  $j = json_decode($raw ?: '[]', true);
  return is_array($j) ? $j : [];
}
function v($x){ return is_string($x) ? trim($x) : $x; }
function param($keys,$default=null){
  $IN = $GLOBALS['_IN'];
  $ks = is_array($keys)?$keys:[$keys];
  foreach($ks as $k){
    if (is_array($IN) && array_key_exists($k,$IN)) return $IN[$k];
    if (isset($_POST[$k])) return $_POST[$k];
    if (isset($_GET[$k]))  return $_GET[$k];
  }
  return $default;
}
function normalizeDateTime(?string $v): ?string {
  $v = is_null($v) ? null : trim($v ?? '');
  if ($v === null || $v === '') return null;
  $c=[];
  if(preg_match('/^\d{4}-\d{2}-\d{2}$/',$v)) $c[]=$v.' 00:00:00';
  if(preg_match('/^\d{4}-\d{2}-\d{2}\s+\d{2}:\d{2}$/',$v)) $c[]=$v.=':00';
  if(preg_match('/^\d{4}-\d{2}-\d{2}\s+\d{2}:\d{2}:\d{2}$/',$v)) $c[]=$v;
  if(preg_match('/^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}$/',$v)) $c[]=str_replace('T',' ',$v).':00';
  if(preg_match('/^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}$/',$v)) $c[]=str_replace('T',' ',$v);
  foreach($c as $x){
    $dt=DateTime::createFromFormat('Y-m-d H:i:s',$x);
    if($dt) return $dt->format('Y-m-d H:i:s');
  }
  return null;
}

/* ===== Entrada ===== */
$_IN = body();

/* ===== Campos ===== */
// INT (0 => NULL con NULLIF)
$transportista_id = (int) param(['compania','transportista','id_transportista'], 0);
$destino_id       = (int) param(['destino','id_destino'], 0);

// Strings
$numero_doc    = v((string)param(['numero_doc','cedula','documento'], '')); // "" -> NULL (NULLIF)
$conductor     = v((string)param(['nombre','conductor'], ''));
$placa         = v((string)param('placa', ''));
$tipo          = v((string)param(['tipo','tipo_transporte'], ''));
$guia          = v((string)param(['guia','guias'], ''));
$observaciones = v((string)param(['observaciones','observacion'], ''));

// Fechas (NOW Guayaquil)
$status       = 'ARRIBADO';
$ahora        = (new DateTime('now', new DateTimeZone('America/Guayaquil')))->format('Y-m-d H:i:s');
$fecha        = $ahora;          // siempre NOW()
$fecha_desc   = $ahora;          // siempre NOW()
$fecha_carga  = normalizeDateTime((string)param(['fecha_carga','fecha_carga_local'], '')); // opcional

/* ===== Validación mínima ===== */
$missing = [];
if ($placa === '')       $missing[] = 'placa';
if ($conductor === '')   $missing[] = 'nombre|conductor';
if (!empty($missing)) {
  echo json_encode(['success'=>false,'stage'=>'validate','message'=>'Faltan campos obligatorios','missing'=>$missing], JSON_UNESCAPED_UNICODE);
  exit;
}

/* ===== INSERT =====
 * - localidad = NULL
 * - status literal 'COORDINADO'
 * - numero_doc usa NULLIF(?, '')  -> "" se guarda como NULL
 * - condicion = 1
 * Placeholders: 11 (2 int + 9 string)  ✅
 */
$sql = "INSERT INTO tb_asignar
(transportista, localidad, destino, status, fecha, numero_doc, conductor, placa,
 fecha_carga, fecha_descarga, tipo, guia, condicion, observaciones)
VALUES
(NULLIF(?,0), NULL, NULLIF(?,0), 'ARRIBADO', ?, NULLIF(?, ''), ?, ?, ?, ?, ?, ?, 1, ?)";

$stmt = $db->prepare($sql);
if (!$stmt) { http_response_code(500); echo json_encode(['success'=>false,'stage'=>'prepare','message'=>$db->error]); exit; }

/* Tipos: 2 ints + 9 strings = 11 */
$ok = $stmt->bind_param(
  'iisssssssss',
  $transportista_id, $destino_id, $fecha, $numero_doc, $conductor, $placa,
  $fecha_carga, $fecha_desc, $tipo, $guia, $observaciones
);
if (!$ok) { http_response_code(500); echo json_encode(['success'=>false,'stage'=>'bind','message'=>$stmt->error]); exit; }

if (!$stmt->execute()) {
  http_response_code(500);
  echo json_encode(['success'=>false,'stage'=>'execute','message'=>$stmt->error]); exit;
}
$id = $stmt->insert_id;
$stmt->close();

/* ===== Respuesta ===== */
echo json_encode([
  'success'    => true,
  'message'    => 'Asignación creada',
  'id'         => $id,
  'normalized' => [
    'transportista_id' => $transportista_id ?: null,
    'destino_id'       => $destino_id ?: null,
    'status'           => $status,
    'fecha'            => $fecha,          // NOW Guayaquil
    'numero_doc'       => ($numero_doc === '') ? null : $numero_doc,
    'conductor'        => $conductor,
    'placa'            => $placa,
    'fecha_carga'      => $fecha_carga,    // opcional
    'fecha_descarga'   => $fecha_desc,     // NOW Guayaquil
    'tipo'             => $tipo,
    'guia'             => $guia,
    'observaciones'    => $observaciones
  ]
], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
