<?php
// views/pages/api_app/asignar_anden_create.php
// Crea asignación de espacio (andén) y marca el andén como ocupado (activo=1).
// Tablas:
//   tb_datadai_anden(id, id_datadai, id_anden, fecha_i, fecha_f, dia, condicion)
//   tb_anden(id, id_ubicacion, etiqueta, activo, condicion)
//
// Body (POST JSON o form-data):
//   - id_datadai (int)  [requerido]
//   - id_anden   (int)  [requerido]
//   - fecha_i    (opcional: 'YYYY-mm-dd[ HH:ii[:ss]]', default NOW Guayaquil)
//   - fecha_f    (opcional: 'YYYY-mm-dd[ HH:ii[:ss]]')
//   - dia        (opcional: int >= 1; si mandas fecha_f se recalcula en servidor)
//   - _debug     (opcional: '1')

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS / CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'stage'=>'method','message'=>'Método no permitido; use POST']); exit;
}

/* ===== Conexión ===== */
$pathConex = realpath(__DIR__.'/../conex.php') ?: realpath(__DIR__.'/../../conex.php') ?: realpath(__DIR__.'/conex.php');
if (!$pathConex) { http_response_code(500); echo json_encode(['success'=>false,'stage'=>'conn','message'=>'No se encontró conex.php']); exit; }
require_once $pathConex;

$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) { http_response_code(500); echo json_encode(['success'=>false,'stage'=>'conn','message'=>'Sin conexión a BD']); exit; }
if (function_exists('mysqli_report')) { mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); }
$db->set_charset('utf8mb4');
@$db->query("SET time_zone='-05:00'"); // Guayaquil

/* ===== Helpers ===== */
function fail(int $code, string $msg, array $extra = []): void {
  http_response_code($code);
  echo json_encode(['success'=>false,'stage'=>'validate','message'=>$msg,'data'=>$extra], JSON_UNESCAPED_UNICODE); exit;
}
function normDT(?string $v, DateTimeZone $tz): ?DateTime {
  if (!$v) return null;
  $v = trim($v);
  $cands = [$v, str_replace('T',' ',$v)];
  foreach ($cands as $x) {
    foreach (['Y-m-d H:i:s','Y-m-d H:i','Y-m-d'] as $fmt) {
      $dt = DateTime::createFromFormat($fmt, $x, $tz);
      if ($dt instanceof DateTime) {
        if ($fmt === 'Y-m-d') { $dt->setTime(0,0,0); }
        return $dt;
      }
    }
  }
  return null;
}

/* ===== INPUT ===== */
$raw = file_get_contents('php://input') ?: '';
$in  = json_decode($raw, true);
if (!is_array($in)) { $in = $_POST; }

$id_datadai = isset($in['id_datadai']) ? (int)$in['id_datadai'] : 0;
$id_anden   = isset($in['id_anden'])   ? (int)$in['id_anden']   : 0;
$fecha_i_in = isset($in['fecha_i'])    ? (string)$in['fecha_i'] : '';
$fecha_f_in = isset($in['fecha_f'])    ? (string)$in['fecha_f'] : '';
$dia_in     = isset($in['dia'])        ? (int)$in['dia']        : null;
$debug      = (string)($in['_debug'] ?? '0') === '1';

if ($id_datadai <= 0) fail(422, 'id_datadai requerido');
if ($id_anden   <= 0) fail(422, 'id_anden requerido');

$tz = new DateTimeZone('America/Guayaquil');
$now = new DateTime('now', $tz);

$dt_i = $fecha_i_in !== '' ? normDT($fecha_i_in, $tz) : clone $now;
if (!$dt_i) fail(422, 'fecha_i con formato inválido');

$dt_f = $fecha_f_in !== '' ? normDT($fecha_f_in, $tz) : null;
if ($dt_f && $dt_f < $dt_i) fail(422, 'fecha_f no puede ser menor que fecha_i');

$dia = null;
if ($dt_f) {
  // Días naturales: DATEDIFF + 1, mínimo 1
  $date_i = DateTime::createFromFormat('Y-m-d', $dt_i->format('Y-m-d'), $tz);
  $date_f = DateTime::createFromFormat('Y-m-d', $dt_f->format('Y-m-d'), $tz);
  $diff   = (int)$date_i->diff($date_f)->days + 1;
  $dia    = max(1, $diff);
} elseif ($dia_in !== null) {
  if ($dia_in < 1) fail(422, 'dia debe ser >= 1');
  $dia = $dia_in;
}

/* ===== Transacción ===== */
$db->begin_transaction();
try {
  // 1) Lock del andén para asegurar disponibilidad
  $sqlAnden = "SELECT id, activo, condicion FROM tb_anden WHERE id = ? FOR UPDATE";
  $stA = $db->prepare($sqlAnden);
  $stA->bind_param('i', $id_anden);
  $stA->execute();
  $resA = $stA->get_result();
  $rowA = $resA->fetch_assoc();
  $stA->close();

  if (!$rowA) { throw new RuntimeException('Andén no existe'); }
  if ((int)$rowA['condicion'] !== 1) { throw new RuntimeException('Andén no disponible (condicion <> 1)'); }
  if ((int)$rowA['activo'] === 1) { throw new RuntimeException('Andén ya está ocupado (activo=1)'); }

  // 2) Evitar doble asignación activa (mientras no tenga fecha_f y condicion=1)
  $sqlDup = "SELECT 1 FROM tb_datadai_anden WHERE id_anden = ? AND condicion = 1 AND (fecha_f IS NULL OR fecha_f >= ?) LIMIT 1";
  $stD = $db->prepare($sqlDup);
  $fecha_i_sql = $dt_i->format('Y-m-d H:i:s');
  $stD->bind_param('is', $id_anden, $fecha_i_sql);
  $stD->execute();
  $stD->store_result();
  $busy = $stD->num_rows > 0;
  $stD->close();
  if ($busy) { throw new RuntimeException('El andén tiene una asignación activa/solapada'); }

  // 3) Insert en tb_datadai_anden (condicion=1)
  $sqlIns = "INSERT INTO tb_datadai_anden (id_datadai, id_anden, fecha_i, fecha_f, dia, condicion)
             VALUES (?, ?, ?, ?, ?, 1)";
  $stI = $db->prepare($sqlIns);
  $fecha_f_sql = $dt_f ? $dt_f->format('Y-m-d H:i:s') : null;
  // bind types: i i s s i  (fecha_f puede ser NULL -> usar 's' con null)
  $dia_val = $dia ?? null;
  $stI->bind_param('iissi', $id_datadai, $id_anden, $fecha_i_sql, $fecha_f_sql, $dia_val);
  $stI->execute();
  $new_id = $db->insert_id;
  $stI->close();

  // 4) Marcar andén como activo
  $sqlUpd = "UPDATE tb_anden SET activo = 1 WHERE id = ? AND activo <> 1";
  $stU = $db->prepare($sqlUpd);
  $stU->bind_param('i', $id_anden);
  $stU->execute();
  $anden_changed = $stU->affected_rows > 0;
  $stU->close();

  $db->commit();

  echo json_encode([
    'success' => true,
    'message' => 'Asignación de andén creada y andén marcado como ocupado',
    'data' => [
      'id_datadai_anden' => (int)$new_id,
      'id_datadai'       => (int)$id_datadai,
      'id_anden'         => (int)$id_anden,
      'fecha_i'          => $fecha_i_sql,
      'fecha_f'          => $fecha_f_sql,
      'dia'              => $dia_val,
      'anden_ocupado'    => $anden_changed ? 1 : (int)$rowA['activo'],
    ],
    '_debug' => $debug ? [
      'fecha_i_src' => $fecha_i_in ?: 'NOW',
      'fecha_f_src' => $fecha_f_in ?: null,
    ] : null
  ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

} catch (Throwable $e) {
  $db->rollback();
  http_response_code(409);
  echo json_encode(['success'=>false,'stage'=>'transaction','message'=>$e->getMessage()], JSON_UNESCAPED_UNICODE);
}
