<?php
// views/pages/api_app/anden_move.php (HOTFIX 500)
// Mueve un vehículo a otro andén en una sola transacción.
// - Cierra asignación abierta (condicion=1, fecha_f IS NULL) con hora Ecuador (PHP).
// - Libera andén origen (activo=0).
// - Valida andén destino (condicion=1 y activo != 1).
// - Crea nueva asignación en destino (fecha_i hora Ecuador) y pone activo=1.
// Debug con ?_debug=1

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');
$php_now = date('Y-m-d H:i:s T');
/* ==== DEPURACIÓN (_debug=1) ==== */
$__DBG = (isset($_GET['_debug']) && $_GET['_debug'] == '1') || (isset($_POST['_debug']) && $_POST['_debug'] == '1');
if ($__DBG) {
  ini_set('display_errors', '1');
  ini_set('display_startup_errors', '1');
  error_reporting(E_ALL);
  header('Content-Type: application/json; charset=utf-8');
  register_shutdown_function(function () {
    $e = error_get_last();
    if ($e && in_array($e['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
      http_response_code(500);
      echo json_encode(['success'=>false,'debug_fatal'=>$e], JSON_UNESCAPED_UNICODE);
    }
  });
}
/* ==== FIN DEPURACIÓN ==== */

//date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS / CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== Localizar conex.php con rutas comunes ===== */
$__paths = [
  __DIR__ . '/../conex.php',
  __DIR__ . '/../../conex.php',
  dirname(__DIR__, 2) . '/conex.php',
  dirname(__DIR__, 3) . '/conex.php',
];
$__found = false;
foreach ($__paths as $__p) { if (file_exists($__p)) { require_once $__p; $__found = true; break; } }
if (!$__found) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'No se encontró conex.php','tried'=>$__paths], JSON_UNESCAPED_UNICODE);
  exit;
}

/* ===== CONEXIÓN ===== */
$mysqli = $conn ?? ($conex ?? null);
if (!$mysqli instanceof mysqli) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Conexión MySQL inválida (mysqli no instanciado)']); exit;
}
mysqli_set_charset($mysqli, 'utf8mb4');

/* IMPORTANTE: desactivar reporte estricto antes de SET time_zone para evitar excepción */
if (function_exists('mysqli_report')) { mysqli_report(MYSQLI_REPORT_OFF); }
try {
  $mysqli->query("SET time_zone = 'America/Guayaquil'");
} catch (Throwable $e) {
  try { $mysqli->query("SET time_zone = '-05:00'"); } catch (Throwable $e2) { /* ignorar */ }
}
/* Re-activar modo estricto una vez seteada la zona */
if (function_exists('mysqli_report')) { mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); }

/* ===== INPUT ===== */
function inData(): array {
  $ct = $_SERVER['CONTENT_TYPE'] ?? '';
  if ($_SERVER['REQUEST_METHOD'] === 'POST' && stripos((string)$ct, 'application/json') !== false) {
    $raw = file_get_contents('php://input');
    if ($raw !== false && strlen(trim($raw)) > 0) {
      $data = json_decode($raw, true);
      if (json_last_error() === JSON_ERROR_NONE && is_array($data)) return $data;
    }
  }
  if (!empty($_POST)) return $_POST;
  return $_GET;
}
$in = inData();

$id_anden_dest = isset($in['id_anden_dest']) ? (int)$in['id_anden_dest'] : 0;
$id_datadai    = isset($in['id_datadai']) ? (int)$in['id_datadai'] : 0;
$chasis        = isset($in['chasis']) ? trim((string)$in['chasis']) : '';

if ($id_anden_dest <= 0) {
  http_response_code(400);
  echo json_encode(['success'=>false,'message'=>'Parámetro requerido: id_anden_dest > 0']); exit;
}

/* ===== HELPERS ===== */
function getVehiculoByChasis(mysqli $db, string $chasis): ?array {
  $sql = "SELECT iddatadai, chasis FROM tb_datadai
          WHERE UPPER(TRIM(chasis)) = UPPER(TRIM(?))
          ORDER BY iddatadai DESC
          LIMIT 1";
  $st = $db->prepare($sql); $st->bind_param('s', $chasis); $st->execute();
  $rs = $st->get_result(); $row = $rs->fetch_assoc(); $st->close();
  return $row ?: null;
}
function getAsignacionAbiertaByIdDat(mysqli $db, int $id_datadai): ?array {
  $sql = "SELECT id, id_datadai, id_anden, fecha_i, fecha_f, dia, condicion
          FROM tb_datadai_anden
          WHERE id_datadai = ? AND condicion = 1 AND fecha_f IS NULL
          ORDER BY id DESC
          LIMIT 1";
  $st = $db->prepare($sql); $st->bind_param('i', $id_datadai); $st->execute();
  $rs = $st->get_result(); $row = $rs->fetch_assoc(); $st->close();
  return $row ?: null;
}
function lockAnden(mysqli $db, int $id_anden): ?array {
  $sql = "SELECT id, id_ubicacion, etiqueta, activo, condicion
          FROM tb_anden
          WHERE id = ?
          FOR UPDATE";
  $st = $db->prepare($sql); $st->bind_param('i', $id_anden); $st->execute();
  $rs = $st->get_result(); $row = $rs->fetch_assoc(); $st->close();
  return $row ?: null;
}

/* ===== Hora Ecuador en PHP ===== */
$tzEc    = new DateTimeZone('America/Guayaquil');
$nowEcDT = new DateTime('now', $tzEc);
$nowEc   = $nowEcDT->format('Y-m-d H:i:s');

try {
  // 0) Resolver id_datadai
  if ($id_datadai <= 0) {
    if ($chasis === '') throw new RuntimeException('Debes enviar id_datadai o chasis.');
    $veh = getVehiculoByChasis($mysqli, $chasis);
    if (!$veh) throw new RuntimeException('No se encontró id_datadai para el chasis indicado.');
    $id_datadai = (int)$veh['iddatadai'];
  }

  $mysqli->begin_transaction();

  // 1) Asignación abierta (origen) si existe
  $asig = getAsignacionAbiertaByIdDat($mysqli, $id_datadai);
  $origenAnden = null;
  if ($asig) {
    $origenAnden = lockAnden($mysqli, (int)$asig['id_anden']);
    if (!$origenAnden) throw new RuntimeException('El andén origen de la asignación ya no existe.');
  }

  // 2) Lock del andén destino y validación
  $destAnden = lockAnden($mysqli, $id_anden_dest);
  if (!$destAnden) throw new RuntimeException('El andén destino no existe.');
  if ((int)$destAnden['condicion'] !== 1) throw new RuntimeException('El andén destino no está habilitado (condicion ≠ 1).');
  if ((int)$destAnden['activo'] === 1)   throw new RuntimeException('El andén destino ya está activo/ocupado.');
  if ($asig && (int)$asig['id_anden'] === (int)$destAnden['id']) {
    throw new RuntimeException('El vehículo ya se encuentra asignado a ese mismo andén.');
  }

  // 3) Si hay asignación abierta: cerrar y liberar origen
  $cerrada = null;
  if ($asig) {
    $fi = DateTime::createFromFormat('Y-m-d H:i:s', $asig['fecha_i'], $tzEc);
    if (!$fi) { $fi = DateTime::createFromFormat('Y-m-d H:i:s', $asig['fecha_i']); if ($fi) $fi->setTimezone($tzEc); }
    $diffDays = null;
    if ($fi) {
      $fiEc = (clone $fi)->setTimezone($tzEc);
      $interval = $fiEc->diff($nowEcDT);
      $diffDays = (int)$interval->days; // días completos
    }
    $sqlClose = "UPDATE tb_datadai_anden SET fecha_f = ?, dia = ? WHERE id = ?";
    $st = $mysqli->prepare($sqlClose);
    if ($diffDays === null) { $null = null; $st->bind_param('sii', $nowEc, $null, $asig['id']); }
    else { $st->bind_param('sii', $nowEc, $diffDays, $asig['id']); }
    $st->execute(); $st->close();

    if ((int)$origenAnden['activo'] !== 0) {
      $sqlFree = "UPDATE tb_anden SET activo = 0 WHERE id = ?";
      $st = $mysqli->prepare($sqlFree); $st->bind_param('i', $origenAnden['id']); $st->execute(); $st->close();
      $origenAnden['activo'] = 0;
    }

    $sqlGet = "SELECT id, id_datadai, id_anden, fecha_i, fecha_f, dia, condicion
               FROM tb_datadai_anden WHERE id = ?";
    $st = $mysqli->prepare($sqlGet); $st->bind_param('i', $asig['id']); $st->execute();
    $rs = $st->get_result(); $cerrada = $rs->fetch_assoc(); $st->close();
  }

  // 4) Nueva asignación en destino + activar destino
  $sqlIns = "INSERT INTO tb_datadai_anden (id_datadai, id_anden, fecha_i, condicion)
             VALUES (?, ?, ?, 1)";
  $st = $mysqli->prepare($sqlIns); $st->bind_param('iis', $id_datadai, $id_anden_dest, $nowEc);
  $st->execute(); $newAssignId = $st->insert_id; $st->close();

  $sqlSetDest = "UPDATE tb_anden SET activo = 1 WHERE id = ?";
  $st = $mysqli->prepare($sqlSetDest); $st->bind_param('i', $id_anden_dest); $st->execute(); $st->close();
  $destAnden['activo'] = 1;

  $sqlGetNew = "SELECT id, id_datadai, id_anden, fecha_i, fecha_f, dia, condicion
                FROM tb_datadai_anden WHERE id = ?";
  $st = $mysqli->prepare($sqlGetNew); $st->bind_param('i', $newAssignId); $st->execute();
  $rs = $st->get_result(); $nueva = $rs->fetch_assoc(); $st->close();

  $mysqli->commit();

  $resp = [
    'success' => true,
    'moved'   => true,
    'origen'  => $origenAnden ? [
      'id'        => (int)$origenAnden['id'],
      'etiqueta'  => $origenAnden['etiqueta'] ?? null,
      'activo'    => (int)$origenAnden['activo'],
      'condicion' => isset($origenAnden['condicion']) ? (int)$origenAnden['condicion'] : null
    ] : null,
    'destino' => [
      'id'        => (int)$destAnden['id'],
      'etiqueta'  => $destAnden['etiqueta'] ?? null,
      'activo'    => (int)$destAnden['activo'], // 1
      'condicion' => isset($destAnden['condicion']) ? (int)$destAnden['condicion'] : null
    ],
    'asignacion_cerrada' => $cerrada ? [
      'id'         => (int)$cerrada['id'],
      'id_datadai' => (int)$cerrada['id_datadai'],
      'id_anden'   => (int)$cerrada['id_anden'],
      'fecha_i'    => $cerrada['fecha_i'],
      'fecha_f'    => $cerrada['fecha_f'],
      'dia'        => isset($cerrada['dia']) ? (int)$cerrada['dia'] : null
    ] : null,
    'asignacion_nueva' => $nueva ? [
      'id'         => (int)$nueva['id'],
      'id_datadai' => (int)$nueva['id_datadai'],
      'id_anden'   => (int)$nueva['id_anden'],
      'fecha_i'    => $nueva['fecha_i'],
      'condicion'  => isset($nueva['condicion']) ? (int)$nueva['condicion'] : null
    ] : null,
    'nota' => $asig
      ? 'Movimiento realizado: se cerró la asignación anterior y se creó la nueva en el andén destino.'
      : 'El vehículo no tenía asignación abierta; fue asignado directamente al andén destino.'
  ];

  if ($__DBG) {
    $resp['_debug'] = [
      'php_now_ec' => $nowEc,
      'mysql_now'  => null, 'mysql_tz_session' => null, 'mysql_tz_system' => null
    ];
    try {
      $q = $mysqli->query("SELECT NOW() AS mysql_now, @@session.time_zone AS session_tz, @@system_time_zone AS system_tz");
      if ($q) { $row = $q->fetch_assoc();
        $resp['_debug']['mysql_now'] = $row['mysql_now'] ?? null;
        $resp['_debug']['mysql_tz_session'] = $row['session_tz'] ?? null;
        $resp['_debug']['mysql_tz_system']  = $row['system_tz'] ?? null;
        $q->close();
      }
    } catch (Throwable $e) { $resp['_debug']['mysql_debug_error'] = $e->getMessage(); }
  }

  echo json_encode($resp);

} catch (Throwable $e) {
  try { $mysqli->rollback(); } catch (\Throwable $e2) {}
  http_response_code(400);
  echo json_encode(['success'=>false,'message'=>$e->getMessage()]);
}
