<?php
// views/pages/api_app/anden_libre_por_ubicacion.php
// Lista todos los andenes LIBRES (activo=0, condicion=1) por id_ubicacion
// sin límite de cantidad.
//
// Params (GET/POST):
//   - id_ubicacion : int   (requerido)
//   - q            : str   (opcional; filtra por etiqueta LIKE)
//   - format       : 'json' (default) | 'html' -> salida <option>
//   - _debug       : '1'   (opcional)

declare(strict_types=1);
date_default_timezone_set('America/Guayaquil');

/* ===== HEADERS / CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== Conexión ===== */
$pathConex = realpath(__DIR__.'/../conex.php') ?: realpath(__DIR__.'/../../conex.php') ?: realpath(__DIR__.'/conex.php');
if (!$pathConex) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'No se encontró conex.php']); exit; }
require_once $pathConex;

$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'Sin conexión a BD']); exit; }
if (function_exists('mysqli_report')) mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);
$db->set_charset('utf8mb4');
@$db->query("SET time_zone='-05:00'");

/* ===== Helpers ===== */
function qp(string $k, $def=null) {
  if (isset($_POST[$k])) return $_POST[$k];
  if (isset($_GET[$k]))  return $_GET[$k];
  return $def;
}
function fail(int $code, string $msg, array $extra=[]): void {
  http_response_code($code);
  echo json_encode(['success'=>false,'message'=>$msg,'data'=>$extra], JSON_UNESCAPED_UNICODE);
  exit;
}

/* ===== Input ===== */
$format       = strtolower((string)qp('format','json')) === 'html' ? 'html' : 'json';
$id_ubicacion = (int) qp('id_ubicacion', 0);
$q            = trim((string) qp('q',''));
$debug        = (string) qp('_debug','0') === '1';

if ($id_ubicacion <= 0) {
  fail(422, 'id_ubicacion es requerido');
}

/* ===== WHERE ===== */
$where = "WHERE a.id_ubicacion = ? AND a.condicion = 1";
$params = [$id_ubicacion];
$types  = 'i';

if ($q !== '') {
  $where .= " AND UPPER(a.etiqueta) LIKE UPPER(?)";
  $params[] = '%'.$q.'%';
  $types   .= 's';
}

/* ===== Consulta ===== */
$sql = "SELECT
          a.id,
          a.id_ubicacion,
          a.columna,
          a.fila,
          a.etiqueta,
          a.activo,
          a.condicion
        FROM tb_anden a
        $where
        ORDER BY a.id ASC";
$st = $db->prepare($sql);
$refs = [ &$types ];
foreach ($params as $k => &$v) { $refs[] = &$v; }
call_user_func_array([$st,'bind_param'], $refs);
$st->execute();
$res = $st->get_result();

/* ===== Salida ===== */
if ($format === 'html') {
  header('Content-Type: text/html; charset=utf-8');
  $html = '';
  if ($res->num_rows === 0) {
    echo '<option value="">(Sin andenes libres)</option>';
    exit;
  }
  while ($r = $res->fetch_assoc()) {
    $id   = (int)$r['id'];
    $lbl  = trim((string)$r['etiqueta']);
    if ($lbl === '') {
      $lbl = 'Col '.(string)($r['columna'] ?? '').' - Fila '.(string)($r['fila'] ?? '');
    }
    $text = htmlspecialchars($lbl, ENT_QUOTES, 'UTF-8');
    $html .= '<option value="'.$id.'">'.$text.'</option>';
  }
  $st->close();
  echo $html;
  exit;
}

$items = [];
while ($r = $res->fetch_assoc()) {
  $texto = trim((string)$r['etiqueta']);
  if ($texto === '') {
    $texto = 'Col '.(string)($r['columna'] ?? '').' - Fila '.(string)($r['fila'] ?? '');
  }
  $items[] = [
    'id'           => (int)$r['id'],
    'text'         => $texto,
    'id_ubicacion' => (int)$r['id_ubicacion'],
    'columna'      => $r['columna'],
    'fila'         => $r['fila'],
    'etiqueta'     => (string)($r['etiqueta'] ?? ''),
    'activo'     => (int)($r['activo'] ?? ''),
  ];
}
$st->close();

echo json_encode([
  'success'=>true,
  'total'=>count($items),
  'items'=>$items,
  '_debug'=>$debug ? compact('id_ubicacion','q') : null
], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
