<?php
// api/ubicaciones_por_bodega.php — Ubicaciones por bodega (tolerante a esquema)
// Soporta: u.nombre (opcional), u.activo o u.condicion (uno u otro), filtro por id_bodega y búsqueda.
declare(strict_types=1);
ini_set('display_errors','0');
ini_set('log_errors','1');

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

session_start();
require_once __DIR__ . '/../conex.php';
if (!isset($conn) || !($conn instanceof mysqli)) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Sin conexión a BD']); exit;
}
mysqli_set_charset($conn,'utf8mb4');

/** Verifica si una columna existe en la BD actual */
function column_exists(mysqli $conn, string $table, string $column): bool {
  $sql = "SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS
          WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?
          LIMIT 1";
  $stmt = mysqli_prepare($conn, $sql);
  mysqli_stmt_bind_param($stmt, 'ss', $table, $column);
  mysqli_stmt_execute($stmt);
  mysqli_stmt_store_result($stmt);
  $exists = (mysqli_stmt_num_rows($stmt) > 0);
  mysqli_stmt_close($stmt);
  return $exists;
}

try {
  $id_bodega  = (int)($_GET['id_bodega'] ?? 0);
  $q          = trim((string)($_GET['q'] ?? ''));
  $onlyActive = isset($_GET['onlyActive']) ? (int)$_GET['onlyActive'] : 1; // por defecto 1

  if ($id_bodega <= 0) {
    echo json_encode(['success'=>true,'data'=>[]]); exit;
  }

  // Descubre columnas reales
  $has_nombre    = column_exists($conn, 'tb_inv_ubicacion', 'nombre');
  $has_activo    = column_exists($conn, 'tb_inv_ubicacion', 'activo');
  $has_condicion = column_exists($conn, 'tb_inv_ubicacion', 'condicion');

  // Selección de nombre
  $select_nombre = $has_nombre ? "u.nombre AS ubi_nombre" : "u.codigo AS ubi_nombre";

  // Selección y filtro de estado
  $estado_field = null; $estado_alias = 'estado';
  if     ($has_activo)    { $estado_field = 'u.activo';    $estado_alias = 'activo';    }
  elseif ($has_condicion) { $estado_field = 'u.condicion'; $estado_alias = 'condicion'; }

  $sql = "SELECT
            u.id_ubicacion,
            u.id_bodega,
            u.codigo,
            {$select_nombre},
            ".($estado_field ? "{$estado_field} AS {$estado_alias}" : "1 AS {$estado_alias}").",
            b.nombre AS bodega_nombre
          FROM tb_inv_ubicacion u
          JOIN tb_inv_bodega b ON b.id_bodega = u.id_bodega
          WHERE u.id_bodega = ?";

  $types = 'i';
  $params = [$id_bodega];

  if ($onlyActive === 1 && $estado_field) {
    $sql .= " AND {$estado_field} = 1";
  }

  if ($q !== '') {
    // Buscar por código o (si existe) nombre; si no existe nombre, sólo por código
    if ($has_nombre) {
      $sql .= " AND (u.codigo LIKE CONCAT('%',?,'%') OR u.nombre LIKE CONCAT('%',?,'%'))";
      $types .= 'ss';
      $params[] = $q; $params[] = $q;
    } else {
      $sql .= " AND (u.codigo LIKE CONCAT('%',?,'%'))";
      $types .= 's';
      $params[] = $q;
    }
  }

  $sql .= " ORDER BY b.nombre ASC, u.codigo ASC";

  $stmt = mysqli_prepare($conn, $sql);
  if (!$stmt) { throw new mysqli_sql_exception(mysqli_error($conn)); }
  mysqli_stmt_bind_param($stmt, $types, ...$params);
  mysqli_stmt_execute($stmt);
  $res = mysqli_stmt_get_result($stmt);

  $data = [];
  while ($r = mysqli_fetch_assoc($res)) {
    $nombreUbi = $r['ubi_nombre'] ?? $r['codigo'];
    $data[] = [
      'id_ubicacion' => (int)$r['id_ubicacion'],
      'id_bodega'    => (int)$r['id_bodega'],
      'codigo'       => (string)$r['codigo'],
      'bodega'       => (string)$r['bodega_nombre'],
      'nombre'       => (string)$nombreUbi,                          // para mostrar en el front
      'text'         => $r['bodega_nombre'].' / '.$nombreUbi,        // útil si se usa en <option>
      'estado'       => isset($r[$estado_alias]) ? (int)$r[$estado_alias] : 1,
    ];
  }
  mysqli_stmt_close($stmt);

  echo json_encode(['success'=>true,'data'=>$data]);
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Error en ubicaciones_por_bodega','error'=>$e->getMessage()]);
}
