<?php
// views/pages/api/requerimiento_guardar.php — Inserta requerimientos en tb_pdi (1 fila por vehículo) + LUGAR
// Además: crea/actualiza lavada en tb_lavada con id_tipolavada = 2 (COMPLETA) y estado_lavado = 'PENDIENTE'
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

session_start();
if (!isset($_SESSION['ingreso']) || $_SESSION['ingreso'] !== 'YES') {
  http_response_code(401);
  echo json_encode(['success'=>false,'message'=>'No autorizado']); exit;
}
$usuario    = (string)($_SESSION['usuario'] ?? 'sistema');
$idUsuario  = (int)($_SESSION['idusuario'] ?? 0); // para tb_lavada

require_once __DIR__ . '/../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) { echo json_encode(['success'=>false,'message'=>'Sin conexión BD']); exit; }

mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone = '-05:00'"); // America/Guayaquil

// ===== Helpers =====
function is_valid_dt(string $dt): bool {
  $dt = str_replace('T', ' ', $dt);
  $d = DateTime::createFromFormat('Y-m-d H:i:s', $dt);
  return $d && $d->format('Y-m-d H:i:s') === $dt;
}
function norm_dt(string $dt): string { return str_replace('T', ' ', $dt); }

/**
 * Aplica lógica de lavada "PENDIENTE" para un id_datadai dado.
 * Emula la regla de api/lavada_save_strict.php pero usando directamente id_datadai.
 *
 * - Si no existe tb_tipolavada con ese id o está inactiva -> no hace nada.
 * - Si último registro es del mismo tipo:
 *      * Si ya está PENDIENTE -> solo refleja en tb_datadai.estado_lavado (si existe).
 *      * Si está en otro estado -> lo actualiza a PENDIENTE.
 * - Si no hay registro o es otro tipo -> inserta nuevo con PENDIENTE.
 */
function aplicarLavadaPendiente(
  mysqli $db,
  int $idDatadai,
  int $idtipoLavada,
  int $idUsuario,
  bool $hasEstadoDatadai
): void {
  $estado = 'PENDIENTE';

  // Validar tipo de lavada
  $stVal = $db->prepare("SELECT COUNT(*) c FROM tb_tipolavada WHERE idtipo_lavada=? AND condicion=1");
  if (!$stVal) return;
  $stVal->bind_param('i', $idtipoLavada);
  $stVal->execute();
  $c = (int)($stVal->get_result()->fetch_assoc()['c'] ?? 0);
  $stVal->close();
  if ($c === 0) {
    // Tipo no existe o inactivo
    return;
  }

  // Último registro activo para ese id_datadai
  $stLast = $db->prepare("
    SELECT idlavada, id_tipolavada, estado_lavado
    FROM tb_lavada
    WHERE id_datadai=? AND condicion=1
    ORDER BY fecha_registro DESC, idlavada DESC
    LIMIT 1
    FOR UPDATE
  ");
  if (!$stLast) return;

  $stLast->bind_param('i', $idDatadai);
  $stLast->execute();
  $last = $stLast->get_result()->fetch_assoc();
  $stLast->close();

  if ($last && (int)$last['id_tipolavada'] === $idtipoLavada) {
    // Mismo tipo
    if ((string)$last['estado_lavado'] === $estado) {
      // Ya está PENDIENTE -> reflejar en tb_datadai si existe la columna
      if ($hasEstadoDatadai) {
        $stRef = $db->prepare("UPDATE tb_datadai SET estado_lavado=? WHERE iddatadai=?");
        if ($stRef) {
          $stRef->bind_param('si', $estado, $idDatadai);
          $stRef->execute();
          $stRef->close();
        }
      }
      return;
    }

    // Mismo tipo pero en otro estado -> actualizar a PENDIENTE
    $stU = $db->prepare("
      UPDATE tb_lavada
      SET estado_lavado='PENDIENTE',
          fecha_registro=NOW(),
          usuario_update=?,
          fecha_update=NOW()
      WHERE idlavada=?
    ");
    if ($stU) {
      $idlavada = (int)$last['idlavada'];
      $stU->bind_param('ii', $idUsuario, $idlavada);
      $stU->execute();
      $stU->close();
    }

    if ($hasEstadoDatadai) {
      $stRef = $db->prepare("UPDATE tb_datadai SET estado_lavado='PENDIENTE' WHERE iddatadai=?");
      if ($stRef) {
        $stRef->bind_param('i', $idDatadai);
        $stRef->execute();
        $stRef->close();
      }
    }

    return;
  }

  // Tipo distinto o no había registro -> INSERT con PENDIENTE
  $stI = $db->prepare("
    INSERT INTO tb_lavada (id_datadai, fecha_registro, id_tipolavada, id_usuario, estado_lavado, condicion)
    VALUES (?, NOW(), ?, ?, 'PENDIENTE', 1)
  ");
  if ($stI) {
    $stI->bind_param('iii', $idDatadai, $idtipoLavada, $idUsuario);
    $stI->execute();
    $stI->close();
  }

  if ($hasEstadoDatadai) {
    $stRef = $db->prepare("UPDATE tb_datadai SET estado_lavado='PENDIENTE' WHERE iddatadai=?");
    if ($stRef) {
      $stRef->bind_param('i', $idDatadai);
      $stRef->execute();
      $stRef->close();
    }
  }
}

// ===== Entrada =====
$raw = file_get_contents('php://input');
$in  = json_decode($raw, true) ?: [];

$h  = $in['header']    ?? [];
$vs = $in['vehiculos'] ?? [];

// Header
$alistamiento = trim((string)($h['alistamiento'] ?? ''));
$requerido    = trim((string)($h['requerido']    ?? ''));
$lugar        = trim((string)($h['lugar']        ?? ''));
$requerimDT   = norm_dt(trim((string)($h['requerimiento']?? ''))); // 'YYYY-mm-dd HH:ii:ss'
$entregaDT    = norm_dt(trim((string)($h['entrega']      ?? '')));
$prioridad    = strtoupper(trim((string)($h['prioridad'] ?? 'NORMAL')));
$facturado    = trim((string)($h['facturado']    ?? ''));

// ===== Validaciones =====
if (!in_array($alistamiento, ['Con PDI','Sin PDI'], true)) {
  echo json_encode(['success'=>false,'message'=>'Tipo de alistamiento inválido (Con PDI / Sin PDI)']); exit;
}
if ($prioridad !== 'NORMAL' && $prioridad !== 'URGENTE') {
  echo json_encode(['success'=>false,'message'=>'Prioridad inválida (NORMAL / URGENTE)']); exit;
}
if ($requerido === '' || $lugar === '' || $requerimDT === '' || $entregaDT === '' || $facturado === '') {
  echo json_encode(['success'=>false,'message'=>'Faltan campos obligatorios']); exit;
}
if (!is_valid_dt($requerimDT) || !is_valid_dt($entregaDT)) {
  echo json_encode(['success'=>false,'message'=>'Formato de fecha/hora inválido. Use YYYY-mm-dd HH:ii:ss']); exit;
}
if (strtotime($entregaDT) < strtotime($requerimDT)) {
  echo json_encode(['success'=>false,'message'=>'La fecha de entrega no puede ser menor a la de requerimiento']); exit;
}
if (!is_array($vs) || count($vs) === 0) {
  echo json_encode(['success'=>false,'message'=>'Sin vehículos']); exit;
}

// Normalizar vehículos: iddatadai únicos válidos (>0)
$vehiculos = [];
foreach ($vs as $v) {
  $idd = (int)($v['iddatadai'] ?? 0);
  if ($idd > 0) $vehiculos[] = $idd;
}
$vehiculos = array_values(array_unique($vehiculos));
if (!$vehiculos) { echo json_encode(['success'=>false,'message'=>'Vehículos inválidos']); exit; }

// ===== Preparativos Lavada (COMPLETA = idtipo_lavada 2) =====
$idTipoLavadaDefault = 2;
$hasEstadoDatadai    = false;

try {
  // ¿Existe tb_datadai.estado_lavado?
  $chk = $db->prepare("
    SELECT COUNT(*) c
    FROM INFORMATION_SCHEMA.COLUMNS
    WHERE TABLE_SCHEMA = DATABASE()
      AND TABLE_NAME = 'tb_datadai'
      AND COLUMN_NAME = 'estado_lavado'
    LIMIT 1
  ");
  if ($chk) {
    $chk->execute();
    $hasEstadoDatadai = ((int)($chk->get_result()->fetch_assoc()['c'] ?? 0) > 0);
    $chk->close();
  }

  $db->begin_transaction();

  // Validar que existan en tb_datadai
  $ph    = implode(',', array_fill(0, count($vehiculos), '?'));
  $types = str_repeat('i', count($vehiculos));
  $stmtV = $db->prepare("SELECT iddatadai FROM tb_datadai WHERE iddatadai IN ($ph)");
  if (!$stmtV) { throw new RuntimeException('Prep validación tb_datadai: '.$db->error); }
  $stmtV->bind_param($types, ...$vehiculos);
  $stmtV->execute();
  $rsV = $stmtV->get_result();
  $valid = [];
  while ($r = $rsV->fetch_assoc()) { $valid[(int)$r['iddatadai']] = true; }
  $stmtV->close();

  $insertados = 0; $saltados = 0; $noEncontrados = 0; $duplicados = 0;

  // Duplicado por (id_datadai, fecha_req) con condicion=1
  $stmtDupe = $db->prepare("SELECT COUNT(*) c FROM tb_pdi WHERE id_datadai = ? AND fecha_req = ? AND condicion = 1");
  if (!$stmtDupe) { throw new RuntimeException('Prep verificación duplicado: '.$db->error); }

  // INSERT con LUGAR (nuevo campo)
  // Columnas: id_datadai, alistamiento, fecha_req, fecha_ent, requerido_por, lugar, condicion, prioridad, facturado, fecha_reg
  $stmtIns = $db->prepare("
    INSERT INTO tb_pdi
      (id_datadai, alistamiento, fecha_req, fecha_ent, requerido_por, lugar, condicion, prioridad, facturado, fecha_reg)
    VALUES
      (?,?,?,?,?,?,1,?, ?, NOW())
  ");
  if (!$stmtIns) { throw new RuntimeException('Prep insert: '.$db->error); }

  foreach ($vehiculos as $idd) {
    if (empty($valid[$idd])) { $noEncontrados++; continue; }

    // chequear duplicado
    $stmtDupe->bind_param('is', $idd, $requerimDT);
    $stmtDupe->execute();
    $dupeRow = $stmtDupe->get_result()->fetch_assoc();
    $c = (int)($dupeRow['c'] ?? 0);
    if ($c > 0) { $duplicados++; $saltados++; continue; }

    // bind: i s s s s s s s  (8 parámetros antes de NOW())
    $stmtIns->bind_param(
      'isssssss',
      $idd,            // id_datadai
      $alistamiento,   // alistamiento
      $requerimDT,     // fecha_req
      $entregaDT,      // fecha_ent
      $requerido,      // requerido_por
      $lugar,          // lugar
      $prioridad,      // prioridad
      $facturado       // facturado
    );

    if (!$stmtIns->execute()) { throw new RuntimeException('Insert falló: '.$db->error); }
    $insertados++;

    // Lavada COMPLETA (id_tipolavada = 2) en estado 'PENDIENTE'
    if ($idTipoLavadaDefault > 0) {
      aplicarLavadaPendiente($db, $idd, $idTipoLavadaDefault, $idUsuario, $hasEstadoDatadai);
    }
  }

  $stmtDupe->close();
  $stmtIns->close();

  $db->commit();
  echo json_encode([
    'success'        => true,
    'insertados'     => $insertados,
    'saltados'       => $saltados,
    'duplicados'     => $duplicados,
    'no_encontrados' => $noEncontrados,
    'message'        => 'Proceso finalizado'
  ], JSON_UNESCAPED_UNICODE);
} catch (Throwable $e) {
  $db->rollback();
  http_response_code(500);
  echo json_encode(['success'=>false, 'message'=>'Transacción fallida', 'sql_error'=>$e->getMessage()]);
}
