<?php
// views/pages/api/patio_inventario.php
// API para "Inventario de Vehículos en Patio" usando la consulta solicitada.

declare(strict_types=1);
session_start();
header('Content-Type: application/json; charset=utf-8');
date_default_timezone_set('America/Guayaquil');

/* ========= Cargar conexión ========= */
$loaded = false;
foreach ([__DIR__.'/../conex.php', __DIR__.'/../../conex.php', __DIR__.'/../../../conex.php'] as $p) {
  if (is_file($p)) { require_once $p; $loaded = true; break; }
}
if (!$loaded) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'No se encontró conex.php']); exit; }

$db = $conn ?? ($conex ?? null);
if (!$db) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'No hay conexión ($conn/$conex)']); exit; }
mysqli_set_charset($db, 'utf8mb4');

/* ========= Helpers ========= */
function runq(mysqli $db, string $sql) {
  $rs = mysqli_query($db, $sql);
  if (!$rs) {
    http_response_code(500);
    echo json_encode(['success'=>false,'message'=>'Error de base de datos','mysql_error'=>mysqli_error($db),'sql'=>$sql], JSON_UNESCAPED_UNICODE);
    exit;
  }
  return $rs;
}
function tableExists(mysqli $db, string $table): bool {
  $q = "SELECT 1 FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = '".mysqli_real_escape_string($db,$table)."'";
  $rs = mysqli_query($db, $q);
  return ($rs && mysqli_num_rows($rs) > 0);
}
function columnExists(mysqli $db, string $table, string $col): bool {
  $q = "SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = '".mysqli_real_escape_string($db,$table)."' AND COLUMN_NAME = '".mysqli_real_escape_string($db,$col)."'";
  $rs = mysqli_query($db, $q);
  return ($rs && mysqli_num_rows($rs) > 0);
}

/* ========= ONLY BODEGAS ========= */
if (isset($_GET['only_bodegas'])) {
  // Detectar nombre de columna de "nombre bodega"
  $bodegaCol = columnExists($db,'tb_bodega','bodega') ? 'bodega' : (columnExists($db,'tb_bodega','detalle') ? 'detalle' : null);
  if ($bodegaCol === null) {
    http_response_code(500);
    echo json_encode(['success'=>false,'message'=>"tb_bodega no tiene columna 'bodega' ni 'detalle'"]); exit;
  }
  $sql = "SELECT idbodega AS id, $bodegaCol AS nombre FROM tb_bodega WHERE condicion = 1 ORDER BY nombre";
  $rs = runq($db, $sql);
  $out = [];
  while ($r = mysqli_fetch_assoc($rs)) $out[] = $r;
  echo json_encode(['success'=>true,'bodegas'=>$out], JSON_UNESCAPED_UNICODE); exit;
}

/* ========= Parámetros ========= */
$q         = trim((string)($_GET['q'] ?? ''));
$estado    = trim((string)($_GET['estado'] ?? ''));
$bodega_id = trim((string)($_GET['bodega_id'] ?? ''));
$multi     = trim((string)($_GET['multi'] ?? ''));

$multiChasis = [];
if ($multi !== '') {
  foreach (preg_split('/[,\s;]+/', $multi) as $ch) { $ch = trim($ch); if ($ch !== '') $multiChasis[] = $ch; }
  $multiChasis = array_values(array_unique($multiChasis));
}

/* ========= Detecciones de esquema para tu query ========= */
$tbUbicacion = tableExists($db,'tb_ubicacion');
$colUbic     = $tbUbicacion && columnExists($db,'tb_ubicacion','detalle_ubicacion');
$bodegaCol   = columnExists($db,'tb_bodega','detalle') ? 'detalle' : (columnExists($db,'tb_bodega','bodega') ? 'bodega' : null);
if ($bodegaCol === null) { $bodegaCol = 'bodega'; } // fallback (si no existe, MySQL dará error descriptivo)

/* ========= Armar TU QUERY base + filtros ========= */
/* Nota:
   - Se usa MAX(d.fecha_ingreso) y DATEDIFF(...) como pediste.
   - bodega: MAX(b.<bodegaCol>).
   - ubicacion: MAX(u.detalle_ubicacion) SI existe tabla/columna; si no, será NULL/''.
*/
$selectUbic = $colUbic ? "MAX(u.detalle_ubicacion) AS ubicacion," : "'' AS ubicacion,";

$sql = "
  SELECT
    d.chasis,
    MAX(d.marca)                         AS marca,
    MAX(d.modelo)                        AS modelo,
    MAX(d.color)                         AS color,
    MAX(d.motor)                         AS motor,
    MAX(d.fecha_ingreso)                 AS fecha_ingreso,
    DATEDIFF(CURDATE(), MAX(d.fecha_ingreso)) AS dias_transcurridos,
    MAX(b.$bodegaCol)                    AS bodega,
    $selectUbic
    MAX(a.etiqueta)                      AS ultimo_anden,
    COALESCE(MAX(p.pedidos), '')         AS pedidos
  FROM tb_datadai d

  /* ===== ÚLTIMO ANDÉN POR MAX(ID) ===== */
  LEFT JOIN (
    SELECT x.id_datadai, x.id_anden
    FROM tb_datadai_anden x
    JOIN (
      SELECT id_datadai, MAX(id) AS max_id
      FROM tb_datadai_anden
      WHERE condicion = 1
      GROUP BY id_datadai
    ) m
      ON m.id_datadai = x.id_datadai
     AND m.max_id     = x.id
    WHERE x.condicion = 1
  ) lda
    ON lda.id_datadai = d.iddatadai

  LEFT JOIN tb_anden a
    ON a.id = lda.id_anden

  LEFT JOIN tb_bodega b
    ON b.idbodega = d.id_bodega
";

if ($colUbic) {
  $sql .= "
  LEFT JOIN tb_ubicacion u
    ON u.id_bodega = b.idbodega
  ";
}

$sql .= "
  /* ===== PEDIDOS UNIFICADOS POR CHASIS (pre-agregados) ===== */
  LEFT JOIN (
    SELECT t.chasis,
           GROUP_CONCAT(DISTINCT t.ped ORDER BY t.ped SEPARATOR ', ') AS pedidos
    FROM (
      SELECT d1.chasis, d1.pedido AS ped
      FROM tb_datadai d1
      WHERE d1.condicion = 1 AND d1.pedido IS NOT NULL

      UNION ALL

      SELECT d2.chasis, sp.pedido AS ped
      FROM tb_datadai d2
      JOIN tb_solicitudprevia sp
        ON sp.idsolicitudprevia = d2.id_solicitudprevia
      WHERE d2.condicion = 1 AND sp.condicion = 1 AND sp.pedido IS NOT NULL
    ) t
    GROUP BY t.chasis
  ) p
    ON p.chasis = d.chasis

  WHERE d.condicion = 1
";

/* ========= Filtros dinámicos antes del GROUP BY ========= */
$whereExtra = [];

if ($estado !== '') {
  $whereExtra[] = "d.id_estado = ".(int)$estado;
}
if ($bodega_id !== '') {
  // Si prefieres usar la bodega del andén, cambia por COALESCE(a.id_bodega, d.id_bodega)
  $whereExtra[] = "COALESCE(a.id_bodega, d.id_bodega) = ".(int)$bodega_id;
}
if ($q !== '') {
  $qEsc = mysqli_real_escape_string($db, $q);
  $like = "'%$qEsc%'";
  $whereExtra[] = "(d.marca LIKE $like OR d.modelo LIKE $like OR d.color LIKE $like OR d.pedido LIKE $like)";
}
if (!empty($multiChasis)) {
  $vals = array_map(fn($c) => "'".mysqli_real_escape_string($db, $c)."'", $multiChasis);
  $whereExtra[] = "d.chasis IN (".implode(',', $vals).")";
}

if ($whereExtra) { $sql .= " AND ".implode(" AND ", $whereExtra); }

$sql .= "
  GROUP BY d.chasis
  ORDER BY d.chasis
";

/* ========= Ejecutar ========= */
$rs = runq($db, $sql);

/* ========= Formato de salida (DataTable) =========
   Tu <thead> y columnas JS esperan:
   Pedido | Marca | Modelo | Color | Chasis | Motor | Ingreso | Días | Bodega | Ubicación | Andén
*/
$data = [];
while ($r = mysqli_fetch_assoc($rs)) {
  $data[] = [
    'pedido'        => (string)($r['pedidos'] ?? ''),
    'marca'         => (string)($r['marca'] ?? ''),
    'modelo'        => (string)($r['modelo'] ?? ''),
    'color'         => (string)($r['color'] ?? ''),
    'chasis'        => (string)($r['chasis'] ?? ''),
    'motor'         => (string)($r['motor'] ?? ''),
    'fecha_ingreso' => $r['fecha_ingreso'] ?? null,
    'dias'          => isset($r['dias_transcurridos']) ? (int)$r['dias_transcurridos'] : null,
    'bodega'        => (string)($r['bodega'] ?? ''),
    'ubicacion'     => (string)($r['ubicacion'] ?? ''),
    'anden'         => (string)($r['ultimo_anden'] ?? '')
  ];
}

echo json_encode(['success'=>true,'count'=>count($data),'data'=>$data], JSON_UNESCAPED_UNICODE);
