<?php
// api/nacionalizados_guardar_parcial.php — Guardado parcial (DAI 10 / régimen) para ítems seleccionados
declare(strict_types=1);
session_start();

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');

if (!isset($_SESSION['ingreso']) || $_SESSION['ingreso'] !== 'YES') {
  http_response_code(401);
  echo json_encode(['success'=>false,'message'=>'No autorizado']); exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false,'message'=>'Método no permitido']); exit;
}

require_once __DIR__ . '/../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Sin conexión DB']); exit;
}

mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone = '-05:00'");

// ---------- Inputs ----------
$id_usuario = (int)($_SESSION['idusuario'] ?? 0);
$idfactura  = (int)($_POST['idfactura'] ?? 0);
$dai10      = trim((string)($_POST['dai10'] ?? ''));
$regimen    = strtoupper(trim((string)($_POST['regimen'] ?? '')));
$ids        = $_POST['iddatadai'] ?? $_POST['iddatadai[]'] ?? [];

// Normaliza arreglo de ítems (iddatadai[])
$ids = array_values(array_filter(array_map('intval', (array)$ids), fn($v)=>$v>0));

// Validaciones mínimas
if ($idfactura <= 0 || empty($ids)) {
  http_response_code(422);
  echo json_encode(['success'=>false,'message'=>'Parámetros inválidos: factura o ítems vacíos']); exit;
}
if ($dai10 === '') { // según tu UI, DAI10 es requerido
  http_response_code(422);
  echo json_encode(['success'=>false,'message'=>'Ingrese el número de DAI 10']); exit;
}
if ($regimen === '') {
  http_response_code(422);
  echo json_encode(['success'=>false,'message'=>'Seleccione un régimen']); exit;
}

// Mapea texto a lo que guardarás en tb_datadai.dai
// (ajusta si tu columna espera numérico vs texto)
$valorDaiCampo = match ($regimen) {
  '10'     => '10',
  '70'     => '70',
  'SIMPLE' => 'Simple',
  default  => $regimen, // fallback
};

// Mensaje de auditoría
$actividad = match ($regimen) {
  '10'     => 'NACIONALIZACIÓN PARCIAL (DAI 10)',
  '70'     => 'NACIONALIZACIÓN (DAI 70)',
  'SIMPLE' => 'Nueva DAI régimen Simple',
  default  => 'Actualización de DAI/régimen'
};

// Estado a aplicar (ajusta si corresponde)
$estado = 8;

$my_ip        = $_SERVER['REMOTE_ADDR'] ?? '';
date_default_timezone_set('America/Guayaquil');
$fecha_actual = date('Y-m-d');
$hora_actual  = date('H:i:s'); // ¡OJO!: minutos = i (no m)

// Activa excepciones en mysqli para simplificar manejo de errores
mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

try {
  $db->begin_transaction();

  // ------- PREPAREs -------
  // 1) Obtener iddetallefactura por (id_facturainfo, iddatadai)
  $stGetDet = $db->prepare("
    SELECT iddetallefactura
    FROM tb_detallefactura
    WHERE condicion = 1 AND id_facturainfo = ? AND id_datadai = ?
    LIMIT 1
  ");

  // 2) Insertar daisalida
  //    columnas: numeros, fechas, id_facturainfo, id_detallefactura, id_datadai, condicion
  $stInsDaiSalida = $db->prepare("
    INSERT INTO tb_daisalida (numeros, fechas, id_facturainfo, id_detallefactura, id_datadai, condicion)
    VALUES (?, ?, ?, ?, ?, 1)
  ");

  // 3) Actualizar tb_datadai
  $stUpdDatadai = $db->prepare("
    UPDATE tb_datadai
       SET id_estado = ?, dai = ?, id_daisalida = ?
     WHERE iddatadai = ?
  ");

  // 4) Actualizar detalle
  $stUpdDetalle = $db->prepare("
    UPDATE tb_detallefactura
       SET id_estado = ?
     WHERE iddetallefactura = ?
  ");

  // 5) Auditoría
  $stAudit = $db->prepare("
    INSERT INTO tb_auditoria (fecha, hora, ip, detalle, id_estado, id_datadai, id_daisalida, id_usuario)
    VALUES (?, ?, ?, ?, ?, ?, ?, ?)
  ");

  $procesados = 0;
  foreach ($ids as $idd) {
    // Obtener iddetallefactura
    $stGetDet->bind_param('ii', $idfactura, $idd);
    $stGetDet->execute();
    $resDet = $stGetDet->get_result();
    $rowDet = $resDet->fetch_assoc();
    if (!$rowDet) {
      throw new RuntimeException("No se encontró detalle para iddatadai={$idd} en factura {$idfactura}");
    }
    $iddetalle = (int)$rowDet['iddetallefactura'];

    // Insertar daisalida
    $stInsDaiSalida->bind_param('ssiii', $dai10, $fecha_actual, $idfactura, $iddetalle, $idd);
    $stInsDaiSalida->execute();
    $id_daisalida = (int)$db->insert_id;

    // Actualizar datadai
    $stUpdDatadai->bind_param('isii', $estado, $valorDaiCampo, $id_daisalida, $idd);
    $stUpdDatadai->execute();

    // Actualizar detalle
    $stUpdDetalle->bind_param('ii', $estado, $iddetalle);
    $stUpdDetalle->execute();

    // Auditoría
    $stAudit->bind_param(
      'ssssiiii',
      $fecha_actual, $hora_actual, $my_ip, $actividad, $estado, $idd, $id_daisalida, $id_usuario
    );
    $stAudit->execute();

    $procesados++;
  }

  // Si ya no quedan detalles pendientes (=0 o NULL) en la factura, subir estado de cabecera
  $stCountPend = $db->prepare("
    SELECT COUNT(*) AS c
    FROM tb_detallefactura
    WHERE condicion = 1 AND id_facturainfo = ? AND (id_estado = 0 OR id_estado IS NULL)
  ");
  $stCountPend->bind_param('i', $idfactura);
  $stCountPend->execute();
  $resPend = $stCountPend->get_result()->fetch_assoc();
  $pend = (int)($resPend['c'] ?? 0);
  $stCountPend->close();

  if ($pend === 0) {
    $stUpdCab = $db->prepare("UPDATE tb_facturainfo SET id_estado = ? WHERE idfacturainfo = ?");
    $stUpdCab->bind_param('ii', $estado, $idfactura);
    $stUpdCab->execute();
    $stUpdCab->close();
  }

  $db->commit();

  echo json_encode([
    'success'     => true,
    'message'     => 'Guardado correctamente',
    'procesados'  => $procesados,
    'pendientes'  => $pend
  ], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
  $db->rollback();
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>$e->getMessage()], JSON_UNESCAPED_UNICODE);
}
