<?php
// api/nacionalizados_detalle.php — Detalle de una factura (cabecera + ítems) con bloqueo SOLO por id_estado de tb_detallefactura
declare(strict_types=1);
session_start();

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');

if (!isset($_SESSION['ingreso']) || $_SESSION['ingreso'] !== 'YES') {
  echo json_encode(['success' => false, 'message' => 'No autorizado']); exit;
}

$id = (int)($_GET['id'] ?? $_POST['id'] ?? 0);
if ($id <= 0) {
  echo json_encode(['success' => false, 'message' => 'Parámetro id inválido']); exit;
}

require_once __DIR__ . '/../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db) {
  echo json_encode(['success' => false, 'message' => 'Sin conexión DB']); exit;
}
mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone = '-05:00'");

const ESTADO_BLOQUEADO = 8;

/* ===== Cabecera ===== */
$cabecera = null;
$sqlH = "SELECT fi.idfacturainfo AS id,
           fi.numero,
           DATE_FORMAT(fi.fecha, '%Y-%m-%d') AS fecha,
           CASE
             WHEN COALESCE(fi.exonerado,0)=1 AND COALESCE(fi.nombre,'')<>'' THEN fi.nombre
             ELSE e.razon_social
           END AS cliente
    FROM tb_facturainfo fi
    LEFT JOIN tb_empresa e ON e.idempresa = fi.id_cliente
WHERE fi.condicion = 1 AND fi.idfacturainfo = ?";
if ($stH = mysqli_prepare($db, $sqlH)) {
  mysqli_stmt_bind_param($stH, 'i', $id);
  mysqli_stmt_execute($stH);
  $rsH = mysqli_stmt_get_result($stH);
  $cabecera = $rsH ? $rsH->fetch_assoc() : null;
  mysqli_stmt_close($stH);
}
if (!$cabecera) {
  echo json_encode(['success' => false, 'message' => 'Factura no encontrada']); exit;
}

/* ===== Resumen agregado ===== */
$resumen = ['pedido'=>'','embarque'=>'','sp'=>'','dai'=>''];
$sqlR = "SELECT
           TRIM(BOTH ', ' FROM GROUP_CONCAT(DISTINCT sp.pedido ORDER BY sp.pedido SEPARATOR ', ')) AS pedido,
           TRIM(BOTH ', ' FROM GROUP_CONCAT(DISTINCT COALESCE(sp.cod_embarque,'') ORDER BY sp.cod_embarque SEPARATOR ', ')) AS embarque,
           TRIM(BOTH ', ' FROM GROUP_CONCAT(DISTINCT sp.numerosp ORDER BY sp.numerosp SEPARATOR ', ')) AS sp,
           TRIM(BOTH ', ' FROM GROUP_CONCAT(DISTINCT de.numerodai ORDER BY de.numerodai SEPARATOR ', ')) AS dai
         FROM tb_detallefactura df
         LEFT JOIN tb_datadai d ON d.iddatadai = df.id_datadai
         LEFT JOIN tb_solicitudprevia sp ON sp.idsolicitudprevia = d.id_solicitudprevia
         LEFT JOIN tb_daientrada de ON de.iddaientrada = d.id_daientrada
         WHERE df.id_facturainfo = ? AND df.condicion = 1";
if ($stR = mysqli_prepare($db, $sqlR)) {
  mysqli_stmt_bind_param($stR, 'i', $id);
  mysqli_stmt_execute($stR);
  $rsR = mysqli_stmt_get_result($stR);
  if ($rsR && ($r = $rsR->fetch_assoc())) {
    $resumen = [
      'pedido'   => (string)($r['pedido'] ?? ''),
      'embarque' => (string)($r['embarque'] ?? ''),
      'sp'       => (string)($r['sp'] ?? ''),
      'dai'      => (string)($r['dai'] ?? ''),
    ];
  }
  mysqli_stmt_close($stR);
}

/* ===== Ítems ===== */
$items = [];
$bloqueados = 0;
$habilitados = 0;

/*
 IMPORTANTE: el estado para bloquear viene SOLO de tb_detallefactura.id_estado
 Esto evita que un d.id_estado = 8 (en tb_datadai) te bloquee visualmente otros detalles.
*/
$sqlI = "SELECT
           d.iddatadai,
           d.item, d.motor, d.fobunitario, d.transporte, d.seguro, d.valoraduana,
           d.chasis, d.marca, d.modelo, d.color,
           sp.pedido,
           COALESCE(sp.cod_embarque,'') AS cod_embarque,
           sp.numerosp AS sp,
           de.numerodai AS dai,
           COALESCE(df.id_estado, 0) AS estado_detalle
         FROM tb_detallefactura df
         INNER JOIN tb_datadai d ON d.iddatadai = df.id_datadai
         LEFT JOIN tb_solicitudprevia sp ON sp.idsolicitudprevia = d.id_solicitudprevia
         LEFT JOIN tb_daientrada de ON de.iddaientrada = d.id_daientrada
         WHERE df.id_facturainfo = ? AND df.condicion = 1
         ORDER BY d.iddatadai ASC";

if ($stI = mysqli_prepare($db, $sqlI)) {
  mysqli_stmt_bind_param($stI, 'i', $id);
  mysqli_stmt_execute($stI);
  $rsI = mysqli_stmt_get_result($stI);
  while ($row = $rsI->fetch_assoc()) {
    $estado = (int)($row['estado_detalle'] ?? 0);
    $bloqueado     = ($estado === ESTADO_BLOQUEADO); // bloqueado si df.id_estado == 8
    $seleccionable = ($estado < ESTADO_BLOQUEADO);   // habilitado si df.id_estado < 8

    if ($bloqueado) $bloqueados++; else $habilitados++;

    $items[] = [
      'iddatadai'     => (int)$row['iddatadai'],
      'item'          => (string)($row['item'] ?? ''),
      'motor'         => (string)($row['motor'] ?? ''),
      'fobunitario'   => (string)($row['fobunitario'] ?? ''),
      'transporte'    => (string)($row['transporte'] ?? ''),
      'seguro'        => (string)($row['seguro'] ?? ''),
      'valoraduana'   => (string)($row['valoraduana'] ?? ''),
      'chasis'        => (string)($row['chasis'] ?? ''),
      'marca'         => (string)($row['marca'] ?? ''),
      'modelo'        => (string)($row['modelo'] ?? ''),
      'color'         => (string)($row['color'] ?? ''),
      'pedido'        => (string)($row['pedido'] ?? ''),
      'cod_embarque'  => (string)($row['cod_embarque'] ?? ''),
      'sp'            => (string)($row['sp'] ?? ''),
      'dai'           => (string)($row['dai'] ?? ''),
      'estado'        => $estado,
      'bloqueado'     => $bloqueado,
      'seleccionable' => $seleccionable
    ];
  }
  mysqli_stmt_close($stI);
}

$puede_guardar = ($habilitados > 0);

echo json_encode([
  'success'       => true,
  'cabecera'      => $cabecera,
  'resumen'       => $resumen,
  'items'         => $items,
  'conteo'        => ['bloqueados' => $bloqueados, 'habilitados' => $habilitados],
  'puede_guardar' => $puede_guardar
], JSON_UNESCAPED_UNICODE);
