<?php
// api/mov_refdata.php — RefData para movimientos (accesorios + ubicaciones base)
// Tolerante a esquemas: detecta si tb_inv_ubicacion tiene 'activo' o 'condicion' y si tiene 'nombre'
declare(strict_types=1);
ini_set('display_errors','0');
ini_set('log_errors','1');

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

session_start();
require_once __DIR__ . '/../conex.php';
if (!isset($conn) || !($conn instanceof mysqli)) {
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Sin conexión a BD']); exit;
}
mysqli_set_charset($conn,'utf8mb4');

/** Utilidad: verifica si una columna existe en una tabla */
function column_exists(mysqli $conn, string $table, string $column): bool {
  $sql = "SELECT 1
          FROM INFORMATION_SCHEMA.COLUMNS
          WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?
          LIMIT 1";
  $stmt = mysqli_prepare($conn, $sql);
  mysqli_stmt_bind_param($stmt, 'ss', $table, $column);
  mysqli_stmt_execute($stmt);
  mysqli_stmt_store_result($stmt);
  $exists = (mysqli_stmt_num_rows($stmt) > 0);
  mysqli_stmt_close($stmt);
  return $exists;
}

try {
  // ===== Accesorios (activos por condicion=1 si existe; si no, trae todos) =====
  $accesorios = [];
  $acc_has_condicion = column_exists($conn, 'tb_inv_accesorio', 'condicion');

  $sqlAcc = "SELECT id_accesorio, nombre FROM tb_inv_accesorio";
  if ($acc_has_condicion) {
    $sqlAcc .= " WHERE IFNULL(condicion,1)=1";
  }
  $sqlAcc .= " ORDER BY nombre ASC";

  if ($qa = mysqli_query($conn, $sqlAcc)) {
    while ($r = mysqli_fetch_assoc($qa)) {
      $accesorios[] = [
        'id_accesorio' => (int)$r['id_accesorio'],
        'nombre'       => (string)$r['nombre'],
      ];
    }
    mysqli_free_result($qa);
  } else {
    throw new mysqli_sql_exception('Error consultando accesorios: '.mysqli_error($conn));
  }

  // ===== Ubicaciones base =====
  // Detecta columnas existentes en tb_inv_ubicacion
  $ubi_has_nombre    = column_exists($conn, 'tb_inv_ubicacion', 'nombre');
  $ubi_has_activo    = column_exists($conn, 'tb_inv_ubicacion', 'activo');
  $ubi_has_condicion = column_exists($conn, 'tb_inv_ubicacion', 'condicion');

  // Campo de estado y filtro
  // Preferimos 'activo=1'; si no existe, usamos 'condicion=1'; si ninguno existe, no filtramos
  $estado_field      = $ubi_has_activo ? 'u.activo' : ($ubi_has_condicion ? 'u.condicion' : null);
  $estado_alias      = $ubi_has_activo ? 'activo'    : ($ubi_has_condicion ? 'condicion'    : 'estado');

  // SELECT base (si no hay u.nombre, lo armamos desde u.codigo)
  $select_nombre = $ubi_has_nombre
    ? "u.nombre AS ubi_nombre"
    : "u.codigo AS ubi_nombre"; // fallback

  // SELECT estado
  $select_estado = $estado_field
    ? "{$estado_field} AS {$estado_alias}"
    : "1 AS {$estado_alias}";   // sin columna de estado -> asumimos '1'

  $sqlUbi = "SELECT
               u.id_ubicacion,
               u.id_bodega,
               u.codigo,
               {$select_nombre},
               {$select_estado},
               b.nombre AS bodega_nombre
             FROM tb_inv_ubicacion u
             JOIN tb_inv_bodega b ON b.id_bodega = u.id_bodega";

  // WHERE estado si lo tenemos
  if ($estado_field) {
    $sqlUbi .= " WHERE {$estado_field} = 1";
  }

  $sqlUbi .= " ORDER BY b.nombre ASC, u.codigo ASC";

  $ubicaciones = [];
  if ($qu = mysqli_query($conn, $sqlUbi)) {
    while ($r = mysqli_fetch_assoc($qu)) {
      $nombreUbi = $r['ubi_nombre'] ?? $r['codigo'];
      $ubicaciones[] = [
        'id_ubicacion' => (int)$r['id_ubicacion'],
        'id_bodega'    => (int)$r['id_bodega'],
        'codigo'       => (string)$r['codigo'],
        'bodega'       => (string)$r['bodega_nombre'],
        'nombre'       => (string)$nombreUbi,                       // para el front
        'text'         => $r['bodega_nombre'].' / '.$nombreUbi,     // útil para combos
        'estado'       => isset($r[$estado_alias]) ? (int)$r[$estado_alias] : 1,
      ];
    }
    mysqli_free_result($qu);
  } else {
    throw new mysqli_sql_exception('Error consultando ubicaciones: '.mysqli_error($conn));
  }

  echo json_encode([
    'success'     => true,
    'accesorios'  => $accesorios,
    'ubicaciones' => $ubicaciones,
  ]);
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode([
    'success' => false,
    'message' => 'Error en mov_refdata',
    'error'   => $e->getMessage(),
  ]);
}
