<?php
// api/lavado_toggle.php — activa/desactiva “Lavada para PDS”
declare(strict_types=1);

header('Content-Type: application/json; charset=UTF-8');

try {
  session_start();
  if (!isset($_SESSION['ingreso']) || $_SESSION['ingreso'] !== 'YES') {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'No autorizado']);
    exit;
  }

  require_once __DIR__ . '/../conex.php';
  if (!isset($conn) || !($conn instanceof mysqli)) {
    throw new RuntimeException('Conexión $conn no disponible.');
  }
  mysqli_set_charset($conn, 'utf8mb4');

  $input = json_decode(file_get_contents('php://input'), true);
  $iddatadai = (int)($input['iddatadai'] ?? 0);
  $express   = (int)($input['express'] ?? 0); // 1=ON, 0=OFF

  if ($iddatadai <= 0 || ($express !== 0 && $express !== 1)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Parámetros inválidos']);
    exit;
  }

  // === 1) Obtener id del tipo “Lavada para PDS” ===
  //    (cambia el patrón si en tu tabla se llama distinto: ej. '%lavado express%')
  $sqlTipo = "SELECT idtipo_lavada FROM tb_tipolavada WHERE LOWER(lavada) LIKE '%lavada para pds%' LIMIT 1";
  $rsTipo = $conn->query($sqlTipo);
  $idTipoPDS = 0;
  if ($rsTipo && $row = $rsTipo->fetch_assoc()) {
    $idTipoPDS = (int)$row['idtipo_lavada'];
  }
  if ($express === 1 && $idTipoPDS === 0) {
    throw new RuntimeException('No existe el tipo de lavada "Lavada para PDS" en tb_tipolavada');
  }

  // === 2) Transacción para consistencia ===
  $conn->begin_transaction();

  if ($express === 1) {
    // ON: poner lavada=1 y registrar/actualizar último evento PDS

    // 2.1) Marcar datadai.lavada = 1 (nota: affected_rows puede ser 0 si ya estaba)
    $stmt = $conn->prepare("UPDATE tb_datadai SET lavada = 1 WHERE iddatadai = ?");
    if (!$stmt) { throw new RuntimeException('Prepare UPDATE(1): ' . $conn->error); }
    $stmt->bind_param('i', $iddatadai);
    if (!$stmt->execute()) { throw new RuntimeException('Execute UPDATE(1): ' . $stmt->error); }
    $stmt->close();

    // 2.2) Consultar el último evento de lavada para ese vehículo
    $stmt = $conn->prepare("
      SELECT l.idlavada, LOWER(t.lavada) AS tipo_last
      FROM tb_lavada l
      JOIN tb_tipolavada t ON t.idtipo_lavada = l.id_tipolavada
      WHERE l.id_datadai = ?
      ORDER BY l.idlavada DESC
      LIMIT 1
    ");
    if (!$stmt) { throw new RuntimeException('Prepare SELECT last: ' . $conn->error); }
    $stmt->bind_param('i', $iddatadai);
    if (!$stmt->execute()) { throw new RuntimeException('Execute SELECT last: ' . $stmt->error); }
    $res = $stmt->get_result();
    $last = $res ? $res->fetch_assoc() : null;
    $stmt->close();

    $ultimoEsPDS = $last && isset($last['tipo_last']) && strpos((string)$last['tipo_last'], 'lavada para pds') !== false;

    if ($ultimoEsPDS) {
      // 2.3A) Ya era PDS → refrescar fecha del último
      $idlavada = (int)$last['idlavada'];
      $stmt = $conn->prepare("UPDATE tb_lavada SET fecha_registro = NOW() WHERE idlavada = ?");
      if (!$stmt) { throw new RuntimeException('Prepare UPDATE fecha: ' . $conn->error); }
      $stmt->bind_param('i', $idlavada);
      if (!$stmt->execute()) { throw new RuntimeException('Execute UPDATE fecha: ' . $stmt->error); }
      $stmt->close();
    } else {
      // 2.3B) No era PDS → insertar evento PDS
      //       Ajusta columnas si tu tb_lavada tiene más NOT NULL obligatorios
      $stmt = $conn->prepare("
        INSERT INTO tb_lavada (id_datadai, id_tipolavada, fecha_registro, condicion)
        VALUES (?, ?, NOW(), 1)
      ");
      if (!$stmt) { throw new RuntimeException('Prepare INSERT PDS: ' . $conn->error); }
      $stmt->bind_param('ii', $iddatadai, $idTipoPDS);
      if (!$stmt->execute()) { throw new RuntimeException('Execute INSERT PDS: ' . $stmt->error); }
      $stmt->close();
    }

  } else {
    // OFF: poner lavada=0 y eliminar solo el último evento PDS (si existe)

    // 2.1) Marcar datadai.lavada = 0
    $stmt = $conn->prepare("UPDATE tb_datadai SET lavada = 0 WHERE iddatadai = ?");
    if (!$stmt) { throw new RuntimeException('Prepare UPDATE(0): ' . $conn->error); }
    $stmt->bind_param('i', $iddatadai);
    if (!$stmt->execute()) { throw new RuntimeException('Execute UPDATE(0): ' . $stmt->error); }
    $stmt->close();

    // 2.2) Buscar el último evento “Lavada para PDS”
    $stmt = $conn->prepare("
      SELECT l.idlavada
      FROM tb_lavada l
      JOIN tb_tipolavada t ON t.idtipo_lavada = l.id_tipolavada
      WHERE l.id_datadai = ? AND LOWER(t.lavada) LIKE '%lavada para pds%'
      ORDER BY l.idlavada DESC
      LIMIT 1
    ");
    if (!$stmt) { throw new RuntimeException('Prepare SELECT last PDS: ' . $conn->error); }
    $stmt->bind_param('i', $iddatadai);
    if (!$stmt->execute()) { throw new RuntimeException('Execute SELECT last PDS: ' . $stmt->error); }
    $res = $stmt->get_result();
    $last = $res ? $res->fetch_assoc() : null;
    $stmt->close();

    if ($last && (int)$last['idlavada'] > 0) {
      // 2.3) Borrar solo ese último registro PDS
      $idlavada = (int)$last['idlavada'];
      $stmt = $conn->prepare("DELETE FROM tb_lavada WHERE idlavada = ?");
      if (!$stmt) { throw new RuntimeException('Prepare DELETE PDS: ' . $conn->error); }
      $stmt->bind_param('i', $idlavada);
      if (!$stmt->execute()) { throw new RuntimeException('Execute DELETE PDS: ' . $stmt->error); }
      $stmt->close();
    }
  }

  // 3) Confirmar transacción
  $conn->commit();
  echo json_encode(['success' => true]);

} catch (Throwable $e) {
  if (isset($conn) && $conn instanceof mysqli) {
    $conn->rollback();
  }
  http_response_code(500);
  echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}
