<?php
// api/lavada_save_strict.php — Guarda tipo de lavada para un PDI
// Regla: SIEMPRE establece estado_lavado = 'PENDIENTE'
// - Si el último registro es del MISMO tipo y NO está 'PENDIENTE' -> lo ACTUALIZA a 'PENDIENTE' (y refresca fecha)
// - Si ya es el mismo tipo y ya está 'PENDIENTE' -> no hace nada
// - Si el tipo es distinto o no existe -> INSERT con 'PENDIENTE'
// Opcional: refleja en tb_datadai.estado_lavado si esa columna existe

declare(strict_types=1);

date_default_timezone_set('America/Guayaquil');

/* HEADERS */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* SESIÓN */
session_start();
if (!isset($_SESSION['ingreso']) || $_SESSION['ingreso'] !== 'YES') {
  http_response_code(401);
  echo json_encode(['success'=>false,'message'=>'No autorizado']); exit;
}
$idUsuario = (int)($_SESSION['idusuario'] ?? 0);

/* BD */
require_once __DIR__ . '/../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) { http_response_code(500); echo json_encode(['success'=>false,'message'=>'Sin conexión']); exit; }
if (function_exists('mysqli_report')) { mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); }
mysqli_set_charset($db, 'utf8mb4');
@$db->query("SET time_zone='-05:00'");

/* INPUT */
$raw = file_get_contents('php://input') ?: '';
$in  = json_decode($raw, true);
if (!is_array($in)) { $in = $_POST ?: []; }

$idpdi  = (int)($in['idpdi'] ?? 0);
$idtipo = (int)($in['idtipo_lavada'] ?? 0);

// Forzar SIEMPRE 'PENDIENTE'
$estado = 'PENDIENTE';

if ($idpdi <= 0) { echo json_encode(['success'=>false,'message'=>'idpdi inválido']); exit; }
if ($idtipo <= 0) { echo json_encode(['success'=>true,'action'=>'noop','message'=>'Sin cambios (idtipo vacío)']); exit; }

try {
  // ¿Existe columna tb_datadai.estado_lavado? (para reflejo opcional)
  $hasEstadoDatadai = false;
  $chk = $db->prepare("
    SELECT COUNT(*) c
    FROM INFORMATION_SCHEMA.COLUMNS
    WHERE TABLE_SCHEMA = DATABASE()
      AND TABLE_NAME = 'tb_datadai'
      AND COLUMN_NAME = 'estado_lavado'
    LIMIT 1
  ");
  $chk->execute();
  $hasEstadoDatadai = ((int)($chk->get_result()->fetch_assoc()['c'] ?? 0) > 0);
  $chk->close();

  // Validar tipo
  $stVal = $db->prepare("SELECT COUNT(*) c FROM tb_tipolavada WHERE idtipo_lavada=? AND condicion=1");
  $stVal->bind_param('i', $idtipo);
  $stVal->execute();
  $c = (int)($stVal->get_result()->fetch_assoc()['c'] ?? 0);
  $stVal->close();
  if ($c === 0) { echo json_encode(['success'=>false,'message'=>'Tipo de lavada inválido']); exit; }

  // id_datadai por PDI
  $stP = $db->prepare("SELECT id_datadai FROM tb_pdi WHERE idpdi=? AND condicion=1");
  $stP->bind_param('i', $idpdi);
  $stP->execute();
  $rowP = $stP->get_result()->fetch_assoc();
  $stP->close();
  if (!$rowP) { echo json_encode(['success'=>false,'message'=>'PDI no encontrado']); exit; }
  $idDatadai = (int)$rowP['id_datadai'];

  $db->begin_transaction();
  $db->query("SET SESSION TRANSACTION ISOLATION LEVEL READ COMMITTED");

  // Último registro activo para ese id_datadai
  $stLast = $db->prepare("
    SELECT idlavada, id_tipolavada, estado_lavado
    FROM tb_lavada
    WHERE id_datadai=? AND condicion=1
    ORDER BY fecha_registro DESC, idlavada DESC
    LIMIT 1
    FOR UPDATE
  ");
  $stLast->bind_param('i', $idDatadai);
  $stLast->execute();
  $last = $stLast->get_result()->fetch_assoc();
  $stLast->close();

  if ($last && (int)$last['id_tipolavada'] === $idtipo) {
    // Mismo tipo -> si no está PENDIENTE, actualizar a PENDIENTE; si ya está, no cambiar
    if ((string)$last['estado_lavado'] === $estado) {
      // reflejo opcional
      if ($hasEstadoDatadai) {
        $stRef = $db->prepare("UPDATE tb_datadai SET estado_lavado=? WHERE iddatadai=?");
        $stRef->bind_param('si', $estado, $idDatadai);
        $stRef->execute(); $stRef->close();
      }
      $db->commit();
      echo json_encode(['success'=>true,'action'=>'nochange','message'=>'Sin cambios (ya estaba PENDIENTE)']); exit;
    }
    $stU = $db->prepare("
      UPDATE tb_lavada
      SET estado_lavado='PENDIENTE', fecha_registro=NOW(), usuario_update=?, fecha_update=NOW()
      WHERE idlavada=?
    ");
    $idlavada = (int)$last['idlavada'];
    $stU->bind_param('ii', $idUsuario, $idlavada);
    $stU->execute(); $stU->close();

    if ($hasEstadoDatadai) {
      $stRef = $db->prepare("UPDATE tb_datadai SET estado_lavado='PENDIENTE' WHERE iddatadai=?");
      $stRef->bind_param('i', $idDatadai);
      $stRef->execute(); $stRef->close();
    }

    $db->commit();
    echo json_encode(['success'=>true,'action'=>'update','idlavada'=>$idlavada,'estado'=>'PENDIENTE']); exit;
  }

  // Tipo distinto o no había registro -> INSERT con PENDIENTE
  $stI = $db->prepare("
    INSERT INTO tb_lavada (id_datadai, fecha_registro, id_tipolavada, id_usuario, estado_lavado, condicion)
    VALUES (?, NOW(), ?, ?, 'PENDIENTE', 1)
  ");
  $stI->bind_param('iii', $idDatadai, $idtipo, $idUsuario);
  $stI->execute();
  $idlavada = $stI->insert_id;
  $stI->close();

  if ($hasEstadoDatadai) {
    $stRef = $db->prepare("UPDATE tb_datadai SET estado_lavado='PENDIENTE' WHERE iddatadai=?");
    $stRef->bind_param('i', $idDatadai);
    $stRef->execute(); $stRef->close();
  }

  $db->commit();
  echo json_encode(['success'=>true,'action'=>'insert','idlavada'=>$idlavada,'estado'=>'PENDIENTE']);
} catch (Throwable $e) {
  try { $db->rollback(); } catch(Throwable $ign){}
  http_response_code(500);
  echo json_encode(['success'=>false,'message'=>'Error guardando lavada', 'error'=>$e->getMessage()]);
}
