<?php
// api/ingresos_list.php — Lista unidades "DAI 70" + múltiples PDFs por tarja
declare(strict_types=1);
session_start();
header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/../conex.php';

$db = $conn ?? ($conex ?? null);
if (!$db instanceof mysqli) { echo json_encode(['data'=>[]]); exit; }

if (function_exists('mysqli_report')) { mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT); }
mysqli_set_charset($db, 'utf8mb4');

/* ========= 1) Traer unidades (una fila por iddatadai) ========= */
$sql = "
  SELECT
    d.iddatadai,
    d.item,
    d.marca,
    d.modelo,
    d.chasis,
    d.color,
    d.id_estado,
    e.numerodai,
    MAX(t.numerotarja) AS numerotarja,   -- referencial (última)
    MAX(t.numero_auto) AS numero_auto,   -- referencial (última)
    COALESCE(b.sigla, b.detalle, '') AS sigla,
    es.descripcion AS estado
  FROM tb_datadai AS d
  INNER JOIN tb_daientrada      AS e  ON e.iddaientrada       = d.id_daientrada
  INNER JOIN tb_solicitudprevia AS sp ON sp.idsolicitudprevia = d.id_solicitudprevia
  INNER JOIN tb_estado          AS es ON es.idestado          = d.id_estado
  LEFT  JOIN tb_tarja           AS t  ON t.id_datadai         = d.iddatadai
  LEFT  JOIN tb_bodega          AS b  ON b.idbodega           = d.id_bodega
  WHERE d.condicion = 1
    AND d.dai <> 'Simple'
  GROUP BY
    d.iddatadai, d.item, d.marca, d.modelo, d.chasis, d.color, d.id_estado,
    e.numerodai, es.descripcion, b.sigla, b.detalle
  ORDER BY d.iddatadai DESC
";

$res = mysqli_query($db, $sql);
$rows   = [];
$idList = [];

if ($res) {
  while ($r = mysqli_fetch_assoc($res)) {
    $r['iddatadai'] = (int)$r['iddatadai'];
    $rows[]   = $r;
    $idList[] = $r['iddatadai'];
  }
}

/* ========= 2) Traer TODAS las tarjas activas para esos id_datadai en un solo query ========= */
$tarjasMap = []; // id_datadai => [ {idtarja, numerotarja, numero_auto, fechatarja}, ... ]
if (!empty($idList)) {
  $in = implode(',', array_map('intval', $idList));
  $sqlT = "
    SELECT
      t.id_datadai,
      t.idtarja,
      t.numerotarja,
      t.numero_auto,
      t.fechatarja
    FROM tb_tarja t
    WHERE t.condicion = 1
      AND t.id_datadai IN ($in)
    ORDER BY t.fechatarja DESC, t.idtarja DESC
  ";
  $resT = mysqli_query($db, $sqlT);
  if ($resT) {
    while ($t = mysqli_fetch_assoc($resT)) {
      $idd = (int)$t['id_datadai'];
      if (!isset($tarjasMap[$idd])) $tarjasMap[$idd] = [];
      $tarjasMap[$idd][] = [
        'idtarja'     => (int)$t['idtarja'],
        'numerotarja' => (string)($t['numerotarja'] ?? ''),
        'numero_auto' => isset($t['numero_auto']) ? (int)$t['numero_auto'] : null,
        'fechatarja'  => (string)($t['fechatarja'] ?? '')
      ];
    }
  }
}

/* ========= 3) Salida para DataTables ========= */
$data = [];
foreach ($rows as $r) {
  $iddatadai = (int)$r['iddatadai'];
  $chasis    = htmlspecialchars((string)($r['chasis'] ?? ''), ENT_QUOTES, 'UTF-8');

  // Enlaces múltiples con TUS PARÁMETROS: ?itemdai=<iddatadai>&id=<idtarja>
  $chips = '';
  if (!empty($tarjasMap[$iddatadai])) {
    $parts = [];
    foreach ($tarjasMap[$iddatadai] as $t) {
      $numero = htmlspecialchars($t['numerotarja'], ENT_QUOTES, 'UTF-8');

      // === URL del PDF manteniendo itemdai y añadiendo id para tarja exacta ===
      $pdfUrl = '../../views/pages/pdf/index.php?itemdai=' . $iddatadai;

      // Si prefieres usar numerotarja además del id (opcional):
      // $pdfUrl .= '&numerotarja=' . urlencode($t['numerotarja']);

      $parts[] =
        '<a href="'.$pdfUrl.'" class="btn btn-primary btn-sm me-1 mb-1" '.
        'target="_blank" data-bs-toggle="tooltip" title="Abrir '.$numero.'">'.
        '<i class="fa-solid fa-file-pdf me-1"></i>'.$numero.
        '</a>';
    }
    $chips = implode('', $parts);
  } else {
    $chips = '<span class="badge text-bg-secondary">Sin tarjas</span>';
  }

  $data[] = [
    'dai70'       => htmlspecialchars((string)($r['numerodai'] ?? ''), ENT_QUOTES, 'UTF-8'),
    'item'        => htmlspecialchars((string)($r['item'] ?? ''), ENT_QUOTES, 'UTF-8'),
    'numerotarja' => htmlspecialchars((string)($r['numerotarja'] ?? ''), ENT_QUOTES, 'UTF-8'), // referencial
    'descripcion' => htmlspecialchars((string)($r['marca'] ?? ''), ENT_QUOTES, 'UTF-8'),
    'modelo'      => htmlspecialchars((string)($r['modelo'] ?? ''), ENT_QUOTES, 'UTF-8'),
    'chasis'      => $chasis,
    'color'       => htmlspecialchars((string)($r['color'] ?? ''), ENT_QUOTES, 'UTF-8'),
    'estado'      => htmlspecialchars((string)($r['estado'] ?? ''), ENT_QUOTES, 'UTF-8'),
    'accion'      => $chips
  ];
}

echo json_encode(['data' => $data], JSON_UNESCAPED_UNICODE);
