<?php
// Reemplaza la lista de tipos de reparación de un incidente (JSON estricto y robusto)
declare(strict_types=1);
require_once __DIR__ . '/_api_bootstrap.php';

try {
  // Leer JSON de entrada
  $in = json_decode(file_get_contents('php://input'), true);
  if (!is_array($in)) { $in = []; }

  $incident_id = isset($in['incident_id']) ? (int)$in['incident_id'] : 0;

  // Normalizar array de tipos (enteros únicos > 0) sin usar funciones flecha
  $tipos_ids = array();
  if (isset($in['tipos_ids']) && is_array($in['tipos_ids'])) {
    foreach ($in['tipos_ids'] as $val) {
      $ival = (int)$val;
      if ($ival > 0) { $tipos_ids[] = $ival; }
    }
    $tipos_ids = array_values(array_unique($tipos_ids));
  }

  if ($incident_id <= 0) {
    api_json(['success'=>false,'message'=>'incident_id requerido'], 422);
  }

  // Asegurar tabla (PK compuesta)
  $ddl = "CREATE TABLE IF NOT EXISTS tb_incidente_tipo (
            incident_id INT NOT NULL,
            id_tipo     INT NOT NULL,
            PRIMARY KEY (incident_id, id_tipo),
            KEY idx_tipo (id_tipo)
          ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
  if (!$db->query($ddl)) {
    api_json(['success'=>false,'message'=>'Error creando tabla','sql_error'=>$db->error], 500);
  }

  // Transacción: reemplazar set completo
  if (!$db->begin_transaction()) {
    api_json(['success'=>false,'message'=>'No se pudo iniciar transacción','sql_error'=>$db->error], 500);
  }

  // Borrar anteriores
  $stmtDel = $db->prepare("DELETE FROM tb_incidente_tipo WHERE incident_id=?");
  if (!$stmtDel) {
    $db->rollback();
    api_json(['success'=>false,'message'=>'Error en prepare(DELETE)','sql_error'=>$db->error], 500);
  }
  $stmtDel->bind_param('i', $incident_id);
  if (!$stmtDel->execute()) {
    $db->rollback();
    api_json(['success'=>false,'message'=>'Error ejecutando DELETE','sql_error'=>$stmtDel->error], 500);
  }

  // Insertar nuevos (si hay)
  if (!empty($tipos_ids)) {
    $stmtIns = $db->prepare("INSERT INTO tb_incidente_tipo (incident_id, id_tipo) VALUES (?, ?)");
    if (!$stmtIns) {
      $db->rollback();
      api_json(['success'=>false,'message'=>'Error en prepare(INSERT)','sql_error'=>$db->error], 500);
    }
    foreach ($tipos_ids as $tid) {
      $stmtIns->bind_param('ii', $incident_id, $tid);
      if (!$stmtIns->execute()) {
        $db->rollback();
        api_json([
          'success'=>false,
          'message'=>'Error insertando tipo',
          'sql_error'=>$stmtIns->error,
          'detalle'=>['incident_id'=>$incident_id, 'id_tipo'=>$tid]
        ], 500);
      }
    }
  }

  // Commit
  if (!$db->commit()) {
    $db->rollback();
    api_json(['success'=>false,'message'=>'Error haciendo commit','sql_error'=>$db->error], 500);
  }

  api_json(['success'=>true]);

} catch (Throwable $e) {
  if (isset($db) && ($db->errno ?? 0)) { @mysqli_rollback($db); }
  api_json(['success'=>false,'message'=>'Error guardando tipos','detail'=>$e->getMessage()], 500);
}
