<?php
// views/pages/api/facturas_vehiculos_guardar.php
// Inserta un registro en tb_detallefactura con: item, id_datadai, id_facturainfo
// Defaults: id_estado=5, condicion=1
// Evita duplicados (id_facturainfo + id_datadai) y autocalcula item si no se envía.

declare(strict_types=1);
session_start();

/* ===== Respuesta JSON & CORS ===== */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

/* ===== Preflight ===== */
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

/* ===== Método ===== */
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  http_response_code(405);
  echo json_encode(['success'=>false, 'message'=>'Método no permitido']);
  exit;
}

/* ===== Robustez: capturar fatales como JSON ===== */
error_reporting(E_ALL);
ini_set('display_errors', '0');
ob_start();
register_shutdown_function(function () {
  $e = error_get_last();
  if ($e && in_array($e['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
    if (ob_get_level()) { ob_end_clean(); }
    http_response_code(500);
    echo json_encode([
      'success'=>false,
      'message'=>'Error fatal en servidor',
      'detail'=>($e['message'] ?? '') . ' @ ' . ($e['file'] ?? '') . ':' . ($e['line'] ?? 0)
    ]);
  }
});

/* ===== Conexión ===== */
require_once __DIR__ . '/../conex.php';
$db = $conn ?? ($conex ?? null);
if (!$db) { if (ob_get_level()) ob_end_clean(); echo json_encode(['success'=>false,'message'=>'Sin conexión a la base de datos']); exit; }
mysqli_set_charset($db, 'utf8mb4');
@mysqli_query($db, "SET time_zone='-05:00'"); // Ecuador

/* ===== Entradas ===== */
$id_facturainfo = (int)($_POST['idfactura']  ?? 0);
$id_datadai     = (int)($_POST['iddatadai']  ?? 0);
$item           = (int)($_POST['item']       ?? 0);

// Defaults de la fila
$id_estado = 5; // creado/pendiente
$condicion = 1; // activo

/* ===== Validaciones básicas ===== */
if ($id_facturainfo <= 0 || $id_datadai <= 0) {
  if (ob_get_level()) ob_end_clean();
  echo json_encode(['success'=>false,'message'=>'Parámetros incompletos: idfactura e iddatadai son obligatorios.']);
  exit;
}

/* (Opcional) Validar que el DAI exista */
$validDat = false;
if ($stmt = mysqli_prepare($db, "SELECT 1 FROM tb_datadai WHERE iddatadai = ? LIMIT 1")) {
  mysqli_stmt_bind_param($stmt, 'i', $id_datadai);
  mysqli_stmt_execute($stmt);
  $res = mysqli_stmt_get_result($stmt);
  $validDat = (bool)($res && mysqli_fetch_row($res));
  mysqli_free_result($res);
  mysqli_stmt_close($stmt);
}
if (!$validDat) {
  if (ob_get_level()) ob_end_clean();
  echo json_encode(['success'=>false,'message'=>'El iddatadai no existe en tb_datadai.']);
  exit;
}

/* Evitar duplicado en detalle: misma factura + mismo datadai + activo */
if ($stmt = mysqli_prepare($db, "SELECT iddetallefactura FROM tb_detallefactura WHERE id_facturainfo = ? AND id_datadai = ? AND condicion = 1 LIMIT 1")) {
  mysqli_stmt_bind_param($stmt, 'ii', $id_facturainfo, $id_datadai);
  mysqli_stmt_execute($stmt);
  $res = mysqli_stmt_get_result($stmt);
  if ($res && ($row = mysqli_fetch_assoc($res))) {
    $iddet = (int)$row['iddetallefactura'];
    mysqli_free_result($res);
    mysqli_stmt_close($stmt);
    if (ob_get_level()) ob_end_clean();
    echo json_encode(['success'=>true, 'duplicate'=>true, 'iddetallefactura'=>$iddet, 'message'=>'Este DAI ya está en el detalle de la factura.']);
    exit;
  }
  mysqli_free_result($res);
  mysqli_stmt_close($stmt);
}

/* Si no llega item, autocalcular como MAX(item)+1 por factura */
if ($item <= 0) {
  if ($stmt = mysqli_prepare($db, "SELECT COALESCE(MAX(item),0)+1 AS next_item FROM tb_detallefactura WHERE id_facturainfo = ?")) {
    mysqli_stmt_bind_param($stmt, 'i', $id_facturainfo);
    mysqli_stmt_execute($stmt);
    $res = mysqli_stmt_get_result($stmt);
    if ($res && ($row = mysqli_fetch_assoc($res))) {
      $item = (int)$row['next_item'];
    } else {
      $item = 1;
    }
    mysqli_free_result($res);
    mysqli_stmt_close($stmt);
  } else {
    $item = 1;
  }
}

/* ===== Insertar ===== */
$sql = "INSERT INTO tb_detallefactura (item, id_estado, id_datadai, id_facturainfo, condicion)
        VALUES (?, ?, ?, ?, ?)";
$stmt = mysqli_prepare($db, $sql);
if (!$stmt) {
  if (ob_get_level()) ob_end_clean();
  echo json_encode(['success'=>false,'message'=>'No se pudo preparar la sentencia de inserción.']);
  exit;
}
mysqli_stmt_bind_param($stmt, 'iiiii', $item, $id_estado, $id_datadai, $id_facturainfo, $condicion);

$ok = mysqli_stmt_execute($stmt);
if (!$ok) {
  $err = mysqli_error($db);
  mysqli_stmt_close($stmt);
  if (ob_get_level()) ob_end_clean();
  echo json_encode(['success'=>false,'message'=>'Error al guardar detalle: '.$err]);
  exit;
}
$iddet = (int)mysqli_insert_id($db);
mysqli_stmt_close($stmt);

/* ===== Éxito ===== */
if (ob_get_level()) ob_end_clean();
echo json_encode([
  'success'=>true,
  'iddetallefactura'=>$iddet,
  'item'=>$item,
  'id_facturainfo'=>$id_facturainfo,
  'id_datadai'=>$id_datadai
]);
